/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* $Id$ */

package org.apache.fop.complexscripts.fonts;

import java.io.File;
import java.nio.IntBuffer;
import java.util.List;
import java.util.Map;

import org.apache.fop.complexscripts.fonts.GlyphSubtable;
import org.apache.fop.complexscripts.fonts.GlyphSubstitutionSubtable;
import org.apache.fop.complexscripts.fonts.GlyphSubstitutionTable;
import org.apache.fop.complexscripts.fonts.GlyphTable.LookupSpec;
import org.apache.fop.complexscripts.fonts.GlyphTable.LookupTable;
import org.apache.fop.complexscripts.fonts.ttx.TTXFile;
import org.apache.fop.complexscripts.util.GlyphContextTester;
import org.apache.fop.complexscripts.util.GlyphSequence;
import org.apache.fop.complexscripts.util.ScriptContextTester;

import org.junit.Test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

public class GSUBTestCase implements ScriptContextTester, GlyphContextTester {

    private static String ttxFilesRoot = "test/resources/complexscripts";

    private static String[][] ttxFonts = {
        { "f0", "arab/ttx/arab-001.ttx" },              // simplified arabic
        { "f1", "arab/ttx/arab-002.ttx" },              // traditional arabic
        { "f2", "arab/ttx/arab-003.ttx" },              // lateef
        { "f3", "arab/ttx/arab-004.ttx" },              // scheherazade
    };

    private static Object[][] ltSingle = {
        { GlyphSubstitutionTable.GSUB_LOOKUP_TYPE_SINGLE },
        // arab-001.ttx
        { "f0", "lu2", "arab", "dflt", "isol",
          new String[][][] {
              { { "ainisolated" }, { "ain" } },
              { { "alefmaksuraisolated" }, { "alefmaksura" } },
              { { "behisolated" }, { "beh" } },
              { { "dadisolated" }, { "dad" } },
              { { "dalisolated" }, { "dal" } },
              { { "farsiyehisolated" }, { "farsiyeh" } },
              { { "fehisolated" }, { "feh" } },
              { { "gafisolated" }, { "gaf" } },
              { { "ghainisolated" }, { "ghain" } },
              { { "hahisolated" }, { "hah" } },
              { { "jeemisolated" }, { "jeem" } },
              { { "jehisolated" }, { "jeh" } },
              { { "kafisolated" }, { "arabickaf" } },
              { { "kehehisolated" }, { "keheh" } },
              { { "khahisolated" }, { "khah" } },
              { { "meemisolated" }, { "meem" } },
              { { "noonisolated" }, { "noon" } },
              { { "pehisolated" }, { "peh" } },
              { { "qafisolated" }, { "qaf" } },
              { { "rehisolated" }, { "reh" } },
              { { "sadisolated" }, { "sad" } },
              { { "seenisolated" }, { "seen" } },
              { { "sheenisolated" }, { "sheen" } },
              { { "tahisolated" }, { "tah" } },
              { { "tchehisolated" }, { "tcheh" } },
              { { "tehisolated" }, { "teh" } },
              { { "tehmarbutaisolated" }, { "tehmarbuta" } },
              { { "thalisolated" }, { "thal" } },
              { { "thehisolated" }, { "theh" } },
              { { "vehisolated" }, { "veh" } },
              { { "wawisolated" }, { "waw" } },
              { { "yehisolated" }, { "yeh" } },
              { { "yehwithhamzaaboveisolated" }, { "yehwithhamzaabove" } },
              { { "zahisolated" }, { "zah" } },
              { { "zainisolated" }, { "zain" } },
          },
        },
        { "f0", "lu4", "arab", "dflt", "fina",
          new String[][][] {
              { { "ain" }, { "ainfinal" } },
              { { "alefmaksura" }, { "alefmaksurafinal" } },
              { { "alefwasla" }, { "alefwaslafinal" } },
              { { "alefwithhamzaabove" }, { "alefwithhamzaabovefinal" } },
              { { "alefwithhamzabelow" }, { "alefwithhamzabelowfinal" } },
              { { "alefwithmaddaabove" }, { "alefwithmaddaabovefinal" } },
              { { "arabicae" }, { "hehfinal" } },
              { { "arabicalef" }, { "aleffinal" } },
              { { "arabickaf" }, { "arabickaf" } },
              { { "beh" }, { "beh" } },
              { { "dad" }, { "dad" } },
              { { "dal" }, { "dal" } },
              { { "farsiyeh" }, { "farsiyehfinal" } },
              { { "feh" }, { "feh" } },
              { { "gaf" }, { "gaffinal" } },
              { { "ghain" }, { "ghainfinal" } },
              { { "hah" }, { "hahfinal" } },
              { { "heh" }, { "hehfinal" } },
              { { "jeem" }, { "jeemfinal" } },
              { { "jeh" }, { "jeh" } },
              { { "keheh" }, { "kehehfinal" } },
              { { "khah" }, { "khahfinal" } },
              { { "lam" }, { "lam" } },
              { { "meem" }, { "meem" } },
              { { "noon" }, { "noon" } },
              { { "peh" }, { "peh" } },
              { { "qaf" }, { "qaf" } },
              { { "reh" }, { "reh" } },
              { { "sad" }, { "sad" } },
              { { "seen" }, { "seen" } },
              { { "sheen" }, { "sheen" } },
              { { "tah" }, { "tah" } },
              { { "tcheh" }, { "tchehfinal" } },
              { { "teh" }, { "teh" } },
              { { "tehmarbuta" }, { "tehmarbutafinal" } },
              { { "thal" }, { "thal" } },
              { { "theh" }, { "theh" } },
              { { "veh" }, { "veh" } },
              { { "waw" }, { "waw" } },
              { { "wawwithhamzaabove" }, { "wawwithhamzaabove" } },
              { { "yeh" }, { "yehfinal" } },
              { { "yehwithhamzaabove" }, { "yehwithhamzaabovefinal" } },
              { { "zah" }, { "zah" } },
              { { "zain" }, { "zain" } },
          }
        },
        { "f0", "lu5", "arab", "dflt", "init",
          new String[][][] {
              { { "ain" }, { "aininitial" } },
              { { "alefmaksura" }, { "uni0649.init" } },
              { { "arabickaf" }, { "kafmedial" } },
              { { "beh" }, { "behmedial" } },
              { { "dad" }, { "dadmedial" } },
              { { "farsiyeh" }, { "yehmedial" } },
              { { "feh" }, { "fehinitial" } },
              { { "gaf" }, { "gafinitial" } },
              { { "ghain" }, { "ghaininitial" } },
              { { "hah" }, { "hahmedial" } },
              { { "heh" }, { "hehinitial" } },
              { { "jeem" }, { "jeemmedial" } },
              { { "keheh" }, { "kehehinitial" } },
              { { "khah" }, { "khahmedial" } },
              { { "lam" }, { "lamisolated" } },
              { { "meem" }, { "meemmedial" } },
              { { "noon" }, { "noonmedial" } },
              { { "peh" }, { "pehmedial" } },
              { { "qaf" }, { "qafinitial" } },
              { { "sad" }, { "sadmedial" } },
              { { "seen" }, { "seenmedial" } },
              { { "sheen" }, { "sheenmedial" } },
              { { "tah" }, { "tah" } },
              { { "tcheh" }, { "tchehmedial" } },
              { { "teh" }, { "tehmedial" } },
              { { "theh" }, { "thehmedial" } },
              { { "veh" }, { "uni06A5.init" } },
              { { "yeh" }, { "yehmedial" } },
              { { "yehwithhamzaabove" }, { "yehwithhamzaabovemedial" } },
              { { "zah" }, { "zah" } },
          }
        },
        { "f0", "lu6", "arab", "dflt", "medi",
          new String[][][] {
              { { "ain" }, { "ainmedial" } },
              { { "alefmaksura" }, { "uni0649.init" } },
              { { "arabickaf" }, { "kafmedial" } },
              { { "beh" }, { "behmedial" } },
              { { "dad" }, { "dadmedial" } },
              { { "farsiyeh" }, { "yehmedial" } },
              { { "feh" }, { "fehmedial" } },
              { { "gaf" }, { "gafmedial" } },
              { { "ghain" }, { "ghainmedial" } },
              { { "hah" }, { "hahmedial" } },
              { { "heh" }, { "hehmedial" } },
              { { "jeem" }, { "jeemmedial" } },
              { { "keheh" }, { "kehehmedial" } },
              { { "khah" }, { "khahmedial" } },
              { { "lam" }, { "lammedial" } },
              { { "meem" }, { "meemmedial" } },
              { { "noon" }, { "noonmedial" } },
              { { "peh" }, { "pehmedial" } },
              { { "qaf" }, { "qafmedial" } },
              { { "sad" }, { "sadmedial" } },
              { { "seen" }, { "seenmedial" } },
              { { "sheen" }, { "sheenmedial" } },
              { { "tah" }, { "tah" } },
              { { "tcheh" }, { "tchehmedial" } },
              { { "teh" }, { "tehmedial" } },
              { { "theh" }, { "thehmedial" } },
              { { "veh" }, { "vehmedial" } },
              { { "yeh" }, { "yehmedial" } },
              { { "yehwithhamzaabove" }, { "yehwithhamzaabovemedial" } },
              { { "zah" }, { "zah" } },
          }
        },
        // arab-002.ttx
        { "f1", "lu1", "arab", "*", "isol",
          new String[][][] {
              { { "ainisolated" }, { "ain" } },
              { { "alefmaksuraisolated" }, { "alefmaksura" } },
              { { "behisolated" }, { "beh" } },
              { { "dadisolated" }, { "dad" } },
              { { "dalisolated" }, { "dal" } },
              { { "farsiyehisolated" }, { "farsiyeh" } },
              { { "fehisolated" }, { "feh" } },
              { { "gafisolated" }, { "gaf" } },
              { { "ghainisolated" }, { "ghain" } },
              { { "hahisolated" }, { "hah" } },
              { { "jeemisolated" }, { "jeem" } },
              { { "jehisolated" }, { "jeh" } },
              { { "kafisolated" }, { "arabickaf" } },
              { { "kehehisolated" }, { "keheh" } },
              { { "khahisolated" }, { "khah" } },
              { { "meemisolated" }, { "meem" } },
              { { "noonisolated" }, { "noon" } },
              { { "pehisolated" }, { "peh" } },
              { { "qafisolated" }, { "qaf" } },
              { { "rehisolated" }, { "reh" } },
              { { "sadisolated" }, { "sad" } },
              { { "seenisolated" }, { "seen" } },
              { { "sheenisolated" }, { "sheen" } },
              { { "tahisolated" }, { "tah" } },
              { { "tchehisolated" }, { "tcheh" } },
              { { "tehisolated" }, { "teh" } },
              { { "tehmarbutaisolated" }, { "tehmarbuta" } },
              { { "thalisolated" }, { "thal" } },
              { { "thehisolated" }, { "theh" } },
              { { "vehisolated" }, { "veh" } },
              { { "wawisolated" }, { "waw" } },
              { { "yehisolated" }, { "yeh" } },
              { { "yehwithhamzaaboveisolated" }, { "yehwithhamzaabove" } },
              { { "zahisolated" }, { "zah" } },
              { { "zainisolated" }, { "zain" } },
          }
        },
        { "f1", "lu3", "arab", "*", "fina",
          new String[][][] {
              { { "ain" }, { "ainfinal" } },
              { { "alefmaksura" }, { "alefmaksurafinal" } },
              { { "alefwasla" }, { "alefwaslafinal" } },
              { { "alefwithhamzaabove" }, { "alefwithhamzaabovefinal" } },
              { { "alefwithhamzabelow" }, { "alefwithhamzabelowfinal" } },
              { { "alefwithmaddaabove" }, { "alefwithmaddaabovefinal" } },
              { { "arabicae" }, { "hehfinal" } },
              { { "arabicalef" }, { "aleffinal" } },
              { { "arabickaf" }, { "arabickaffinal" } },
              { { "beh" }, { "behfinal" } },
              { { "dad" }, { "dadfinal" } },
              { { "dal" }, { "dalfinal" } },
              { { "farsiyeh" }, { "farsiyehfinal" } },
              { { "feh" }, { "fehfinal" } },
              { { "gaf" }, { "gaffinal" } },
              { { "ghain" }, { "ghainfinal" } },
              { { "hah" }, { "hahfinal" } },
              { { "heh" }, { "hehfinal" } },
              { { "jeem" }, { "jeemfinal" } },
              { { "jeh" }, { "jehfinal" } },
              { { "keheh" }, { "kehehfinal" } },
              { { "khah" }, { "khahfinal" } },
              { { "lam" }, { "lamfinal" } },
              { { "meem" }, { "meemfinal" } },
              { { "noon" }, { "noonfinal" } },
              { { "peh" }, { "pehfinal" } },
              { { "qaf" }, { "qaffinal" } },
              { { "reh" }, { "rehfinal" } },
              { { "sad" }, { "sadfinal" } },
              { { "seen" }, { "seenfinal" } },
              { { "sheen" }, { "sheenfinal" } },
              { { "tah" }, { "tahfinal" } },
              { { "tcheh" }, { "tchehfinal" } },
              { { "teh" }, { "tehfinal" } },
              { { "tehmarbuta" }, { "tehmarbutafinal" } },
              { { "thal" }, { "thalfinal" } },
              { { "theh" }, { "thehfinal" } },
              { { "veh" }, { "vehfinal" } },
              { { "waw" }, { "wawfinal" } },
              { { "wawwithhamzaabove" }, { "wawwithhamzaabovefinal" } },
              { { "yeh" }, { "yehfinal" } },
              { { "yehwithhamzaabove" }, { "yehwithhamzaabovefinal" } },
              { { "zah" }, { "zahfinal" } },
              { { "zain" }, { "zainfinal" } },
          }
        },
        { "f1", "lu4", "arab", "*", "init",
          new String[][][] {
              { { "ain" }, { "aininitial" } },
              { { "alefmaksura" }, { "uni0649.init" } },
              { { "arabickaf" }, { "kafinitial" } },
              { { "beh" }, { "behinitial" } },
              { { "dad" }, { "dadinitial" } },
              { { "farsiyeh" }, { "yehinitial" } },
              { { "feh" }, { "fehinitial" } },
              { { "gaf" }, { "gafinitial" } },
              { { "ghain" }, { "ghaininitial" } },
              { { "hah" }, { "hahinitial" } },
              { { "heh" }, { "hehinitial" } },
              { { "jeem" }, { "jeeminitial" } },
              { { "keheh" }, { "kehehinitial" } },
              { { "khah" }, { "khahinitial" } },
              { { "lam" }, { "laminitial" } },
              { { "meem" }, { "meeminitial" } },
              { { "noon" }, { "nooninitial" } },
              { { "peh" }, { "pehinitial" } },
              { { "qaf" }, { "qafinitial" } },
              { { "sad" }, { "sadinitial" } },
              { { "seen" }, { "seeninitial" } },
              { { "sheen" }, { "sheeninitial" } },
              { { "tah" }, { "tahinitial" } },
              { { "tcheh" }, { "tchehinitial" } },
              { { "teh" }, { "tehinitial" } },
              { { "theh" }, { "thehinitial" } },
              { { "veh" }, { "vehinitial" } },
              { { "yeh" }, { "yehinitial" } },
              { { "yehwithhamzaabove" }, { "yehwithhamzaaboveinitial" } },
              { { "zah" }, { "zahinitial" } },
          }
        },
        { "f1", "lu5", "arab", "*", "medi",
          new String[][][] {
              { { "ain" }, { "ainmedial" } },
              { { "alefmaksura" }, { "uni0649.medi" } },
              { { "arabickaf" }, { "kafmedial" } },
              { { "beh" }, { "behmedial" } },
              { { "dad" }, { "dadmedial" } },
              { { "farsiyeh" }, { "yehmedial" } },
              { { "feh" }, { "fehmedial" } },
              { { "gaf" }, { "gafmedial" } },
              { { "ghain" }, { "ghainmedial" } },
              { { "hah" }, { "hahmedial" } },
              { { "heh" }, { "hehmedial" } },
              { { "jeem" }, { "jeemmedial" } },
              { { "keheh" }, { "kehehmedial" } },
              { { "khah" }, { "khahmedial" } },
              { { "lam" }, { "lammedial" } },
              { { "meem" }, { "meemmedial" } },
              { { "noon" }, { "noonmedial" } },
              { { "peh" }, { "pehmedial" } },
              { { "qaf" }, { "qafmedial" } },
              { { "sad" }, { "sadmedial" } },
              { { "seen" }, { "seenmedial" } },
              { { "sheen" }, { "sheenmedial" } },
              { { "tah" }, { "tahmedial" } },
              { { "tcheh" }, { "tchehmedial" } },
              { { "teh" }, { "tehmedial" } },
              { { "theh" }, { "thehmedial" } },
              { { "veh" }, { "vehmedial" } },
              { { "yeh" }, { "yehmedial" } },
              { { "yehwithhamzaabove" }, { "yehwithhamzaabovemedial" } },
              { { "zah" }, { "zahmedial" } },
          }
        },
        { "f1", "lu13", "arab", "*", "*",
          new String[][][] {
              { { "heh" }, { "hehisolated" } },
          }
        },
        // arab-003.ttx
        { "f2", "lu1", "arab", "dflt", "init",
          new String[][][] {
              { { "absJeemRetro1" }, { "absJeemRetro1Ini" } },
              { { "absJeemRetro2" }, { "absJeemRetro2Ini" } },
              { { "absJeemRetro3" }, { "absJeemRetro3Ini" } },
              { { "absLamRetro" }, { "absLamRetroIni" } },
              { { "absSheenRetro1" }, { "absSheenRetro1Ini" } },
              { { "absSheenRetro2" }, { "absSheenRetro2Ini" } },
              { { "absTchehRetro1" }, { "absTchehRetro1Ini" } },
              { { "absTchehRetro2" }, { "absTchehRetro2Ini" } },
              { { "uni0626" }, { "uni0626.init" } },
              { { "uni0628" }, { "uni0628.init" } },
              { { "uni062A" }, { "uni062A.init" } },
              { { "uni062B" }, { "uni062B.init" } },
              { { "uni062C" }, { "uni062C.init" } },
              { { "uni062D" }, { "uni062D.init" } },
              { { "uni062E" }, { "uni062E.init" } },
              { { "uni0633" }, { "uni0633.init" } },
              { { "uni0634" }, { "uni0634.init" } },
              { { "uni0635" }, { "uni0635.init" } },
              { { "uni0636" }, { "uni0636.init" } },
              { { "uni0637" }, { "uni0637.init" } },
              { { "uni0638" }, { "uni0638.init" } },
              { { "uni0639" }, { "uni0639.init" } },
              { { "uni063A" }, { "uni063A.init" } },
              { { "uni0641" }, { "uni0641.init" } },
              { { "uni0642" }, { "uni0642.init" } },
              { { "uni0643" }, { "uni0643.init" } },
              { { "uni0644" }, { "uni0644.init" } },
              { { "uni0645" }, { "uni0645.init" } },
              { { "uni0646" }, { "uni0646.init" } },
              { { "uni0647" }, { "uni0647.init" } },
              { { "uni0649" }, { "uni0649.init" } },
              { { "uni064A" }, { "uni064A.init" } },
              { { "uni064A.noDots" }, { "uni064A.init.noDots" } },
              { { "uni066E" }, { "uni066E.init" } },
              { { "uni066F" }, { "uni066F.init" } },
              { { "uni0678" }, { "uni0678.init" } },
              { { "uni0679" }, { "uni0679.init" } },
              { { "uni067A" }, { "uni067A.init" } },
              { { "uni067B" }, { "uni067B.init" } },
              { { "uni067C" }, { "uni067C.init" } },
              { { "uni067D" }, { "uni067D.init" } },
              { { "uni067E" }, { "uni067E.init" } },
              { { "uni067F" }, { "uni067F.init" } },
              { { "uni0680" }, { "uni0680.init" } },
              { { "uni0681" }, { "uni0681.init" } },
              { { "uni0682" }, { "uni0682.init" } },
              { { "uni0683" }, { "uni0683.init" } },
              { { "uni0684" }, { "uni0684.init" } },
              { { "uni0685" }, { "uni0685.init" } },
              { { "uni0686" }, { "uni0686.init" } },
              { { "uni0687" }, { "uni0687.init" } },
              { { "uni069A" }, { "uni069A.init" } },
              { { "uni069B" }, { "uni069B.init" } },
              { { "uni069C" }, { "uni069C.init" } },
              { { "uni069D" }, { "uni069D.init" } },
              { { "uni069E" }, { "uni069E.init" } },
              { { "uni069F" }, { "uni069F.init" } },
              { { "uni06A0" }, { "uni06A0.init" } },
              { { "uni06A1" }, { "uni06A1.init" } },
              { { "uni06A2" }, { "uni06A2.init" } },
              { { "uni06A3" }, { "uni06A3.init" } },
              { { "uni06A4" }, { "uni06A4.init" } },
              { { "uni06A5" }, { "uni06A5.init" } },
              { { "uni06A6" }, { "uni06A6.init" } },
              { { "uni06A7" }, { "uni06A7.init" } },
              { { "uni06A8" }, { "uni06A8.init" } },
              { { "uni06A9" }, { "uni06A9.init" } },
              { { "uni06AA" }, { "uni06AA.init" } },
              { { "uni06AB" }, { "uni06AB.init" } },
              { { "uni06AC" }, { "uni06AC.init" } },
              { { "uni06AD" }, { "uni06AD.init" } },
              { { "uni06AE" }, { "uni06AE.init" } },
              { { "uni06AF" }, { "uni06AF.init" } },
              { { "uni06B0" }, { "uni06B0.init" } },
              { { "uni06B1" }, { "uni06B1.init" } },
              { { "uni06B2" }, { "uni06B2.init" } },
              { { "uni06B3" }, { "uni06B3.init" } },
              { { "uni06B4" }, { "uni06B4.init" } },
              { { "uni06B5" }, { "uni06B5.init" } },
              { { "uni06B6" }, { "uni06B6.init" } },
              { { "uni06B7" }, { "uni06B7.init" } },
              { { "uni06B8" }, { "uni06B8.init" } },
              { { "uni06B9" }, { "uni06B9.init" } },
              { { "uni06BA" }, { "uni06BA.init" } },
              { { "uni06BB" }, { "uni06BB.init" } },
              { { "uni06BC" }, { "uni06BC.init" } },
              { { "uni06BD" }, { "uni06BD.init" } },
              { { "uni06BE" }, { "uni06BE.init" } },
              { { "uni06BF" }, { "uni06BF.init" } },
              { { "uni06C1" }, { "uni06C1.init" } },
              { { "uni06C2" }, { "uni06C2.init" } },
              { { "uni06CC" }, { "uni06CC.init" } },
              { { "uni06CE" }, { "uni06CE.init" } },
              { { "uni06D0" }, { "uni06D0.init" } },
              { { "uni06D1" }, { "uni06D1.init" } },
              { { "uni06FA" }, { "uni06FA.init" } },
              { { "uni06FB" }, { "uni06FB.init" } },
              { { "uni06FC" }, { "uni06FC.init" } },
              { { "uni06FF" }, { "uni06FF.init" } },
              { { "uni0750" }, { "uni0750.init" } },
              { { "uni0751" }, { "uni0751.init" } },
              { { "uni0752" }, { "uni0752.init" } },
              { { "uni0753" }, { "uni0753.init" } },
              { { "uni0754" }, { "uni0754.init" } },
              { { "uni0755" }, { "uni0755.init" } },
              { { "uni0756" }, { "uni0756.init" } },
              { { "uni0757" }, { "uni0757.init" } },
              { { "uni0758" }, { "uni0758.init" } },
              { { "uni075C" }, { "uni075C.init" } },
              { { "uni075D" }, { "uni075D.init" } },
              { { "uni075E" }, { "uni075E.init" } },
              { { "uni075F" }, { "uni075F.init" } },
              { { "uni0760" }, { "uni0760.init" } },
              { { "uni0761" }, { "uni0761.init" } },
              { { "uni0762" }, { "uni0762.init" } },
              { { "uni0763" }, { "uni0763.init" } },
              { { "uni0764" }, { "uni0764.init" } },
              { { "uni0765" }, { "uni0765.init" } },
              { { "uni0766" }, { "uni0766.init" } },
              { { "uni0767" }, { "uni0767.init" } },
              { { "uni0768" }, { "uni0768.init" } },
              { { "uni0769" }, { "uni0769.init" } },
              { { "uni076A" }, { "uni076A.init" } },
              { { "uni076D" }, { "uni076D.init" } },
           }
        },
        { "f2", "lu2", "arab", "dflt", "fina",
          new String[][][] {
              { { "absJeemRetro1" }, { "absJeemRetro1Fin" } },
              { { "absJeemRetro2" }, { "absJeemRetro2Fin" } },
              { { "absJeemRetro3" }, { "absJeemRetro3Fin" } },
              { { "absJehRetro1" }, { "absJehRetro1Fin" } },
              { { "absJehRetro2" }, { "absJehRetro2Fin" } },
              { { "absLamRetro" }, { "absLamRetroFin" } },
              { { "absSheenRetro1" }, { "absSheenRetro1Fin" } },
              { { "absSheenRetro2" }, { "absSheenRetro2Fin" } },
              { { "absTchehRetro1" }, { "absTchehRetro1Fin" } },
              { { "absTchehRetro2" }, { "absTchehRetro2Fin" } },
              { { "absWawDotBelow" }, { "absWawDotBelowFin" } },
              { { "uni0622" }, { "uni0622.fina" } },
              { { "uni0623" }, { "uni0623.fina" } },
              { { "uni0624" }, { "uni0624.fina" } },
              { { "uni0625" }, { "uni0625.fina" } },
              { { "uni0626" }, { "uni0626.fina" } },
              { { "uni0627" }, { "uni0627.fina" } },
              { { "uni0628" }, { "uni0628.fina" } },
              { { "uni0629" }, { "uni0629.fina" } },
              { { "uni062A" }, { "uni062A.fina" } },
              { { "uni062B" }, { "uni062B.fina" } },
              { { "uni062C" }, { "uni062C.fina" } },
              { { "uni062D" }, { "uni062D.fina" } },
              { { "uni062E" }, { "uni062E.fina" } },
              { { "uni062F" }, { "uni062F.fina" } },
              { { "uni0630" }, { "uni0630.fina" } },
              { { "uni0631" }, { "uni0631.fina" } },
              { { "uni0632" }, { "uni0632.fina" } },
              { { "uni0633" }, { "uni0633.fina" } },
              { { "uni0634" }, { "uni0634.fina" } },
              { { "uni0635" }, { "uni0635.fina" } },
              { { "uni0636" }, { "uni0636.fina" } },
              { { "uni0637" }, { "uni0637.fina" } },
              { { "uni0638" }, { "uni0638.fina" } },
              { { "uni0639" }, { "uni0639.fina" } },
              { { "uni063A" }, { "uni063A.fina" } },
              { { "uni0641" }, { "uni0641.fina" } },
              { { "uni0642" }, { "uni0642.fina" } },
              { { "uni0643" }, { "uni0643.fina" } },
              { { "uni0644" }, { "uni0644.fina" } },
              { { "uni0645" }, { "uni0645.fina" } },
              { { "uni0646" }, { "uni0646.fina" } },
              { { "uni0647" }, { "uni0647.fina" } },
              { { "uni0648" }, { "uni0648.fina" } },
              { { "uni0649" }, { "uni0649.fina" } },
              { { "uni064A" }, { "uni064A.fina" } },
              { { "uni064A.noDots" }, { "uni064A.fina.noDots" } },
              { { "uni066E" }, { "uni066E.fina" } },
              { { "uni066F" }, { "uni066F.fina" } },
              { { "uni0671" }, { "uni0671.fina" } },
              { { "uni0672" }, { "uni0672.fina" } },
              { { "uni0673" }, { "uni0673.fina" } },
              { { "uni0675" }, { "uni0675.fina" } },
              { { "uni0676" }, { "uni0676.fina" } },
              { { "uni0677" }, { "uni0677.fina" } },
              { { "uni0678" }, { "uni0678.fina" } },
              { { "uni0679" }, { "uni0679.fina" } },
              { { "uni067A" }, { "uni067A.fina" } },
              { { "uni067B" }, { "uni067B.fina" } },
              { { "uni067C" }, { "uni067C.fina" } },
              { { "uni067D" }, { "uni067D.fina" } },
              { { "uni067E" }, { "uni067E.fina" } },
              { { "uni067F" }, { "uni067F.fina" } },
              { { "uni0680" }, { "uni0680.fina" } },
              { { "uni0681" }, { "uni0681.fina" } },
              { { "uni0682" }, { "uni0682.fina" } },
              { { "uni0683" }, { "uni0683.fina" } },
              { { "uni0684" }, { "uni0684.fina" } },
              { { "uni0685" }, { "uni0685.fina" } },
              { { "uni0686" }, { "uni0686.fina" } },
              { { "uni0687" }, { "uni0687.fina" } },
              { { "uni0688" }, { "uni0688.fina" } },
              { { "uni0689" }, { "uni0689.fina" } },
              { { "uni068A" }, { "uni068A.fina" } },
              { { "uni068B" }, { "uni068B.fina" } },
              { { "uni068C" }, { "uni068C.fina" } },
              { { "uni068D" }, { "uni068D.fina" } },
              { { "uni068E" }, { "uni068E.fina" } },
              { { "uni068F" }, { "uni068F.fina" } },
              { { "uni0690" }, { "uni0690.fina" } },
              { { "uni0691" }, { "uni0691.fina" } },
              { { "uni0692" }, { "uni0692.fina" } },
              { { "uni0693" }, { "uni0693.fina" } },
              { { "uni0694" }, { "uni0694.fina" } },
              { { "uni0695" }, { "uni0695.fina" } },
              { { "uni0696" }, { "uni0696.fina" } },
              { { "uni0697" }, { "uni0697.fina" } },
              { { "uni0698" }, { "uni0698.fina" } },
              { { "uni0698.dotHat" }, { "uni0698.fina.dotHat" } },
              { { "uni0699" }, { "uni0699.fina" } },
              { { "uni069A" }, { "uni069A.fina" } },
              { { "uni069B" }, { "uni069B.fina" } },
              { { "uni069C" }, { "uni069C.fina" } },
              { { "uni069D" }, { "uni069D.fina" } },
              { { "uni069E" }, { "uni069E.fina" } },
              { { "uni069F" }, { "uni069F.fina" } },
              { { "uni06A0" }, { "uni06A0.fina" } },
              { { "uni06A1" }, { "uni06A1.fina" } },
              { { "uni06A2" }, { "uni06A2.fina" } },
              { { "uni06A3" }, { "uni06A3.fina" } },
              { { "uni06A4" }, { "uni06A4.fina" } },
              { { "uni06A5" }, { "uni06A5.fina" } },
              { { "uni06A6" }, { "uni06A6.fina" } },
              { { "uni06A7" }, { "uni06A7.fina" } },
              { { "uni06A8" }, { "uni06A8.fina" } },
              { { "uni06A9" }, { "uni06A9.fina" } },
              { { "uni06AA" }, { "uni06AA.fina" } },
              { { "uni06AB" }, { "uni06AB.fina" } },
              { { "uni06AC" }, { "uni06AC.fina" } },
              { { "uni06AD" }, { "uni06AD.fina" } },
              { { "uni06AE" }, { "uni06AE.fina" } },
              { { "uni06AF" }, { "uni06AF.fina" } },
              { { "uni06B0" }, { "uni06B0.fina" } },
              { { "uni06B1" }, { "uni06B1.fina" } },
              { { "uni06B2" }, { "uni06B2.fina" } },
              { { "uni06B3" }, { "uni06B3.fina" } },
              { { "uni06B4" }, { "uni06B4.fina" } },
              { { "uni06B5" }, { "uni06B5.fina" } },
              { { "uni06B6" }, { "uni06B6.fina" } },
              { { "uni06B7" }, { "uni06B7.fina" } },
              { { "uni06B8" }, { "uni06B8.fina" } },
              { { "uni06B9" }, { "uni06B9.fina" } },
              { { "uni06BA" }, { "uni06BA.fina" } },
              { { "uni06BB" }, { "uni06BB.fina" } },
              { { "uni06BC" }, { "uni06BC.fina" } },
              { { "uni06BD" }, { "uni06BD.fina" } },
              { { "uni06BE" }, { "uni06BE.fina" } },
              { { "uni06BF" }, { "uni06BF.fina" } },
              { { "uni06C0" }, { "uni06C0.fina" } },
              { { "uni06C1" }, { "uni06C1.fina" } },
              { { "uni06C2" }, { "uni06C2.fina" } },
              { { "uni06C3" }, { "uni06C3.fina" } },
              { { "uni06C4" }, { "uni06C4.fina" } },
              { { "uni06C5" }, { "uni06C5.fina" } },
              { { "uni06C6" }, { "uni06C6.fina" } },
              { { "uni06C7" }, { "uni06C7.fina" } },
              { { "uni06C8" }, { "uni06C8.fina" } },
              { { "uni06C9" }, { "uni06C9.fina" } },
              { { "uni06CA" }, { "uni06CA.fina" } },
              { { "uni06CB" }, { "uni06CB.fina" } },
              { { "uni06CC" }, { "uni06CC.fina" } },
              { { "uni06CD" }, { "uni06CD.fina" } },
              { { "uni06CE" }, { "uni06CE.fina" } },
              { { "uni06CF" }, { "uni06CF.fina" } },
              { { "uni06D0" }, { "uni06D0.fina" } },
              { { "uni06D1" }, { "uni06D1.fina" } },
              { { "uni06D2" }, { "uni06D2.fina" } },
              { { "uni06D3" }, { "uni06D3.fina" } },
              { { "uni06D5" }, { "uni06D5.fina" } },
              { { "uni06EE" }, { "uni06EE.fina" } },
              { { "uni06EF" }, { "uni06EF.fina" } },
              { { "uni06FA" }, { "uni06FA.fina" } },
              { { "uni06FB" }, { "uni06FB.fina" } },
              { { "uni06FC" }, { "uni06FC.fina" } },
              { { "uni06FF" }, { "uni06FF.fina" } },
              { { "uni0750" }, { "uni0750.fina" } },
              { { "uni0751" }, { "uni0751.fina" } },
              { { "uni0752" }, { "uni0752.fina" } },
              { { "uni0753" }, { "uni0753.fina" } },
              { { "uni0754" }, { "uni0754.fina" } },
              { { "uni0755" }, { "uni0755.fina" } },
              { { "uni0756" }, { "uni0756.fina" } },
              { { "uni0757" }, { "uni0757.fina" } },
              { { "uni0758" }, { "uni0758.fina" } },
              { { "uni0759" }, { "uni0759.fina" } },
              { { "uni075A" }, { "uni075A.fina" } },
              { { "uni075B" }, { "uni075B.fina" } },
              { { "uni075C" }, { "uni075C.fina" } },
              { { "uni075D" }, { "uni075D.fina" } },
              { { "uni075E" }, { "uni075E.fina" } },
              { { "uni075F" }, { "uni075F.fina" } },
              { { "uni0760" }, { "uni0760.fina" } },
              { { "uni0761" }, { "uni0761.fina" } },
              { { "uni0762" }, { "uni0762.fina" } },
              { { "uni0763" }, { "uni0763.fina" } },
              { { "uni0764" }, { "uni0764.fina" } },
              { { "uni0765" }, { "uni0765.fina" } },
              { { "uni0766" }, { "uni0766.fina" } },
              { { "uni0767" }, { "uni0767.fina" } },
              { { "uni0768" }, { "uni0768.fina" } },
              { { "uni0769" }, { "uni0769.fina" } },
              { { "uni076A" }, { "uni076A.fina" } },
              { { "uni076B" }, { "uni076B.fina" } },
              { { "uni076C" }, { "uni076C.fina" } },
              { { "uni076D" }, { "uni076D.fina" } },
          }
        },
        { "f2", "lu3", "arab", "dflt", "medi",
          new String[][][] {
              { { "absJeemRetro1" }, { "absJeemRetro1Med" } },
              { { "absJeemRetro2" }, { "absJeemRetro2Med" } },
              { { "absJeemRetro3" }, { "absJeemRetro3Med" } },
              { { "absLamRetro" }, { "absLamRetroMed" } },
              { { "absSheenRetro1" }, { "absSheenRetro1Med" } },
              { { "absSheenRetro2" }, { "absSheenRetro2Med" } },
              { { "absTchehRetro1" }, { "absTchehRetro1Med" } },
              { { "absTchehRetro2" }, { "absTchehRetro2Med" } },
              { { "uni0626" }, { "uni0626.medi" } },
              { { "uni0628" }, { "uni0628.medi" } },
              { { "uni062A" }, { "uni062A.medi" } },
              { { "uni062B" }, { "uni062B.medi" } },
              { { "uni062C" }, { "uni062C.medi" } },
              { { "uni062D" }, { "uni062D.medi" } },
              { { "uni062E" }, { "uni062E.medi" } },
              { { "uni0633" }, { "uni0633.medi" } },
              { { "uni0634" }, { "uni0634.medi" } },
              { { "uni0635" }, { "uni0635.medi" } },
              { { "uni0636" }, { "uni0636.medi" } },
              { { "uni0637" }, { "uni0637.medi" } },
              { { "uni0638" }, { "uni0638.medi" } },
              { { "uni0639" }, { "uni0639.medi" } },
              { { "uni063A" }, { "uni063A.medi" } },
              { { "uni0641" }, { "uni0641.medi" } },
              { { "uni0642" }, { "uni0642.medi" } },
              { { "uni0643" }, { "uni0643.medi" } },
              { { "uni0644" }, { "uni0644.medi" } },
              { { "uni0645" }, { "uni0645.medi" } },
              { { "uni0646" }, { "uni0646.medi" } },
              { { "uni0647" }, { "uni0647.medi" } },
              { { "uni0649" }, { "uni0649.medi" } },
              { { "uni064A" }, { "uni064A.medi" } },
              { { "uni064A.noDots" }, { "uni064A.medi.noDots" } },
              { { "uni066E" }, { "uni066E.medi" } },
              { { "uni066F" }, { "uni066F.medi" } },
              { { "uni0678" }, { "uni0678.medi" } },
              { { "uni0679" }, { "uni0679.medi" } },
              { { "uni067A" }, { "uni067A.medi" } },
              { { "uni067B" }, { "uni067B.medi" } },
              { { "uni067C" }, { "uni067C.medi" } },
              { { "uni067D" }, { "uni067D.medi" } },
              { { "uni067E" }, { "uni067E.medi" } },
              { { "uni067F" }, { "uni067F.medi" } },
              { { "uni0680" }, { "uni0680.medi" } },
              { { "uni0681" }, { "uni0681.medi" } },
              { { "uni0682" }, { "uni0682.medi" } },
              { { "uni0683" }, { "uni0683.medi" } },
              { { "uni0684" }, { "uni0684.medi" } },
              { { "uni0685" }, { "uni0685.medi" } },
              { { "uni0686" }, { "uni0686.medi" } },
              { { "uni0687" }, { "uni0687.medi" } },
              { { "uni069A" }, { "uni069A.medi" } },
              { { "uni069B" }, { "uni069B.medi" } },
              { { "uni069C" }, { "uni069C.medi" } },
              { { "uni069D" }, { "uni069D.medi" } },
              { { "uni069E" }, { "uni069E.medi" } },
              { { "uni069F" }, { "uni069F.medi" } },
              { { "uni06A0" }, { "uni06A0.medi" } },
              { { "uni06A1" }, { "uni06A1.medi" } },
              { { "uni06A2" }, { "uni06A2.medi" } },
              { { "uni06A3" }, { "uni06A3.medi" } },
              { { "uni06A4" }, { "uni06A4.medi" } },
              { { "uni06A5" }, { "uni06A5.medi" } },
              { { "uni06A6" }, { "uni06A6.medi" } },
              { { "uni06A7" }, { "uni06A7.medi" } },
              { { "uni06A8" }, { "uni06A8.medi" } },
              { { "uni06A9" }, { "uni06A9.medi" } },
              { { "uni06AA" }, { "uni06AA.medi" } },
              { { "uni06AB" }, { "uni06AB.medi" } },
              { { "uni06AC" }, { "uni06AC.medi" } },
              { { "uni06AD" }, { "uni06AD.medi" } },
              { { "uni06AE" }, { "uni06AE.medi" } },
              { { "uni06AF" }, { "uni06AF.medi" } },
              { { "uni06B0" }, { "uni06B0.medi" } },
              { { "uni06B1" }, { "uni06B1.medi" } },
              { { "uni06B2" }, { "uni06B2.medi" } },
              { { "uni06B3" }, { "uni06B3.medi" } },
              { { "uni06B4" }, { "uni06B4.medi" } },
              { { "uni06B5" }, { "uni06B5.medi" } },
              { { "uni06B6" }, { "uni06B6.medi" } },
              { { "uni06B7" }, { "uni06B7.medi" } },
              { { "uni06B8" }, { "uni06B8.medi" } },
              { { "uni06B9" }, { "uni06B9.medi" } },
              { { "uni06BA" }, { "uni06BA.medi" } },
              { { "uni06BB" }, { "uni06BB.medi" } },
              { { "uni06BC" }, { "uni06BC.medi" } },
              { { "uni06BD" }, { "uni06BD.medi" } },
              { { "uni06BE" }, { "uni06BE.medi" } },
              { { "uni06BF" }, { "uni06BF.medi" } },
              { { "uni06C1" }, { "uni06C1.medi" } },
              { { "uni06C2" }, { "uni06C2.medi" } },
              { { "uni06CC" }, { "uni06CC.medi" } },
              { { "uni06CE" }, { "uni06CE.medi" } },
              { { "uni06D0" }, { "uni06D0.medi" } },
              { { "uni06D1" }, { "uni06D1.medi" } },
              { { "uni06FA" }, { "uni06FA.medi" } },
              { { "uni06FB" }, { "uni06FB.medi" } },
              { { "uni06FC" }, { "uni06FC.medi" } },
              { { "uni06FF" }, { "uni06FF.medi" } },
              { { "uni0750" }, { "uni0750.medi" } },
              { { "uni0751" }, { "uni0751.medi" } },
              { { "uni0752" }, { "uni0752.medi" } },
              { { "uni0753" }, { "uni0753.medi" } },
              { { "uni0754" }, { "uni0754.medi" } },
              { { "uni0755" }, { "uni0755.medi" } },
              { { "uni0756" }, { "uni0756.medi" } },
              { { "uni0757" }, { "uni0757.medi" } },
              { { "uni0758" }, { "uni0758.medi" } },
              { { "uni075C" }, { "uni075C.medi" } },
              { { "uni075D" }, { "uni075D.medi" } },
              { { "uni075E" }, { "uni075E.medi" } },
              { { "uni075F" }, { "uni075F.medi" } },
              { { "uni0760" }, { "uni0760.medi" } },
              { { "uni0761" }, { "uni0761.medi" } },
              { { "uni0762" }, { "uni0762.medi" } },
              { { "uni0763" }, { "uni0763.medi" } },
              { { "uni0764" }, { "uni0764.medi" } },
              { { "uni0765" }, { "uni0765.medi" } },
              { { "uni0766" }, { "uni0766.medi" } },
              { { "uni0767" }, { "uni0767.medi" } },
              { { "uni0768" }, { "uni0768.medi" } },
              { { "uni0769" }, { "uni0769.medi" } },
              { { "uni076A" }, { "uni076A.medi" } },
              { { "uni076D" }, { "uni076D.medi" } },
          }
        },
        { "f2", "lu9", "arab", "SND ", "calt",
          new String[][][] {
              { { "uni060C" }, { "uni060C.downward" } },
              { { "uni061B" }, { "uni061B.downward" } },
              { { "uni0645" }, { "uni0645.sindhi" } },
              { { "uni0645.fina" }, { "uni0645.fina.sindhi" } },
              { { "uni0647" }, { "uni0647.knotted" } },
              { { "uni0647.fina" }, { "uni0647.fina.knottedHigh" } },
              { { "uni0647.medi" }, { "uni0647.medi.knottedHigh" } },
              { { "uni064C" }, { "uni064C.sixNine" } },
              { { "uni06F6" }, { "uni06F6.urdu" } },
              { { "uni06F7" }, { "uni06F7.urdu" } },
          }
        },
        { "f2", "lu10", "arab", "URD ", "calt",
          new String[][][] {
              { { "uni0647.fina" }, { "uni0647.fina.hooked" } },
              { { "uni0647.init" }, { "uni0647.init.hooked" } },
              { { "uni0647.medi" }, { "uni0647.medi.hooked" } },
              { { "uni06F4" }, { "uni06F4.urdu" } },
              { { "uni06F6" }, { "uni06F6.urdu" } },
              { { "uni06F7" }, { "uni06F7.urdu" } },
          }
        },
        { "f2", "lu11", "arab", "KUR ", "calt",
          new String[][][] {
              { { "uni0647" }, { "uni0647.knotted" } },
              { { "uni0647.fina" }, { "uni0647.fina.knottedHigh" } },
          }
        },
        { "f2", "lu12", "latn", "dflt", "ccmp",
          new String[][][] {
              { { "asterisk.arab" }, { "asterisk" } },
              { { "colon.arab" }, { "colon" } },
              { { "exclam.arab" }, { "exclam" } },
              { { "parenleft.arab" }, { "parenleft" } },
              { { "parenright.arab" }, { "parenright" } },
              { { "quotedblleft.arab" }, { "quotedblleft" } },
              { { "quotedblright.arab" }, { "quotedblright" } },
              { { "quoteleft.arab" }, { "quoteleft" } },
              { { "quoteright.arab" }, { "quoteright" } },
          }
        },
        { "f2", "lu14", "arab", "*", "*",
          new String[][][] {
              { { "absLamRetroIni" }, { "absLamRetroIni.preAlef" } },
              { { "absLamRetroMed" }, { "absLamRetroMed.preAlef" } },
              { { "uni0644.init" }, { "uni0644.init.preAlef" } },
              { { "uni0644.medi" }, { "uni0644.medi.preAlef" } },
              { { "uni06B5.init" }, { "uni06B5.init.preAlef" } },
              { { "uni06B5.medi" }, { "uni06B5.medi.preAlef" } },
              { { "uni06B6.init" }, { "uni06B6.init.preAlef" } },
              { { "uni06B6.medi" }, { "uni06B6.medi.preAlef" } },
              { { "uni06B7.init" }, { "uni06B7.init.preAlef" } },
              { { "uni06B7.medi" }, { "uni06B7.medi.preAlef" } },
              { { "uni06B8.init" }, { "uni06B8.init.preAlef" } },
              { { "uni06B8.medi" }, { "uni06B8.medi.preAlef" } },
              { { "uni076A.init" }, { "uni076A.init.preAlef" } },
              { { "uni076A.medi" }, { "uni076A.medi.preAlef" } },
          }
        },
        { "f2", "lu15", "arab", "*", "*",
          new String[][][] {
              { { "uni0622.fina" }, { "uni0622.fina.postLamIni" } },
              { { "uni0623.fina" }, { "uni0623.fina.postLamIni" } },
              { { "uni0625.fina" }, { "uni0625.fina.postLamIni" } },
              { { "uni0627.fina" }, { "uni0627.fina.postLamIni" } },
              { { "uni0671.fina" }, { "uni0671.fina.postLamIni" } },
              { { "uni0672.fina" }, { "uni0672.fina.postLamIni" } },
              { { "uni0673.fina" }, { "uni0673.fina.postLamIni" } },
              { { "uni0675.fina" }, { "uni0675.fina.postLamIni" } },
          }
        },
        { "f2", "lu16", "arab", "*", "*",
          new String[][][] {
              { { "uni0622.fina" }, { "uni0622.fina.postLamMed" } },
              { { "uni0623.fina" }, { "uni0623.fina.postLamMed" } },
              { { "uni0625.fina" }, { "uni0625.fina.postLamMed" } },
              { { "uni0627.fina" }, { "uni0627.fina.postLamMed" } },
              { { "uni0671.fina" }, { "uni0671.fina.postLamMed" } },
              { { "uni0672.fina" }, { "uni0672.fina.postLamMed" } },
              { { "uni0673.fina" }, { "uni0673.fina.postLamMed" } },
              { { "uni0675.fina" }, { "uni0675.fina.postLamMed" } },
          }
        },
        { "f2", "lu17", "arab", "*", "*",
          new String[][][] {
              { { "uni0670" }, { "uni0670.large" } },
          }
        },
        { "f2", "lu18", "arab", "*", "*",
          new String[][][] {
              { { "uni06DD" }, { "uni06DD.3" } },
          }
        },
        { "f2", "lu19", "arab", "*", "*",
          new String[][][] {
              { { "uni06DD" }, { "uni06DD.2" } },
          }
        },
        { "f2", "lu20", "arab", "*", "*",
          new String[][][] {
              { { "eight" }, { "eightMedium" } },
              { { "five" }, { "fiveMedium" } },
              { { "four" }, { "fourMedium" } },
              { { "nine" }, { "nineMedium" } },
              { { "one" }, { "oneMedium" } },
              { { "seven" }, { "sevenMedium" } },
              { { "six" }, { "sixMedium" } },
              { { "three" }, { "threeMedium" } },
              { { "two" }, { "twoMedium" } },
              { { "uni0660" }, { "uni0660.Medium" } },
              { { "uni0661" }, { "uni0661.Medium" } },
              { { "uni0662" }, { "uni0662.Medium" } },
              { { "uni0663" }, { "uni0663.Medium" } },
              { { "uni0664" }, { "uni0664.Medium" } },
              { { "uni0665" }, { "uni0665.Medium" } },
              { { "uni0666" }, { "uni0666.Medium" } },
              { { "uni0667" }, { "uni0667.Medium" } },
              { { "uni0668" }, { "uni0668.Medium" } },
              { { "uni0669" }, { "uni0669.Medium" } },
              { { "uni06F0" }, { "uni06F0.Medium" } },
              { { "uni06F1" }, { "uni06F1.Medium" } },
              { { "uni06F2" }, { "uni06F2.Medium" } },
              { { "uni06F3" }, { "uni06F3.Medium" } },
              { { "uni06F4" }, { "uni06F4.Medium" } },
              { { "uni06F4.urdu" }, { "uni06F4.Medium.urdu" } },
              { { "uni06F5" }, { "uni06F5.Medium" } },
              { { "uni06F6" }, { "uni06F6.Medium" } },
              { { "uni06F6.urdu" }, { "uni06F6.Medium.urdu" } },
              { { "uni06F7" }, { "uni06F7.Medium" } },
              { { "uni06F7.urdu" }, { "uni06F7.Medium.urdu" } },
              { { "uni06F8" }, { "uni06F8.Medium" } },
              { { "uni06F9" }, { "uni06F9.Medium" } },
              { { "zero" }, { "zeroMedium" } },
          }
        },
        { "f2", "lu21", "arab", "*", "*",
          new String[][][] {
              { { "eight" }, { "eightSmall" } },
              { { "five" }, { "fiveSmall" } },
              { { "four" }, { "fourSmall" } },
              { { "nine" }, { "nineSmall" } },
              { { "one" }, { "oneSmall" } },
              { { "seven" }, { "sevenSmall" } },
              { { "six" }, { "sixSmall" } },
              { { "three" }, { "threeSmall" } },
              { { "two" }, { "twoSmall" } },
              { { "uni0660" }, { "uni0660.Small" } },
              { { "uni0661" }, { "uni0661.Small" } },
              { { "uni0662" }, { "uni0662.Small" } },
              { { "uni0663" }, { "uni0663.Small" } },
              { { "uni0664" }, { "uni0664.Small" } },
              { { "uni0665" }, { "uni0665.Small" } },
              { { "uni0666" }, { "uni0666.Small" } },
              { { "uni0667" }, { "uni0667.Small" } },
              { { "uni0668" }, { "uni0668.Small" } },
              { { "uni0669" }, { "uni0669.Small" } },
              { { "uni06F0" }, { "uni06F0.Small" } },
              { { "uni06F1" }, { "uni06F1.Small" } },
              { { "uni06F2" }, { "uni06F2.Small" } },
              { { "uni06F3" }, { "uni06F3.Small" } },
              { { "uni06F4" }, { "uni06F4.Small" } },
              { { "uni06F4.urdu" }, { "uni06F4.Small.urdu" } },
              { { "uni06F5" }, { "uni06F5.Small" } },
              { { "uni06F6" }, { "uni06F6.Small" } },
              { { "uni06F6.urdu" }, { "uni06F6.Small.urdu" } },
              { { "uni06F7" }, { "uni06F7.Small" } },
              { { "uni06F7.urdu" }, { "uni06F7.Small.urdu" } },
              { { "uni06F8" }, { "uni06F8.Small" } },
              { { "uni06F9" }, { "uni06F9.Small" } },
              { { "zero" }, { "zeroSmall" } },
          }
        },
        // arab-004.ttx
        { "f3", "lu1", "arab", "dflt", "init",
          new String[][][] {
              { { "absJeemRetro1" }, { "absJeemRetro1Ini" } },
              { { "absJeemRetro2" }, { "absJeemRetro2Ini" } },
              { { "absJeemRetro3" }, { "absJeemRetro3Ini" } },
              { { "absLamRetro" }, { "absLamRetroIni" } },
              { { "absSheenRetro1" }, { "absSheenRetro1Ini" } },
              { { "absSheenRetro2" }, { "absSheenRetro2Ini" } },
              { { "absTchehRetro1" }, { "absTchehRetro1Ini" } },
              { { "absTchehRetro2" }, { "absTchehRetro2Ini" } },
              { { "uni0626" }, { "uni0626.init" } },
              { { "uni0628" }, { "uni0628.init" } },
              { { "uni062A" }, { "uni062A.init" } },
              { { "uni062B" }, { "uni062B.init" } },
              { { "uni062C" }, { "uni062C.init" } },
              { { "uni062D" }, { "uni062D.init" } },
              { { "uni062E" }, { "uni062E.init" } },
              { { "uni0633" }, { "uni0633.init" } },
              { { "uni0634" }, { "uni0634.init" } },
              { { "uni0635" }, { "uni0635.init" } },
              { { "uni0636" }, { "uni0636.init" } },
              { { "uni0637" }, { "uni0637.init" } },
              { { "uni0638" }, { "uni0638.init" } },
              { { "uni0639" }, { "uni0639.init" } },
              { { "uni063A" }, { "uni063A.init" } },
              { { "uni0641" }, { "uni0641.init" } },
              { { "uni0642" }, { "uni0642.init" } },
              { { "uni0643" }, { "uni0643.init" } },
              { { "uni0644" }, { "uni0644.init" } },
              { { "uni0645" }, { "uni0645.init" } },
              { { "uni0646" }, { "uni0646.init" } },
              { { "uni0647" }, { "uni0647.init" } },
              { { "uni0649" }, { "uni0649.init" } },
              { { "uni064A" }, { "uni064A.init" } },
              { { "uni064A.noDots" }, { "uni064A.init.noDots" } },
              { { "uni066E" }, { "uni066E.init" } },
              { { "uni066F" }, { "uni066F.init" } },
              { { "uni0678" }, { "uni0678.init" } },
              { { "uni0679" }, { "uni0679.init" } },
              { { "uni067A" }, { "uni067A.init" } },
              { { "uni067B" }, { "uni067B.init" } },
              { { "uni067C" }, { "uni067C.init" } },
              { { "uni067D" }, { "uni067D.init" } },
              { { "uni067E" }, { "uni067E.init" } },
              { { "uni067F" }, { "uni067F.init" } },
              { { "uni0680" }, { "uni0680.init" } },
              { { "uni0681" }, { "uni0681.init" } },
              { { "uni0682" }, { "uni0682.init" } },
              { { "uni0683" }, { "uni0683.init" } },
              { { "uni0684" }, { "uni0684.init" } },
              { { "uni0685" }, { "uni0685.init" } },
              { { "uni0686" }, { "uni0686.init" } },
              { { "uni0687" }, { "uni0687.init" } },
              { { "uni069A" }, { "uni069A.init" } },
              { { "uni069B" }, { "uni069B.init" } },
              { { "uni069C" }, { "uni069C.init" } },
              { { "uni069D" }, { "uni069D.init" } },
              { { "uni069E" }, { "uni069E.init" } },
              { { "uni069F" }, { "uni069F.init" } },
              { { "uni06A0" }, { "uni06A0.init" } },
              { { "uni06A1" }, { "uni06A1.init" } },
              { { "uni06A2" }, { "uni06A2.init" } },
              { { "uni06A3" }, { "uni06A3.init" } },
              { { "uni06A4" }, { "uni06A4.init" } },
              { { "uni06A5" }, { "uni06A5.init" } },
              { { "uni06A6" }, { "uni06A6.init" } },
              { { "uni06A7" }, { "uni06A7.init" } },
              { { "uni06A8" }, { "uni06A8.init" } },
              { { "uni06A9" }, { "uni06A9.init" } },
              { { "uni06AA" }, { "uni06AA.init" } },
              { { "uni06AB" }, { "uni06AB.init" } },
              { { "uni06AC" }, { "uni06AC.init" } },
              { { "uni06AD" }, { "uni06AD.init" } },
              { { "uni06AE" }, { "uni06AE.init" } },
              { { "uni06AF" }, { "uni06AF.init" } },
              { { "uni06B0" }, { "uni06B0.init" } },
              { { "uni06B1" }, { "uni06B1.init" } },
              { { "uni06B2" }, { "uni06B2.init" } },
              { { "uni06B3" }, { "uni06B3.init" } },
              { { "uni06B4" }, { "uni06B4.init" } },
              { { "uni06B5" }, { "uni06B5.init" } },
              { { "uni06B6" }, { "uni06B6.init" } },
              { { "uni06B7" }, { "uni06B7.init" } },
              { { "uni06B8" }, { "uni06B8.init" } },
              { { "uni06B9" }, { "uni06B9.init" } },
              { { "uni06BA" }, { "uni06BA.init" } },
              { { "uni06BB" }, { "uni06BB.init" } },
              { { "uni06BC" }, { "uni06BC.init" } },
              { { "uni06BD" }, { "uni06BD.init" } },
              { { "uni06BE" }, { "uni06BE.init" } },
              { { "uni06BF" }, { "uni06BF.init" } },
              { { "uni06C1" }, { "uni06C1.init" } },
              { { "uni06CC" }, { "uni06CC.init" } },
              { { "uni06CE" }, { "uni06CE.init" } },
              { { "uni06D0" }, { "uni06D0.init" } },
              { { "uni06D1" }, { "uni06D1.init" } },
              { { "uni06FA" }, { "uni06FA.init" } },
              { { "uni06FB" }, { "uni06FB.init" } },
              { { "uni06FC" }, { "uni06FC.init" } },
              { { "uni06FF" }, { "uni06FF.init" } },
              { { "uni0750" }, { "uni0750.init" } },
              { { "uni0751" }, { "uni0751.init" } },
              { { "uni0752" }, { "uni0752.init" } },
              { { "uni0753" }, { "uni0753.init" } },
              { { "uni0754" }, { "uni0754.init" } },
              { { "uni0755" }, { "uni0755.init" } },
              { { "uni0756" }, { "uni0756.init" } },
              { { "uni0757" }, { "uni0757.init" } },
              { { "uni0758" }, { "uni0758.init" } },
              { { "uni075C" }, { "uni075C.init" } },
              { { "uni075D" }, { "uni075D.init" } },
              { { "uni075E" }, { "uni075E.init" } },
              { { "uni075F" }, { "uni075F.init" } },
              { { "uni0760" }, { "uni0760.init" } },
              { { "uni0761" }, { "uni0761.init" } },
              { { "uni0762" }, { "uni0762.init" } },
              { { "uni0763" }, { "uni0763.init" } },
              { { "uni0764" }, { "uni0764.init" } },
              { { "uni0765" }, { "uni0765.init" } },
              { { "uni0766" }, { "uni0766.init" } },
              { { "uni0767" }, { "uni0767.init" } },
              { { "uni0768" }, { "uni0768.init" } },
              { { "uni0769" }, { "uni0769.init" } },
              { { "uni076A" }, { "uni076A.init" } },
              { { "uni076D" }, { "uni076D.init" } },
          }
        },
        { "f3", "lu2", "arab", "dflt", "fina",
          new String[][][] {
              { { "absJeemRetro1" }, { "absJeemRetro1Fin" } },
              { { "absJeemRetro2" }, { "absJeemRetro2Fin" } },
              { { "absJeemRetro3" }, { "absJeemRetro3Fin" } },
              { { "absJehRetro1" }, { "absJehRetro1Fin" } },
              { { "absJehRetro2" }, { "absJehRetro2Fin" } },
              { { "absLamRetro" }, { "absLamRetroFin" } },
              { { "absSheenRetro1" }, { "absSheenRetro1Fin" } },
              { { "absSheenRetro2" }, { "absSheenRetro2Fin" } },
              { { "absTchehRetro1" }, { "absTchehRetro1Fin" } },
              { { "absTchehRetro2" }, { "absTchehRetro2Fin" } },
              { { "absWawDotBelow" }, { "absWawDotBelowFin" } },
              { { "uni0622" }, { "uni0622.fina" } },
              { { "uni0623" }, { "uni0623.fina" } },
              { { "uni0624" }, { "uni0624.fina" } },
              { { "uni0625" }, { "uni0625.fina" } },
              { { "uni0626" }, { "uni0626.fina" } },
              { { "uni0627" }, { "uni0627.fina" } },
              { { "uni0628" }, { "uni0628.fina" } },
              { { "uni0629" }, { "uni0629.fina" } },
              { { "uni062A" }, { "uni062A.fina" } },
              { { "uni062B" }, { "uni062B.fina" } },
              { { "uni062C" }, { "uni062C.fina" } },
              { { "uni062D" }, { "uni062D.fina" } },
              { { "uni062E" }, { "uni062E.fina" } },
              { { "uni062F" }, { "uni062F.fina" } },
              { { "uni0630" }, { "uni0630.fina" } },
              { { "uni0631" }, { "uni0631.fina" } },
              { { "uni0632" }, { "uni0632.fina" } },
              { { "uni0633" }, { "uni0633.fina" } },
              { { "uni0634" }, { "uni0634.fina" } },
              { { "uni0635" }, { "uni0635.fina" } },
              { { "uni0636" }, { "uni0636.fina" } },
              { { "uni0637" }, { "uni0637.fina" } },
              { { "uni0638" }, { "uni0638.fina" } },
              { { "uni0639" }, { "uni0639.fina" } },
              { { "uni063A" }, { "uni063A.fina" } },
              { { "uni0641" }, { "uni0641.fina" } },
              { { "uni0642" }, { "uni0642.fina" } },
              { { "uni0643" }, { "uni0643.fina" } },
              { { "uni0644" }, { "uni0644.fina" } },
              { { "uni0645" }, { "uni0645.fina" } },
              { { "uni0646" }, { "uni0646.fina" } },
              { { "uni0647" }, { "uni0647.fina" } },
              { { "uni0648" }, { "uni0648.fina" } },
              { { "uni0649" }, { "uni0649.fina" } },
              { { "uni064A" }, { "uni064A.fina" } },
              { { "uni064A.noDots" }, { "uni064A.fina.noDots" } },
              { { "uni066E" }, { "uni066E.fina" } },
              { { "uni066F" }, { "uni066F.fina" } },
              { { "uni0671" }, { "uni0671.fina" } },
              { { "uni0672" }, { "uni0672.fina" } },
              { { "uni0673" }, { "uni0673.fina" } },
              { { "uni0675" }, { "uni0675.fina" } },
              { { "uni0676" }, { "uni0676.fina" } },
              { { "uni0677" }, { "uni0677.fina" } },
              { { "uni0678" }, { "uni0678.fina" } },
              { { "uni0679" }, { "uni0679.fina" } },
              { { "uni067A" }, { "uni067A.fina" } },
              { { "uni067B" }, { "uni067B.fina" } },
              { { "uni067C" }, { "uni067C.fina" } },
              { { "uni067D" }, { "uni067D.fina" } },
              { { "uni067E" }, { "uni067E.fina" } },
              { { "uni067F" }, { "uni067F.fina" } },
              { { "uni0680" }, { "uni0680.fina" } },
              { { "uni0681" }, { "uni0681.fina" } },
              { { "uni0682" }, { "uni0682.fina" } },
              { { "uni0683" }, { "uni0683.fina" } },
              { { "uni0684" }, { "uni0684.fina" } },
              { { "uni0685" }, { "uni0685.fina" } },
              { { "uni0686" }, { "uni0686.fina" } },
              { { "uni0687" }, { "uni0687.fina" } },
              { { "uni0688" }, { "uni0688.fina" } },
              { { "uni0689" }, { "uni0689.fina" } },
              { { "uni068A" }, { "uni068A.fina" } },
              { { "uni068B" }, { "uni068B.fina" } },
              { { "uni068C" }, { "uni068C.fina" } },
              { { "uni068D" }, { "uni068D.fina" } },
              { { "uni068E" }, { "uni068E.fina" } },
              { { "uni068F" }, { "uni068F.fina" } },
              { { "uni0690" }, { "uni0690.fina" } },
              { { "uni0691" }, { "uni0691.fina" } },
              { { "uni0692" }, { "uni0692.fina" } },
              { { "uni0693" }, { "uni0693.fina" } },
              { { "uni0694" }, { "uni0694.fina" } },
              { { "uni0695" }, { "uni0695.fina" } },
              { { "uni0696" }, { "uni0696.fina" } },
              { { "uni0697" }, { "uni0697.fina" } },
              { { "uni0698" }, { "uni0698.fina" } },
              { { "uni0698.dotHat" }, { "uni0698.fina.dotHat" } },
              { { "uni0699" }, { "uni0699.fina" } },
              { { "uni069A" }, { "uni069A.fina" } },
              { { "uni069B" }, { "uni069B.fina" } },
              { { "uni069C" }, { "uni069C.fina" } },
              { { "uni069D" }, { "uni069D.fina" } },
              { { "uni069E" }, { "uni069E.fina" } },
              { { "uni069F" }, { "uni069F.fina" } },
              { { "uni06A0" }, { "uni06A0.fina" } },
              { { "uni06A1" }, { "uni06A1.fina" } },
              { { "uni06A2" }, { "uni06A2.fina" } },
              { { "uni06A3" }, { "uni06A3.fina" } },
              { { "uni06A4" }, { "uni06A4.fina" } },
              { { "uni06A5" }, { "uni06A5.fina" } },
              { { "uni06A6" }, { "uni06A6.fina" } },
              { { "uni06A7" }, { "uni06A7.fina" } },
              { { "uni06A8" }, { "uni06A8.fina" } },
              { { "uni06A9" }, { "uni06A9.fina" } },
              { { "uni06AA" }, { "uni06AA.fina" } },
              { { "uni06AB" }, { "uni06AB.fina" } },
              { { "uni06AC" }, { "uni06AC.fina" } },
              { { "uni06AD" }, { "uni06AD.fina" } },
              { { "uni06AE" }, { "uni06AE.fina" } },
              { { "uni06AF" }, { "uni06AF.fina" } },
              { { "uni06B0" }, { "uni06B0.fina" } },
              { { "uni06B1" }, { "uni06B1.fina" } },
              { { "uni06B2" }, { "uni06B2.fina" } },
              { { "uni06B3" }, { "uni06B3.fina" } },
              { { "uni06B4" }, { "uni06B4.fina" } },
              { { "uni06B5" }, { "uni06B5.fina" } },
              { { "uni06B6" }, { "uni06B6.fina" } },
              { { "uni06B7" }, { "uni06B7.fina" } },
              { { "uni06B8" }, { "uni06B8.fina" } },
              { { "uni06B9" }, { "uni06B9.fina" } },
              { { "uni06BA" }, { "uni06BA.fina" } },
              { { "uni06BB" }, { "uni06BB.fina" } },
              { { "uni06BC" }, { "uni06BC.fina" } },
              { { "uni06BD" }, { "uni06BD.fina" } },
              { { "uni06BE" }, { "uni06BE.fina" } },
              { { "uni06BF" }, { "uni06BF.fina" } },
              { { "uni06C0" }, { "uni06C0.fina" } },
              { { "uni06C1" }, { "uni06C1.fina" } },
              { { "uni06C2" }, { "uni06C2.fina" } },
              { { "uni06C3" }, { "uni06C3.fina" } },
              { { "uni06C4" }, { "uni06C4.fina" } },
              { { "uni06C5" }, { "uni06C5.fina" } },
              { { "uni06C6" }, { "uni06C6.fina" } },
              { { "uni06C7" }, { "uni06C7.fina" } },
              { { "uni06C8" }, { "uni06C8.fina" } },
              { { "uni06C9" }, { "uni06C9.fina" } },
              { { "uni06CA" }, { "uni06CA.fina" } },
              { { "uni06CB" }, { "uni06CB.fina" } },
              { { "uni06CC" }, { "uni06CC.fina" } },
              { { "uni06CD" }, { "uni06CD.fina" } },
              { { "uni06CE" }, { "uni06CE.fina" } },
              { { "uni06CF" }, { "uni06CF.fina" } },
              { { "uni06D0" }, { "uni06D0.fina" } },
              { { "uni06D1" }, { "uni06D1.fina" } },
              { { "uni06D2" }, { "uni06D2.fina" } },
              { { "uni06D3" }, { "uni06D3.fina" } },
              { { "uni06D5" }, { "uni06D5.fina" } },
              { { "uni06EE" }, { "uni06EE.fina" } },
              { { "uni06EF" }, { "uni06EF.fina" } },
              { { "uni06FA" }, { "uni06FA.fina" } },
              { { "uni06FB" }, { "uni06FB.fina" } },
              { { "uni06FC" }, { "uni06FC.fina" } },
              { { "uni06FF" }, { "uni06FF.fina" } },
              { { "uni0750" }, { "uni0750.fina" } },
              { { "uni0751" }, { "uni0751.fina" } },
              { { "uni0752" }, { "uni0752.fina" } },
              { { "uni0753" }, { "uni0753.fina" } },
              { { "uni0754" }, { "uni0754.fina" } },
              { { "uni0755" }, { "uni0755.fina" } },
              { { "uni0756" }, { "uni0756.fina" } },
              { { "uni0757" }, { "uni0757.fina" } },
              { { "uni0758" }, { "uni0758.fina" } },
              { { "uni0759" }, { "uni0759.fina" } },
              { { "uni075A" }, { "uni075A.fina" } },
              { { "uni075B" }, { "uni075B.fina" } },
              { { "uni075C" }, { "uni075C.fina" } },
              { { "uni075D" }, { "uni075D.fina" } },
              { { "uni075E" }, { "uni075E.fina" } },
              { { "uni075F" }, { "uni075F.fina" } },
              { { "uni0760" }, { "uni0760.fina" } },
              { { "uni0761" }, { "uni0761.fina" } },
              { { "uni0762" }, { "uni0762.fina" } },
              { { "uni0763" }, { "uni0763.fina" } },
              { { "uni0764" }, { "uni0764.fina" } },
              { { "uni0765" }, { "uni0765.fina" } },
              { { "uni0766" }, { "uni0766.fina" } },
              { { "uni0767" }, { "uni0767.fina" } },
              { { "uni0768" }, { "uni0768.fina" } },
              { { "uni0769" }, { "uni0769.fina" } },
              { { "uni076A" }, { "uni076A.fina" } },
              { { "uni076B" }, { "uni076B.fina" } },
              { { "uni076C" }, { "uni076C.fina" } },
              { { "uni076D" }, { "uni076D.fina" } },
          }
        },
        { "f3", "lu3", "arab", "dflt", "medi",
          new String[][][] {
              { { "absJeemRetro1" }, { "absJeemRetro1Med" } },
              { { "absJeemRetro2" }, { "absJeemRetro2Med" } },
              { { "absJeemRetro3" }, { "absJeemRetro3Med" } },
              { { "absLamRetro" }, { "absLamRetroMed" } },
              { { "absSheenRetro1" }, { "absSheenRetro1Med" } },
              { { "absSheenRetro2" }, { "absSheenRetro2Med" } },
              { { "absTchehRetro1" }, { "absTchehRetro1Med" } },
              { { "absTchehRetro2" }, { "absTchehRetro2Med" } },
              { { "uni0626" }, { "uni0626.medi" } },
              { { "uni0628" }, { "uni0628.medi" } },
              { { "uni062A" }, { "uni062A.medi" } },
              { { "uni062B" }, { "uni062B.medi" } },
              { { "uni062C" }, { "uni062C.medi" } },
              { { "uni062D" }, { "uni062D.medi" } },
              { { "uni062E" }, { "uni062E.medi" } },
              { { "uni0633" }, { "uni0633.medi" } },
              { { "uni0634" }, { "uni0634.medi" } },
              { { "uni0635" }, { "uni0635.medi" } },
              { { "uni0636" }, { "uni0636.medi" } },
              { { "uni0637" }, { "uni0637.medi" } },
              { { "uni0638" }, { "uni0638.medi" } },
              { { "uni0639" }, { "uni0639.medi" } },
              { { "uni063A" }, { "uni063A.medi" } },
              { { "uni0641" }, { "uni0641.medi" } },
              { { "uni0642" }, { "uni0642.medi" } },
              { { "uni0643" }, { "uni0643.medi" } },
              { { "uni0644" }, { "uni0644.medi" } },
              { { "uni0645" }, { "uni0645.medi" } },
              { { "uni0646" }, { "uni0646.medi" } },
              { { "uni0647" }, { "uni0647.medi" } },
              { { "uni0649" }, { "uni0649.medi" } },
              { { "uni064A" }, { "uni064A.medi" } },
              { { "uni064A.noDots" }, { "uni064A.medi.noDots" } },
              { { "uni066E" }, { "uni066E.medi" } },
              { { "uni066F" }, { "uni066F.medi" } },
              { { "uni0678" }, { "uni0678.medi" } },
              { { "uni0679" }, { "uni0679.medi" } },
              { { "uni067A" }, { "uni067A.medi" } },
              { { "uni067B" }, { "uni067B.medi" } },
              { { "uni067C" }, { "uni067C.medi" } },
              { { "uni067D" }, { "uni067D.medi" } },
              { { "uni067E" }, { "uni067E.medi" } },
              { { "uni067F" }, { "uni067F.medi" } },
              { { "uni0680" }, { "uni0680.medi" } },
              { { "uni0681" }, { "uni0681.medi" } },
              { { "uni0682" }, { "uni0682.medi" } },
              { { "uni0683" }, { "uni0683.medi" } },
              { { "uni0684" }, { "uni0684.medi" } },
              { { "uni0685" }, { "uni0685.medi" } },
              { { "uni0686" }, { "uni0686.medi" } },
              { { "uni0687" }, { "uni0687.medi" } },
              { { "uni069A" }, { "uni069A.medi" } },
              { { "uni069B" }, { "uni069B.medi" } },
              { { "uni069C" }, { "uni069C.medi" } },
              { { "uni069D" }, { "uni069D.medi" } },
              { { "uni069E" }, { "uni069E.medi" } },
              { { "uni069F" }, { "uni069F.medi" } },
              { { "uni06A0" }, { "uni06A0.medi" } },
              { { "uni06A1" }, { "uni06A1.medi" } },
              { { "uni06A2" }, { "uni06A2.medi" } },
              { { "uni06A3" }, { "uni06A3.medi" } },
              { { "uni06A4" }, { "uni06A4.medi" } },
              { { "uni06A5" }, { "uni06A5.medi" } },
              { { "uni06A6" }, { "uni06A6.medi" } },
              { { "uni06A7" }, { "uni06A7.medi" } },
              { { "uni06A8" }, { "uni06A8.medi" } },
              { { "uni06A9" }, { "uni06A9.medi" } },
              { { "uni06AA" }, { "uni06AA.medi" } },
              { { "uni06AB" }, { "uni06AB.medi" } },
              { { "uni06AC" }, { "uni06AC.medi" } },
              { { "uni06AD" }, { "uni06AD.medi" } },
              { { "uni06AE" }, { "uni06AE.medi" } },
              { { "uni06AF" }, { "uni06AF.medi" } },
              { { "uni06B0" }, { "uni06B0.medi" } },
              { { "uni06B1" }, { "uni06B1.medi" } },
              { { "uni06B2" }, { "uni06B2.medi" } },
              { { "uni06B3" }, { "uni06B3.medi" } },
              { { "uni06B4" }, { "uni06B4.medi" } },
              { { "uni06B5" }, { "uni06B5.medi" } },
              { { "uni06B6" }, { "uni06B6.medi" } },
              { { "uni06B7" }, { "uni06B7.medi" } },
              { { "uni06B8" }, { "uni06B8.medi" } },
              { { "uni06B9" }, { "uni06B9.medi" } },
              { { "uni06BA" }, { "uni06BA.medi" } },
              { { "uni06BB" }, { "uni06BB.medi" } },
              { { "uni06BC" }, { "uni06BC.medi" } },
              { { "uni06BD" }, { "uni06BD.medi" } },
              { { "uni06BE" }, { "uni06BE.medi" } },
              { { "uni06BF" }, { "uni06BF.medi" } },
              { { "uni06C1" }, { "uni06C1.medi" } },
              { { "uni06CC" }, { "uni06CC.medi" } },
              { { "uni06CE" }, { "uni06CE.medi" } },
              { { "uni06D0" }, { "uni06D0.medi" } },
              { { "uni06D1" }, { "uni06D1.medi" } },
              { { "uni06FA" }, { "uni06FA.medi" } },
              { { "uni06FB" }, { "uni06FB.medi" } },
              { { "uni06FC" }, { "uni06FC.medi" } },
              { { "uni06FF" }, { "uni06FF.medi" } },
              { { "uni0750" }, { "uni0750.medi" } },
              { { "uni0751" }, { "uni0751.medi" } },
              { { "uni0752" }, { "uni0752.medi" } },
              { { "uni0753" }, { "uni0753.medi" } },
              { { "uni0754" }, { "uni0754.medi" } },
              { { "uni0755" }, { "uni0755.medi" } },
              { { "uni0756" }, { "uni0756.medi" } },
              { { "uni0757" }, { "uni0757.medi" } },
              { { "uni0758" }, { "uni0758.medi" } },
              { { "uni075C" }, { "uni075C.medi" } },
              { { "uni075D" }, { "uni075D.medi" } },
              { { "uni075E" }, { "uni075E.medi" } },
              { { "uni075F" }, { "uni075F.medi" } },
              { { "uni0760" }, { "uni0760.medi" } },
              { { "uni0761" }, { "uni0761.medi" } },
              { { "uni0762" }, { "uni0762.medi" } },
              { { "uni0763" }, { "uni0763.medi" } },
              { { "uni0764" }, { "uni0764.medi" } },
              { { "uni0765" }, { "uni0765.medi" } },
              { { "uni0766" }, { "uni0766.medi" } },
              { { "uni0767" }, { "uni0767.medi" } },
              { { "uni0768" }, { "uni0768.medi" } },
              { { "uni0769" }, { "uni0769.medi" } },
              { { "uni076A" }, { "uni076A.medi" } },
              { { "uni076D" }, { "uni076D.medi" } },
          }
        },
        { "f3", "lu11", "arab", "SND ", "calt",
          new String[][][] {
              { { "uni0645" }, { "uni0645.sindhi" } },
              { { "uni0645.fina" }, { "uni0645.fina.sindhi" } },
              { { "uni0647" }, { "uni0647.knotted" } },
              { { "uni0647.fina" }, { "uni0647.fina.knottedHigh" } },
              { { "uni0647.medi" }, { "uni0647.medi.knottedHigh" } },
              { { "uni06F6" }, { "uni06F6.urdu" } },
              { { "uni06F7" }, { "uni06F7.urdu" } },
          }
        },
        { "f3", "lu12", "arab", "KUR ", "calt",
          new String[][][] {
              { { "uni0647" }, { "uni0647.knotted" } },
              { { "uni0647.fina" }, { "uni0647.fina.knottedHigh" } },
          }
        },
        { "f3", "lu13", "arab", "URD ", "calt",
          new String[][][] {
              { { "uni0647.fina" }, { "uni0647.fina.hooked" } },
              { { "uni0647.init" }, { "uni0647.init.hooked" } },
              { { "uni0647.medi" }, { "uni0647.medi.hooked" } },
              { { "uni06F4" }, { "uni06F4.urdu" } },
              { { "uni06F6" }, { "uni06F6.urdu" } },
              { { "uni06F7" }, { "uni06F7.urdu" } },
          }
        },
        { "f3", "lu15", "arab", "*", "*",
          new String[][][] {
              { { "absLamRetroIni" }, { "absLamRetroIni.preAlef" } },
              { { "absLamRetroMed" }, { "absLamRetroMed.preAlef" } },
              { { "uni0644.init" }, { "uni0644.init.preAlef" } },
              { { "uni0644.medi" }, { "uni0644.medi.preAlef" } },
              { { "uni06B5.init" }, { "uni06B5.init.preAlef" } },
              { { "uni06B5.medi" }, { "uni06B5.medi.preAlef" } },
              { { "uni06B6.init" }, { "uni06B6.init.preAlef" } },
              { { "uni06B6.medi" }, { "uni06B6.medi.preAlef" } },
              { { "uni06B7.init" }, { "uni06B7.init.preAlef" } },
              { { "uni06B7.medi" }, { "uni06B7.medi.preAlef" } },
              { { "uni06B8.init" }, { "uni06B8.init.preAlef" } },
              { { "uni06B8.medi" }, { "uni06B8.medi.preAlef" } },
              { { "uni076A.init" }, { "uni076A.init.preAlef" } },
              { { "uni076A.medi" }, { "uni076A.medi.preAlef" } },
          }
        },
        { "f3", "lu16", "arab", "*", "*",
          new String[][][] {
              { { "uni0622.fina" }, { "uni0622.fina.postLamIni" } },
              { { "uni0623.fina" }, { "uni0623.fina.postLamIni" } },
              { { "uni0625.fina" }, { "uni0625.fina.postLamIni" } },
              { { "uni0627.fina" }, { "uni0627.fina.postLamIni" } },
              { { "uni0671.fina" }, { "uni0671.fina.postLamIni" } },
              { { "uni0672.fina" }, { "uni0672.fina.postLamIni" } },
              { { "uni0673.fina" }, { "uni0673.fina.postLamIni" } },
              { { "uni0675.fina" }, { "uni0675.fina.postLamIni" } },
          }
        },
        { "f3", "lu17", "arab", "*", "*",
          new String[][][] {
              { { "uni0622.fina" }, { "uni0622.fina.postLamMed" } },
              { { "uni0623.fina" }, { "uni0623.fina.postLamMed" } },
              { { "uni0625.fina" }, { "uni0625.fina.postLamMed" } },
              { { "uni0627.fina" }, { "uni0627.fina.postLamMed" } },
              { { "uni0671.fina" }, { "uni0671.fina.postLamMed" } },
              { { "uni0672.fina" }, { "uni0672.fina.postLamMed" } },
              { { "uni0673.fina" }, { "uni0673.fina.postLamMed" } },
              { { "uni0675.fina" }, { "uni0675.fina.postLamMed" } },
          }
        },
        { "f3", "lu18", "arab", "*", "*",
          new String[][][] {
              { { "uni0601" }, { "uni0601.4" } },
          }
        },
        { "f3", "lu19", "arab", "*", "*",
          new String[][][] {
              { { "uni0600" }, { "uni0600.3" } },
              { { "uni0601" }, { "uni0601.3" } },
              { { "uni0603" }, { "uni0603.3" } },
              { { "uni06DD" }, { "uni06DD.3" } },
              { { "uni06DD.alt" }, { "uni06DD.alt.3" } },
              { { "uni06DD.altB" }, { "uni06DD.altB.3" } },
          }
        },
        { "f3", "lu20", "arab", "*", "*",
          new String[][][] {
              { { "uni0600" }, { "uni0600.2" } },
              { { "uni0601" }, { "uni0601.2" } },
              { { "uni0602" }, { "uni0602.2" } },
              { { "uni0603" }, { "uni0603.2" } },
              { { "uni06DD" }, { "uni06DD.2" } },
              { { "uni06DD.alt" }, { "uni06DD.alt.2" } },
              { { "uni06DD.altB" }, { "uni06DD.altB.2" } },
          }
        },
        { "f3", "lu21", "arab", "*", "*",
          new String[][][] {
              { { "eight" }, { "eightMedium" } },
              { { "five" }, { "fiveMedium" } },
              { { "four" }, { "fourMedium" } },
              { { "nine" }, { "nineMedium" } },
              { { "one" }, { "oneMedium" } },
              { { "seven" }, { "sevenMedium" } },
              { { "six" }, { "sixMedium" } },
              { { "three" }, { "threeMedium" } },
              { { "two" }, { "twoMedium" } },
              { { "uni0660" }, { "uni0660.Medium" } },
              { { "uni0661" }, { "uni0661.Medium" } },
              { { "uni0662" }, { "uni0662.Medium" } },
              { { "uni0663" }, { "uni0663.Medium" } },
              { { "uni0664" }, { "uni0664.Medium" } },
              { { "uni0665" }, { "uni0665.Medium" } },
              { { "uni0666" }, { "uni0666.Medium" } },
              { { "uni0667" }, { "uni0667.Medium" } },
              { { "uni0668" }, { "uni0668.Medium" } },
              { { "uni0669" }, { "uni0669.Medium" } },
              { { "uni06F0" }, { "uni06F0.Medium" } },
              { { "uni06F1" }, { "uni06F1.Medium" } },
              { { "uni06F2" }, { "uni06F2.Medium" } },
              { { "uni06F3" }, { "uni06F3.Medium" } },
              { { "uni06F4" }, { "uni06F4.Medium" } },
              { { "uni06F4.urdu" }, { "uni06F4.Medium.urdu" } },
              { { "uni06F5" }, { "uni06F5.Medium" } },
              { { "uni06F6" }, { "uni06F6.Medium" } },
              { { "uni06F6.urdu" }, { "uni06F6.Medium.urdu" } },
              { { "uni06F7" }, { "uni06F7.Medium" } },
              { { "uni06F7.urdu" }, { "uni06F7.Medium.urdu" } },
              { { "uni06F8" }, { "uni06F8.Medium" } },
              { { "uni06F9" }, { "uni06F9.Medium" } },
              { { "zero" }, { "zeroMedium" } },
          }
        },
        { "f3", "lu22", "arab", "*", "*",
          new String[][][] {
              { { "eight" }, { "eightSmall" } },
              { { "five" }, { "fiveSmall" } },
              { { "four" }, { "fourSmall" } },
              { { "nine" }, { "nineSmall" } },
              { { "one" }, { "oneSmall" } },
              { { "seven" }, { "sevenSmall" } },
              { { "six" }, { "sixSmall" } },
              { { "three" }, { "threeSmall" } },
              { { "two" }, { "twoSmall" } },
              { { "uni0660" }, { "uni0660.Small" } },
              { { "uni0661" }, { "uni0661.Small" } },
              { { "uni0662" }, { "uni0662.Small" } },
              { { "uni0663" }, { "uni0663.Small" } },
              { { "uni0664" }, { "uni0664.Small" } },
              { { "uni0665" }, { "uni0665.Small" } },
              { { "uni0666" }, { "uni0666.Small" } },
              { { "uni0667" }, { "uni0667.Small" } },
              { { "uni0668" }, { "uni0668.Small" } },
              { { "uni0669" }, { "uni0669.Small" } },
              { { "uni06F0" }, { "uni06F0.Small" } },
              { { "uni06F1" }, { "uni06F1.Small" } },
              { { "uni06F2" }, { "uni06F2.Small" } },
              { { "uni06F3" }, { "uni06F3.Small" } },
              { { "uni06F4" }, { "uni06F4.Small" } },
              { { "uni06F4.urdu" }, { "uni06F4.Small.urdu" } },
              { { "uni06F5" }, { "uni06F5.Small" } },
              { { "uni06F6" }, { "uni06F6.Small" } },
              { { "uni06F6.urdu" }, { "uni06F6.Small.urdu" } },
              { { "uni06F7" }, { "uni06F7.Small" } },
              { { "uni06F7.urdu" }, { "uni06F7.Small.urdu" } },
              { { "uni06F8" }, { "uni06F8.Small" } },
              { { "uni06F9" }, { "uni06F9.Small" } },
              { { "zero" }, { "zeroSmall" } },
          }
        },
        { "f3", "lu23", "arab", "*", "*",
          new String[][][] {
              { { "uni0670" }, { "uni0670.large" } },
          }
        },
    };

    private static Object[][] ltMultiple = {
        { GlyphSubstitutionTable.GSUB_LOOKUP_TYPE_MULTIPLE },
        // arab-001.ttx
        { "f0", "lu9", "arab", "*", "*",
          new String[][][] {
              { { "alefwithhamzabelow" }, { "arabicalef", "uni0655" } },
          }
        },
        // arab-002.ttx
        { "f1", "lu14", "arab", "*", "*",
          new String[][][] {
              { { "pehinitial" }, { "pehinitial", "tatweel" } },
              { { "yehwithhamzaaboveinitial" }, { "yehwithhamzaaboveinitial", "tatweel" } },
              { { "behinitial" }, { "behinitial", "tatweel" } },
              { { "tehinitial" }, { "tehinitial", "tatweel" } },
              { { "thehinitial" }, { "thehinitial", "tatweel" } },
              { { "fehinitial" }, { "fehinitial", "tatweel" } },
              { { "qafinitial" }, { "qafinitial", "tatweel" } },
              { { "nooninitial" }, { "nooninitial", "tatweel" } },
              { { "yehinitial" }, { "yehinitial", "tatweel" } },
              { { "uni0649.init" }, { "uni0649.init", "tatweel" } },
          }
        },
        { "f1", "lu15", "arab", "*", "*",
          new String[][][] {
              { { "pehmedial" }, { "pehmedial", "tatweel" } },
              { { "yehwithhamzaabovemedial" }, { "yehwithhamzaabovemedial", "tatweel" } },
              { { "behmedial" }, { "behmedial", "tatweel" } },
              { { "tehmedial" }, { "tehmedial", "tatweel" } },
              { { "thehmedial" }, { "thehmedial", "tatweel" } },
              { { "noonmedial" }, { "noonmedial", "tatweel" } },
              { { "yehmedial" }, { "yehmedial", "tatweel" } },
              { { "uni0649.medi" }, { "uni0649.medi", "tatweel" } },
          }
        },
        // arab-003.ttx
        { "f2", "lu0", "arab", "dflt", "ccmp",
          new String[][][] {
              { { "uni0622" }, { "uni0627", "uni0653" } },
              { { "uni0623" }, { "uni0627", "uni0654" } },
              { { "uni0625" }, { "uni0627", "uni0655" } },
              { { "uni0626" }, { "uni064A", "uni0654" } },
          }
        },
        // arab-004.ttx
        { "f3", "lu0", "arab", "dflt", "ccmp",
          new String[][][] {
              { { "uni0622" }, { "uni0627", "uni0653" } },
              { { "uni0623" }, { "uni0627", "uni0654" } },
              { { "uni0625" }, { "uni0627", "uni0655" } },
          }
        },
    };

    private static Object[][] ltAlternate = {
        { GlyphSubstitutionTable.GSUB_LOOKUP_TYPE_ALTERNATE },
        // arab-001.ttx - none used
        // arab-002.ttx - none used
        // arab-003.ttx - none used
        // arab-004.ttx - add tests
        { "f3", "lu14", "arab", "dflt", "salt" },
    };

    private static Object[][] ltLigature = {
        { GlyphSubstitutionTable.GSUB_LOOKUP_TYPE_LIGATURE },
        // arab-001.ttx
        { "f0", "lu0", "arab", "dflt", "ccmp",
          new String[][][] {
              { { "damma", "shadda" }, { "shaddawithdammalow" } },
              { { "damma", "highhamza" }, { "dammaonhamza" } },
              { { "dammatan", "shadda" }, { "shaddawithdammatanlow" } },
              { { "dammatan", "highhamza" }, { "dammatanonhamza" } },
              { { "fatha", "shadda" }, { "shaddawithfathalow" } },
              { { "fatha", "highhamza" }, { "fathaonhamza" } },
              { { "fathatan", "shadda" }, { "shaddawithfathatanlow" } },
              { { "fathatan", "highhamza" }, { "fathatanonhamza" } },
              { { "highhamza", "fatha" }, { "fathaonhamza" } },
              { { "highhamza", "fathatan" }, { "fathatanonhamza" } },
              { { "highhamza", "sukun" }, { "sukunonhamza" } },
              { { "highhamza", "damma" }, { "dammaonhamza" } },
              { { "highhamza", "dammatan" }, { "dammatanonhamza" } },
              { { "kasra", "shadda" }, { "shaddawithkasralow" } },
              { { "kasra", "uni0655" }, { "uni06550650" } },
              { { "kasratan", "shadda" }, { "shaddawithkasratanlow" } },
              { { "kasratan", "uni0655" }, { "uni0655064D" } },
              { { "shadda", "dammatan" }, { "shaddawithdammatanlow" } },
              { { "shadda", "fatha" }, { "shaddawithfathalow" } },
              { { "shadda", "damma" }, { "shaddawithdammalow" } },
              { { "shadda", "fathatan" }, { "shaddawithfathatanlow" } },
              { { "shadda", "kasratan" }, { "shaddawithkasratanlow" } },
              { { "shadda", "kasra" }, { "shaddawithkasralow" } },
              { { "sukun", "highhamza" }, { "sukunonhamza" } },
              { { "uni0655", "kasratan" }, { "uni0655064D" } },
              { { "uni0655", "kasra" }, { "uni06550650" } },
          }
        },
        { "f0", "lu7", "arab", "dflt", "rlig",
          new String[][][] {
              { { "lamisolated", "alefwithmaddaabovefinal" }, { "lamwithalefmaddaaboveisolatedd" } },
              { { "lamisolated", "alefwithhamzaabovefinal" }, { "lamwithalefhamzaaboveisolatedd" } },
              { { "lamisolated", "alefwithhamzabelowfinal" }, { "lamwithalefhamzabelowisolated" } },
              { { "lamisolated", "aleffinal" }, { "lamwithalefisolated" } },
              { { "lammedial", "alefwithmaddaabovefinal" }, { "lamwithalefmaddaabovefinal" } },
              { { "lammedial", "alefwithhamzaabovefinal" }, { "lamwithalefhamzaabovefinal" } },
              { { "lammedial", "alefwithhamzabelowfinal" }, { "lamwithalefhamzabelowfinal" } },
              { { "lammedial", "aleffinal" }, { "lamwithaleffinal" } },
          }
        },
        { "f0", "lu8", "arab", "dflt", "liga",
          new String[][][] {
              { { "lamisolated", "lammedial", "hehfinal" }, { "allahisolated" } },
              { { "reh", "yehmedial", "aleffinal", "lam" }, { "rayaleflam" } },
          }
        },
        // arab-002.ttx
        { "f1", "lu0", "arab", "dflt", "ccmp",
          new String[][][] {
              { { "damma", "shadda" }, { "shaddawithdammaisolatedlow" } },
              { { "damma", "highhamza" }, { "dammaonhamza" } },
              { { "dammatan", "shadda" }, { "shaddawithdammatanisolatedlow" } },
              { { "dammatan", "highhamza" }, { "dammatanonhamza" } },
              { { "fatha", "shadda" }, { "shaddawithfathaisolatedlow" } },
              { { "fatha", "highhamza" }, { "fathaonhamza" } },
              { { "fathatan", "shadda" }, { "shaddawithfathatanisolatedlow" } },
              { { "fathatan", "highhamza" }, { "fathatanonhamza" } },
              { { "highhamza", "fatha" }, { "fathaonhamza" } },
              { { "highhamza", "fathatan" }, { "fathatanonhamza" } },
              { { "highhamza", "sukun" }, { "sukunonhamza" } },
              { { "highhamza", "damma" }, { "dammaonhamza" } },
              { { "highhamza", "dammatan" }, { "dammatanonhamza" } },
              { { "kasra", "shadda" }, { "shaddawithkasraisolatedlow" } },
              { { "kasra", "uni0655" }, { "uni06550650" } },
              { { "kasratan", "shadda" }, { "shaddawithkasratanisolatedlow" } },
              { { "kasratan", "uni0655" }, { "uni0655064D" } },
              { { "shadda", "dammatan" }, { "shaddawithdammatanisolatedlow" } },
              { { "shadda", "fatha" }, { "shaddawithfathaisolatedlow" } },
              { { "shadda", "damma" }, { "shaddawithdammaisolatedlow" } },
              { { "shadda", "fathatan" }, { "shaddawithfathatanisolatedlow" } },
              { { "shadda", "kasratan" }, { "shaddawithkasratanisolatedlow" } },
              { { "shadda", "kasra" }, { "shaddawithkasraisolatedlow" } },
              { { "sukun", "highhamza" }, { "sukunonhamza" } },
              { { "uni0655", "kasratan" }, { "uni0655064D" } },
              { { "uni0655", "kasra" }, { "uni06550650" } },
          }
        },
        { "f1", "lu6", "arab", "dflt", "liga",
          new String[][][] {
              { { "behinitial", "hehmedial" }, { "behwithhehinitial" } },
              { { "behinitial", "meemfinal" }, { "behwithmeemisolated" } },
              { { "behinitial", "meemmedial" }, { "behwithmeeminitial" } },
              { { "behinitial", "alefmaksurafinal" }, { "behwithalefmaksuraisolated" } },
              { { "behinitial", "yehfinal" }, { "behwithyehisolated" } },
              { { "behinitial", "jeemmedial" }, { "behwithjeeminitial" } },
              { { "behinitial", "hahmedial" }, { "behwithhahinitial" } },
              { { "behinitial", "khahmedial" }, { "behwithkhahinitial" } },
              { { "behmedial", "alefmaksurafinal" }, { "behwithalefmaksurafinal" } },
              { { "behmedial", "yehfinal" }, { "behwithyehfinal" } },
              { { "behmedial", "rehfinal" }, { "behwithrehfinal" } },
              { { "behmedial", "noonfinal" }, { "behwithnoonfinal" } },
              { { "fehinitial", "alefmaksurafinal" }, { "fehwithalefmaksuraisolated" } },
              { { "fehinitial", "yehfinal" }, { "fehwithyehisolated" } },
              { { "hahinitial", "meemmedial" }, { "hahwithmeeminitial" } },
              { { "hehinitial", "meemmedial" }, { "hehwithmeeminitial" } },
              { { "jeeminitial", "meemmedial" }, { "jeemwithmeeminitial" } },
              { { "khahinitial", "meemmedial" }, { "khahwithmeeminitial" } },
              { { "laminitial", "jeemmedial" }, { "lamwithjeeminitial" } },
              { { "laminitial", "hahmedial" }, { "lamwithhahinitial" } },
              { { "laminitial", "khahmedial" }, { "lamwithkhahinitial" } },
              { { "laminitial", "hehmedial" }, { "lamwithhehinitial" } },
              { { "laminitial", "meemfinal" }, { "lamwithmeemisolated" } },
              { { "laminitial", "alefmaksurafinal" }, { "lamwithalefmaksuraisolated" } },
              { { "laminitial", "yehfinal" }, { "lamwithyehisolated" } },
              { { "meeminitial", "jeemmedial" }, { "meemwithjeeminitial" } },
              { { "meeminitial", "hahmedial" }, { "meemwithhahinitial" } },
              { { "meeminitial", "khahmedial" }, { "meemwithkhahinitial" } },
              { { "meeminitial", "meemmedial" }, { "meemwithmeeminitial" } },
              { { "nooninitial", "hehmedial" }, { "noonwithhehinitial" } },
              { { "nooninitial", "meemfinal" }, { "noonwithmeemisolated" } },
              { { "nooninitial", "meemmedial" }, { "noonwithmeeminitial" } },
              { { "nooninitial", "alefmaksurafinal" }, { "noonwithalefmaksuraisolated" } },
              { { "nooninitial", "yehfinal" }, { "noonwithyehisolated" } },
              { { "nooninitial", "jeemmedial" }, { "noonwithjeeminitial" } },
              { { "nooninitial", "hahmedial" }, { "noonwithhahinitial" } },
              { { "nooninitial", "khahmedial" }, { "noonwithkhahinitial" } },
              { { "noonmedial", "alefmaksurafinal" }, { "noonwithalefmaksurafinal" } },
              { { "noonmedial", "yehfinal" }, { "noonwithyehfinal" } },
              { { "pehinitial", "hehmedial" }, { "pehwithhehinitial" } },
              { { "seeninitial", "meemmedial" }, { "seenwithmeeminitial" } },
              { { "sheeninitial", "meemmedial" }, { "sheenwithmeeminitial" } },
              { { "tchehinitial", "meemmedial" }, { "uniE817" } },
              { { "tehinitial", "hehmedial" }, { "tehwithhehinitial" } },
              { { "tehinitial", "meemfinal" }, { "tehwithmeemisolated" } },
              { { "tehinitial", "meemmedial" }, { "tehwithmeeminitial" } },
              { { "tehinitial", "yehfinal" }, { "tehwithyehisolated" } },
              { { "tehinitial", "jeemmedial" }, { "tehwithjeeminitial" } },
              { { "tehinitial", "hahmedial" }, { "tehwithhahinitial" } },
              { { "tehinitial", "khahmedial" }, { "tehwithkhahinitial" } },
              { { "tehmedial", "alefmaksurafinal" }, { "tehwithalefmaksurafinal" } },
              { { "tehmedial", "yehfinal" }, { "tehwithyehfinal" } },
              { { "tehmedial", "rehfinal" }, { "noonwithzainfinal" } },
              { { "tehmedial", "noonfinal" }, { "tehwithnoonfinal" } },
              { { "thehinitial", "meemfinal" }, { "thehwithmeemisolated" } },
              { { "thehinitial", "meemmedial" }, { "thehwithmeeminitial" } },
              { { "yehinitial", "meemfinal" }, { "yehwithmeemisolated" } },
              { { "yehinitial", "meemmedial" }, { "yehwithmeeminitial" } },
              { { "yehinitial", "alefmaksurafinal" }, { "yehwithalefmaksuraisolated" } },
              { { "yehinitial", "jeemmedial" }, { "yehwithjeeminitial" } },
              { { "yehinitial", "hahmedial" }, { "yehwithhahinitial" } },
              { { "yehinitial", "khahmedial" }, { "yehwithkhahinitial" } },
              { { "yehmedial", "alefmaksurafinal" }, { "yehwithalefmaksurafinal" } },
              { { "yehmedial", "rehfinal" }, { "yehwithrehfinal" } },
              { { "yehmedial", "noonfinal" }, { "yehwithnoonfinal" } },
          }
        },
        { "f1", "lu7", "arab", "dflt", "liga",
          new String[][][] {
              { { "laminitial", "meemmedial", "jeemmedial" }, { "lamwithmeemwithjeeminitial" } },
              { { "laminitial", "meemmedial", "hahmedial" }, { "lamwithmeemwithhahinitial" } },
              { { "laminitial", "meemmedial" }, { "lamwithmeeminitial" } },
          }
        },
        { "f1", "lu8", "arab", "dflt", "liga",
          new String[][][] {
              { { "laminitial", "jeemfinal" }, { "lamwithjeemisolated" } },
              { { "laminitial", "hahfinal" }, { "lamwithhahisolated" } },
              { { "laminitial", "khahfinal" }, { "lamwithkhahisolated" } },
          }
        },
        { "f1", "lu9", "arab", "dflt", "rlig",
          new String[][][] {
              { { "laminitial", "alefwithmaddaabovefinal" }, { "lamwithalefmaddaaboveisolatedd" } },
              { { "laminitial", "alefwithhamzaabovefinal" }, { "lamwithalefhamzaaboveisolatedd" } },
              { { "laminitial", "alefwithhamzabelowfinal" }, { "lamwithalefhamzabelowisolated" } },
              { { "laminitial", "aleffinal" }, { "lamwithalefisolated" } },
              { { "lammedial", "alefwithmaddaabovefinal" }, { "lamwithalefmaddaabovefinal" } },
              { { "lammedial", "alefwithhamzaabovefinal" }, { "lamwithalefhamzaabovefinal" } },
              { { "lammedial", "alefwithhamzabelowfinal" }, { "lamwithalefhamzabelowfinal" } },
              { { "lammedial", "aleffinal" }, { "lamwithaleffinal" } },
          }
        },
        { "f1", "lu10", "arab", "dflt", "liga",
          new String[][][] {
              { { "laminitial", "lammedial", "hehfinal" }, { "allahisolated" } },
              { { "reh", "yehinitial", "aleffinal", "lam" }, { "rayaleflam" } },
          }
        },
        // arab-003.ttx
        { "f2", "lu5", "arab", "dflt", "rlig",
          new String[][][] {
              { { "uni064B", "uni0651" }, { "uni0651064B" } },
              { { "uni064C", "uni0651" }, { "uni0651064C" } },
              { { "uni064E", "uni0651" }, { "uni0651064E" } },
              { { "uni064F", "uni0651" }, { "uni0651064F" } },
              { { "uni0651", "uni064B" }, { "uni0651064B" } },
              { { "uni0651", "uni064C" }, { "uni0651064C" } },
              { { "uni0651", "uni064E" }, { "uni0651064E" } },
              { { "uni0651", "uni064F" }, { "uni0651064F" } },
              { { "uni0651", "uni0670" }, { "absShaddaAlef" } },
              { { "uni0670", "uni0651" }, { "absShaddaAlef" } },
          }
        },
        { "f2", "lu6", "arab", "dflt", "rlig",
          new String[][][] {
              { { "uni064D", "uni0651" }, { "uni0651064D" } },
              { { "uni0650", "uni0651" }, { "uni06510650" } },
              { { "uni0651", "uni0650" }, { "uni06510650" } },
              { { "uni0651", "uni064D" }, { "uni0651064D" } },
          }
        },
        { "f2", "lu7", "arab", "dflt", "rlig",
          new String[][][] {
              { { "uni0647", "uni0654" }, { "uni06C0" } },
              { { "uni0647.fina", "uni0654" }, { "uni06C0.fina" } },
              { { "uni0647.init", "uni0654" }, { "uni06C0.init" } },
              { { "uni0647.medi", "uni0654" }, { "uni06C0.medi" } },
              { { "uni0648", "uni0654" }, { "uni0624" } },
              { { "uni0648.fina", "uni0654" }, { "uni0624.fina" } },
              { { "uni064A", "uni0654" }, { "uni0626" } },
              { { "uni064A.fina", "uni0654" }, { "uni0626.fina" } },
              { { "uni064A.init", "uni0654" }, { "uni0626.init" } },
              { { "uni064A.medi", "uni0654" }, { "uni0626.medi" } },
              { { "uni06C1", "uni0654" }, { "uni06C2" } },
              { { "uni06C1.fina", "uni0654" }, { "uni06C2.fina" } },
              { { "uni06C1.init", "uni0654" }, { "uni06C2.init" } },
              { { "uni06C1.medi", "uni0654" }, { "uni06C2.medi" } },
          }
        },
        // arab-004.ttx
        { "f3", "lu5", "arab", "dflt", "rlig",
          new String[][][] {
              { { "uni064B", "uni0651" }, { "uni0651064B" } },
              { { "uni064C", "uni0651" }, { "uni0651064C" } },
              { { "uni064E", "uni0651" }, { "uni0651064E" } },
              { { "uni064F", "uni0651" }, { "uni0651064F" } },
              { { "uni0651", "uni064B" }, { "uni0651064B" } },
              { { "uni0651", "uni064C" }, { "uni0651064C" } },
              { { "uni0651", "uni064E" }, { "uni0651064E" } },
              { { "uni0651", "uni064F" }, { "uni0651064F" } },
              { { "uni0651", "uni0670" }, { "absShaddaAlef" } },
              { { "uni0670", "uni0651" }, { "absShaddaAlef" } },
          }
        },
        { "f3", "lu6", "arab", "dflt", "rlig",
          new String[][][] {
              { { "uni064D", "uni0651" }, { "uni0651064D" } },
              { { "uni0650", "uni0651" }, { "uni06510650" } },
              { { "uni0651", "uni0650" }, { "uni06510650" } },
              { { "uni0651", "uni064D" }, { "uni0651064D" } },
          }
        },
        { "f3", "lu7", "arab", "dflt", "rlig",
          new String[][][] {
              { { "uni0647", "uni0654" }, { "uni06C0" } },
              { { "uni0647.fina", "uni0654" }, { "uni06C0.fina" } },
              { { "uni0647.init", "uni0654" }, { "uni06C0.init" } },
              { { "uni0647.medi", "uni0654" }, { "uni06C0.medi" } },
              { { "uni0648", "uni0654" }, { "uni0624" } },
              { { "uni0648.fina", "uni0654" }, { "uni0624.fina" } },
              { { "uni064A", "uni0654" }, { "uni0626" } },
              { { "uni064A.fina", "uni0654" }, { "uni0626.fina" } },
              { { "uni064A.init", "uni0654" }, { "uni0626.init" } },
              { { "uni064A.medi", "uni0654" }, { "uni0626.medi" } },
              { { "uni06C1", "uni0654" }, { "uni06C2" } },
              { { "uni06C1.fina", "uni0654" }, { "uni06C2.fina" } },
              { { "uni06C1.init", "uni0654" }, { "uni06C2.init" } },
              { { "uni06C1.medi", "uni0654" }, { "uni06C2.medi" } },
           }
        },
        { "f3", "lu8", "arab", "dflt", "rlig",
          new String[][][] {
              { { "uni064E", "uni0654" }, { "uni0654064E" } },
              { { "uni064F", "uni0654" }, { "uni0654064F" } },
              { { "uni0654", "uni064E" }, { "uni0654064E" } },
              { { "uni0654", "uni064F" }, { "uni0654064F" } },
          }
        },
    };

    private static Object[][] ltContextual = {
        { GlyphSubstitutionTable.GSUB_LOOKUP_TYPE_CONTEXTUAL },
        // arab-001.ttx - none used
        // arab-002.ttx - none used
        // arab-003.ttx - none used
        // arab-004.ttx - none used
    };

    private static Object[][] ltChainedContextual = {
        { GlyphSubstitutionTable.GSUB_LOOKUP_TYPE_CHAINED_CONTEXTUAL },
        // arab-001.ttx
        { "f0", "lu1", "arab", "dflt", "ccmp",
          new String[][][] {
              { { "wawwithhamzaabove", "alefwithhamzabelow" }, { "wawwithhamzaabove", "arabicalef", "uni0655" } },
              { { "reh", "alefwithhamzabelow" }, { "reh", "arabicalef", "uni0655" } },
              { { "zain", "alefwithhamzabelow" }, { "zain", "arabicalef", "uni0655" } },
              { { "waw", "alefwithhamzabelow" }, { "waw", "arabicalef", "uni0655" } },
              { { "jeh", "alefwithhamzabelow" }, { "jeh", "arabicalef", "uni0655" } },
          }
        },
        { "f0", "lu3", "arab", "dflt", "isol",
          new String[][][] {
              { { "hamza", "heh" }, { "hamza", "hehisolated" } },
              { { "alefwithmaddaabove", "heh" }, { "alefwithmaddaabove", "hehisolated" } },
              { { "alefwithhamzaabove", "heh" }, { "alefwithhamzaabove", "hehisolated" } },
              { { "wawwithhamzaabove", "heh" }, { "wawwithhamzaabove", "hehisolated" } },
              { { "alefwithhamzabelow", "heh" }, { "alefwithhamzabelow", "hehisolated" } },
              { { "arabicalef", "heh" }, { "arabicalef", "hehisolated" } },
              { { "tehmarbuta", "heh" }, { "tehmarbuta", "hehisolated" } },
              { { "dal", "heh" }, { "dal", "hehisolated" } },
              { { "thal", "heh" }, { "thal", "hehisolated" } },
              { { "reh", "heh" }, { "reh", "hehisolated" } },
              { { "zain", "heh" }, { "zain", "hehisolated" } },
              { { "waw", "heh" }, { "waw", "hehisolated" } },
              { { "alefwasla", "heh" }, { "alefwasla", "hehisolated" } },
              { { "jeh", "heh" }, { "jeh", "hehisolated" } },
              { { "arabicae", "heh" }, { "arabicae", "hehisolated" } },
              { { "alefwaslafinal", "heh" }, { "alefwaslafinal", "hehisolated" } },
              { { "alefwithmaddaabovefinal", "heh" }, { "alefwithmaddaabovefinal", "hehisolated" } },
              { { "alefwithhamzaabovefinal", "heh" }, { "alefwithhamzaabovefinal", "hehisolated" } },
              { { "alefwithhamzabelowfinal", "heh" }, { "alefwithhamzabelowfinal", "hehisolated" } },
              { { "aleffinal", "heh" }, { "aleffinal", "hehisolated" } },
              { { "tehmarbutafinal", "heh" }, { "tehmarbutafinal", "hehisolated" } },
              { { "lamwithalefmaddaaboveisolatedd", "heh" }, { "lamwithalefmaddaaboveisolatedd", "hehisolated" } },
              { { "lamwithalefmaddaabovefinal", "heh" }, { "lamwithalefmaddaabovefinal", "hehisolated" } },
              { { "lamwithalefhamzaaboveisolatedd", "heh" }, { "lamwithalefhamzaaboveisolatedd", "hehisolated" } },
              { { "lamwithalefhamzaabovefinal", "heh" }, { "lamwithalefhamzaabovefinal", "hehisolated" } },
              { { "lamwithalefhamzabelowisolated", "heh" }, { "lamwithalefhamzabelowisolated", "hehisolated" } },
              { { "lamwithalefhamzabelowfinal", "heh" }, { "lamwithalefhamzabelowfinal", "hehisolated" } },
              { { "lamwithalefisolated", "heh" }, { "lamwithalefisolated", "hehisolated" } },
              { { "lamwithaleffinal", "heh" }, { "lamwithaleffinal", "hehisolated" } },
          }
        },
        // arab-002.ttx
        { "f1", "lu2", "arab", "dflt", "isol",
          new String[][][] {
              { { "hamza", "heh" }, { "hamza", "hehisolated" } },
              { { "alefwithmaddaabove", "heh" }, { "alefwithmaddaabove", "hehisolated" } },
              { { "alefwithhamzaabove", "heh" }, { "alefwithhamzaabove", "hehisolated" } },
              { { "wawwithhamzaabove", "heh" }, { "wawwithhamzaabove", "hehisolated" } },
              { { "alefwithhamzabelow", "heh" }, { "alefwithhamzabelow", "hehisolated" } },
              { { "arabicalef", "heh" }, { "arabicalef", "hehisolated" } },
              { { "tehmarbuta", "heh" }, { "tehmarbuta", "hehisolated" } },
              { { "dal", "heh" }, { "dal", "hehisolated" } },
              { { "thal", "heh" }, { "thal", "hehisolated" } },
              { { "reh", "heh" }, { "reh", "hehisolated" } },
              { { "zain", "heh" }, { "zain", "hehisolated" } },
              { { "waw", "heh" }, { "waw", "hehisolated" } },
              { { "alefwasla", "heh" }, { "alefwasla", "hehisolated" } },
              { { "jeh", "heh" }, { "jeh", "hehisolated" } },
              { { "arabicae", "heh" }, { "arabicae", "hehisolated" } },
              { { "alefwaslafinal", "heh" }, { "alefwaslafinal", "hehisolated" } },
              { { "alefwithmaddaabovefinal", "heh" }, { "alefwithmaddaabovefinal", "hehisolated" } },
              { { "alefwithhamzaabovefinal", "heh" }, { "alefwithhamzaabovefinal", "hehisolated" } },
              { { "alefwithhamzabelowfinal", "heh" }, { "alefwithhamzabelowfinal", "hehisolated" } },
              { { "aleffinal", "heh" }, { "aleffinal", "hehisolated" } },
              { { "tehmarbutafinal", "heh" }, { "tehmarbutafinal", "hehisolated" } },
              { { "lamwithalefmaddaaboveisolatedd", "heh" }, { "lamwithalefmaddaaboveisolatedd", "hehisolated" } },
              { { "lamwithalefmaddaabovefinal", "heh" }, { "lamwithalefmaddaabovefinal", "hehisolated" } },
              { { "lamwithalefhamzaaboveisolatedd", "heh" }, { "lamwithalefhamzaaboveisolatedd", "hehisolated" } },
              { { "lamwithalefhamzaabovefinal", "heh" }, { "lamwithalefhamzaabovefinal", "hehisolated" } },
              { { "lamwithalefhamzabelowisolated", "heh" }, { "lamwithalefhamzabelowisolated", "hehisolated" } },
              { { "lamwithalefhamzabelowfinal", "heh" }, { "lamwithalefhamzabelowfinal", "hehisolated" } },
              { { "lamwithalefisolated", "heh" }, { "lamwithalefisolated", "hehisolated" } },
              { { "lamwithaleffinal", "heh" }, { "lamwithaleffinal", "hehisolated" } },
            }
        },
        { "f1", "lu11", "arab", "dflt", "calt",
          new String[][][] {
              { { "pehinitial", "fatha", "pehmedial", "fatha" }, { "pehinitial", "tatweel", "fatha", "pehmedial", "fatha" } },
              { { "yehwithhamzaaboveinitial", "damma", "vehmedial", "damma" }, { "yehwithhamzaaboveinitial", "tatweel", "damma", "vehmedial", "damma" } },
              { { "behinitial", "shadda", "jehfinal", "shadda" }, { "behinitial", "tatweel", "shadda", "jehfinal", "shadda" } },
              { { "tehinitial", "sukun", "behmedial", "sukun" }, { "tehinitial", "tatweel", "sukun", "behmedial", "sukun" } },
              { { "thehinitial", "smallhighmadda", "tehmedial", "smallhighmadda" }, { "thehinitial", "tatweel", "smallhighmadda", "tehmedial", "smallhighmadda" } },
              { { "fehinitial", "fathaonhamza", "ainmedial", "fathaonhamza" }, { "fehinitial", "tatweel", "fathaonhamza", "ainmedial", "fathaonhamza" } },
              { { "qafinitial", "dammaonhamza", "qafmedial", "dammaonhamza" }, { "qafinitial", "tatweel", "dammaonhamza", "qafmedial", "dammaonhamza" } },
              { { "nooninitial", "superscriptalef", "wawfinal", "superscriptalef" }, { "nooninitial", "tatweel", "superscriptalef", "wawfinal", "superscriptalef" } },
              { { "yehinitial", "dammatanonhamza", "rehfinal", "dammatanonhamza" }, { "yehinitial", "tatweel", "dammatanonhamza", "rehfinal", "dammatanonhamza" } },
              { { "uni0649.init", "uni0654", "wawwithhamzaabovefinal", "uni0654" }, { "uni0649.init", "tatweel", "uni0654", "wawwithhamzaabovefinal", "uni0654" } },
          }
        },
        { "f1", "lu12", "arab", "dflt", "calt",
          new String[][][] {
              { { "pehmedial", "fatha", "pehmedial", "fatha" }, { "pehmedial", "tatweel", "fatha", "pehmedial", "fatha" } },
              { { "yehwithhamzaabovemedial", "damma", "vehmedial", "damma" }, { "yehwithhamzaabovemedial", "tatweel", "damma", "vehmedial", "damma" } },
              { { "behmedial", "shadda", "wawwithhamzaabovefinal", "shadda" }, { "behmedial", "tatweel", "shadda", "wawwithhamzaabovefinal", "shadda" } },
              { { "tehmedial", "sukun", "rehfinal", "sukun" }, { "tehmedial", "tatweel", "sukun", "rehfinal", "sukun" } },
              { { "thehmedial", "smallhighmadda", "zainfinal", "smallhighmadda" }, { "thehmedial", "tatweel", "smallhighmadda", "zainfinal", "smallhighmadda" } },
              { { "noonmedial", "superscriptalef", "ainmedial", "superscriptalef" }, { "noonmedial", "tatweel", "superscriptalef", "ainmedial", "superscriptalef" } },
              { { "yehmedial", "dammatanonhamza", "wawfinal", "dammatanonhamza" }, { "yehmedial", "tatweel", "dammatanonhamza", "wawfinal", "dammatanonhamza" } },
              { { "uni0649.medi", "uni0654", "yehmedial", "uni0654" }, { "uni0649.medi", "tatweel", "uni0654", "yehmedial", "uni0654" } },
          }
        },
        // arab-003.ttx
        { "f2", "lu4", "arab", "dflt", "rlig",
          new String[][][] {
              { { "uni0644.medi", "uni0622.fina" }, { "uni0644.medi.preAlef", "uni0622.fina.postLamMed" } },
              { { "uni06B5.medi", "uni0622.fina" }, { "uni06B5.medi.preAlef", "uni0622.fina.postLamMed" } },
              { { "uni06B6.medi", "uni0622.fina" }, { "uni06B6.medi.preAlef", "uni0622.fina.postLamMed" } },
              { { "uni06B7.medi", "uni0622.fina" }, { "uni06B7.medi.preAlef", "uni0622.fina.postLamMed" } },
              { { "uni06B8.medi", "uni0622.fina" }, { "uni06B8.medi.preAlef", "uni0622.fina.postLamMed" } },
              { { "absLamRetroMed", "uni0622.fina" }, { "absLamRetroMed.preAlef", "uni0622.fina.postLamMed" } },
              { { "uni076A.medi", "uni0622.fina" }, { "uni076A.medi.preAlef", "uni0622.fina.postLamMed" } },
              { { "uni0644.init", "uni0622.fina" }, { "uni0644.init.preAlef", "uni0622.fina.postLamIni" } },
              { { "uni06B5.init", "uni0622.fina" }, { "uni06B5.init.preAlef", "uni0622.fina.postLamIni" } },
              { { "uni06B6.init", "uni0622.fina" }, { "uni06B6.init.preAlef", "uni0622.fina.postLamIni" } },
              { { "uni06B7.init", "uni0622.fina" }, { "uni06B7.init.preAlef", "uni0622.fina.postLamIni" } },
              { { "uni06B8.init", "uni0622.fina" }, { "uni06B8.init.preAlef", "uni0622.fina.postLamIni" } },
              { { "absLamRetroIni", "uni0622.fina" }, { "absLamRetroIni.preAlef", "uni0622.fina.postLamIni" } },
              { { "uni076A.init", "uni0622.fina" }, { "uni076A.init.preAlef", "uni0622.fina.postLamIni" } },
          }
        },
        { "f2", "lu8", "arab", "dflt", "calt",
          new String[][][] {
              { { "uni064A", "uni0670" }, { "uni064A", "uni0670.large" } },
          }
        },
        { "f2", "lu13", "arab", "dflt", "calt",
          new String[][][] {
              { { "uni06DD", "one" }, { "uni06DD", "oneMedium" } },
              { { "uni06DD", "one", "two" }, { "uni06DD.2", "oneMedium", "twoMedium" } },
              { { "uni06DD", "one", "two", "three" }, { "uni06DD.3", "oneSmall", "twoSmall", "threeSmall" } },
          }
        },
        // arab-004.ttx
        { "f3", "lu4", "arab", "dflt", "rlig",
          new String[][][] {
              { { "uni0644.medi", "uni0622.fina" }, { "uni0644.medi.preAlef", "uni0622.fina.postLamMed" } },
              { { "uni06B5.medi", "uni0622.fina" }, { "uni06B5.medi.preAlef", "uni0622.fina.postLamMed" } },
              { { "uni06B6.medi", "uni0622.fina" }, { "uni06B6.medi.preAlef", "uni0622.fina.postLamMed" } },
              { { "uni06B7.medi", "uni0622.fina" }, { "uni06B7.medi.preAlef", "uni0622.fina.postLamMed" } },
              { { "uni06B8.medi", "uni0622.fina" }, { "uni06B8.medi.preAlef", "uni0622.fina.postLamMed" } },
              { { "absLamRetroMed", "uni0622.fina" }, { "absLamRetroMed.preAlef", "uni0622.fina.postLamMed" } },
              { { "uni076A.medi", "uni0622.fina" }, { "uni076A.medi.preAlef", "uni0622.fina.postLamMed" } },
              { { "uni0644.init", "uni0622.fina" }, { "uni0644.init.preAlef", "uni0622.fina.postLamIni" } },
              { { "uni06B5.init", "uni0622.fina" }, { "uni06B5.init.preAlef", "uni0622.fina.postLamIni" } },
              { { "uni06B6.init", "uni0622.fina" }, { "uni06B6.init.preAlef", "uni0622.fina.postLamIni" } },
              { { "uni06B7.init", "uni0622.fina" }, { "uni06B7.init.preAlef", "uni0622.fina.postLamIni" } },
              { { "uni06B8.init", "uni0622.fina" }, { "uni06B8.init.preAlef", "uni0622.fina.postLamIni" } },
              { { "absLamRetroIni", "uni0622.fina" }, { "absLamRetroIni.preAlef", "uni0622.fina.postLamIni" } },
              { { "uni076A.init", "uni0622.fina" }, { "uni076A.init.preAlef", "uni0622.fina.postLamIni" } },
          }
        },
        { "f3", "lu9", "arab", "dflt", "calt",
          new String[][][] {
              { { "uni0601", "uni0661" }, { "uni0601", "uni0661.Medium" } },
              { { "uni0601", "uni0661", "uni0662" }, { "uni0601.2", "uni0661.Medium", "uni0662.Medium" } },
              { { "uni0601", "uni0661", "uni0662", "uni0663" }, { "uni0601.3", "uni0661.Medium", "uni0662.Medium", "uni0663.Medium", } },
              { { "uni0601", "uni0661", "uni0662", "uni0663", "uni0664" }, { "uni0601.4", "uni0661.Medium", "uni0662.Medium", "uni0663.Medium", "uni0664.Medium" } },
          }
        },
        { "f3", "lu10", "arab", "dflt", "calt",
          new String[][][] {
              { { "uni064A", "uni0670" }, { "uni064A", "uni0670.large" } },
          }
        },
    };

    @Test
    public void testGSUBSingle() throws Exception {
        performSubstitutions ( ltSingle );
    }

    @Test
    public void testGSUBMultiple() throws Exception {
        performSubstitutions ( ltMultiple );
    }

    @Test
    public void testGSUBAlternate() throws Exception {
        performSubstitutions ( ltAlternate );
    }

    @Test
    public void testGSUBLigature() throws Exception {
        performSubstitutions ( ltLigature );
    }

    @Test
    public void testGSUBContextual() throws Exception {
        performSubstitutions ( ltContextual );
    }

    @Test
    public void testGSUBChainedContextual() throws Exception {
        performSubstitutions ( ltChainedContextual );
    }

    /**
     * Perform substitutions on all test data in test specification TS.
     * @param ts test specification
     */
    private void performSubstitutions ( Object[][] ts ) {
        assert ts.length > 0;
        Object[] tp = ts[0];
        for ( int i = 1; i < ts.length; i++ ) {
            performSubstitutions ( tp, ts[i] );
        }
    }

    /**
     * Perform substitutions on all test data TD using test parameters TP.
     * @param tp test parameters
     * @param td test data
     */
    private void performSubstitutions ( Object[] tp, Object[] td ) {
        assert tp.length > 0;
        if ( td.length > 5 ) {
            String fid = (String) td[0];
            String lid = (String) td[1];
            String script = (String) td[2];
            String language = (String) td[3];
            String feature = (String) td[4];
            TTXFile tf = findTTX ( fid );
            assertTrue ( tf != null );
            GlyphSubstitutionTable gsub = tf.getGSUB();
            assertTrue ( gsub != null );
            GlyphSubstitutionSubtable[] sta = findGSUBSubtables ( gsub, script, language, feature, lid );
            assertTrue ( sta != null );
            assertTrue ( sta.length > 0 );
            ScriptContextTester sct = findScriptContextTester ( script, language, feature );
            String[][][] tia = (String[][][]) td[5];            // test instance array
            for ( String[][] ti : tia ) {                       // test instance
                if ( ti != null ) {
                    if ( ti.length > 1 ) {                      // must have at least input and output glyph id arrays
                        String[] igia = ti[0];                  // input glyph id array
                        String[] ogia = ti[1];                  // output glyph id array
                        GlyphSequence igs = tf.getGlyphSequence ( igia );
                        GlyphSequence ogs = tf.getGlyphSequence ( ogia );
                        GlyphSequence tgs = GlyphSubstitutionSubtable.substitute ( igs, script, language, feature, sta, sct );
                        assertSameGlyphs ( ogs, tgs );
                    }
                }
            }
        }
    }

    private String findTTXPath ( String fid ) {
        for ( String[] fs : ttxFonts ) {
            if ( ( fs != null ) && ( fs.length > 1 ) ) {
                if ( fs[0].equals ( fid ) ) {
                    return ttxFilesRoot + File.separator + fs[1];
                }
            }
        }
        return null;
    }

    private TTXFile findTTX ( String fid ) {
        String pn = findTTXPath ( fid );
        assertTrue ( pn != null );
        try {
            TTXFile tf = TTXFile.getFromCache ( pn );
            return tf;
        } catch ( Exception e ) {
            fail ( e.getMessage() );
            return null;
        }
    }

    private GlyphSubstitutionSubtable[] findGSUBSubtables ( GlyphSubstitutionTable gsub, String script, String language, String feature, String lid ) {
        LookupTable lt = gsub.getLookupTable ( lid );
        if ( lt != null ) {
            return (GlyphSubstitutionSubtable[]) lt.getSubtables();
        } else {
            return null;
        }
    }

    private ScriptContextTester findScriptContextTester ( String script, String language, String feature ) {
        return this;
    }

    public GlyphContextTester getTester ( String feature ) {
        return this;
    }

    public boolean test ( String script, String language, String feature, GlyphSequence gs, int index, int flags ) {
        return true;
    }

    private void assertSameGlyphs ( GlyphSequence gs1, GlyphSequence gs2 ) {
        assertNotNull ( gs1 );
        assertNotNull ( gs2 );
        IntBuffer gb1 = gs1.getGlyphs();
        IntBuffer gb2 = gs2.getGlyphs();
        assertEquals ( "unequal glyph count", gb1.limit(), gb2.limit() );
        for ( int i = 0; i < gb1.limit(); i++ ) {
            int g1 = gb1.get(i);
            int g2 = gb2.get(i);
            assertEquals ( "unequal glyph code", g1, g2 );
        }
    }

}
