//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Sim/Fitting/SimDataPair.h
//! @brief     Defines class SimDataPair.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifdef SWIG
#error no need to expose this header to Swig
#endif // SWIG
#ifndef BORNAGAIN_SIM_FITTING_SIMDATAPAIR_H
#define BORNAGAIN_SIM_FITTING_SIMDATAPAIR_H

#include "Sim/Fitting/FitTypes.h"

class Datafield;
class SimulationResult;

//! Holds pair of simulation/experimental data to fit.

class SimDataPair {
public:
    SimDataPair(simulation_builder_t builder, const Datafield& raw_data,
                std::unique_ptr<Datafield>&& raw_stdv, double user_weight = 1.0);

    SimDataPair(simulation_builder_t builder, const Datafield& raw_data,
                std::unique_ptr<Datafield>&& raw_stdv, std::unique_ptr<Datafield>&& user_weights);

    SimDataPair(SimDataPair&& other);

    ~SimDataPair();

    void execSimulation(const mumufit::Parameters& params);

    bool containsUncertainties() const;

    //! Returns the result of last computed simulation
    SimulationResult simulationResult() const;

    //! Returns the experimental data cut to the ROI area
    SimulationResult experimentalData() const;

    //! Returns the data uncertainties cut to the ROI area
    //! If no uncertainties present, returns zero-filled SimulationResult.
    SimulationResult uncertainties() const;

    //! Returns the user uncertainties cut to the ROI area.
    SimulationResult userWeights() const;

    //! Returns the relative difference between simulated
    //! and experimental data cut to the ROI area
    SimulationResult relativeDifference() const;

    //! Returns the absolute difference between simulated
    //! and experimental data cut to the ROI area
    SimulationResult absoluteDifference() const;

    //! Returns the flattened simulated intensities cut to the ROI area
    std::vector<double> simulation_array() const;

    //! Returns the flattened experimental data cut to the ROI area
    std::vector<double> experimental_array() const;

    //! Returns the flattened experimental uncertainties
    //! cut to the ROI area. If no uncertainties are available,
    //! Returns a zero-filled array sized to the ROI area.
    std::vector<double> uncertainties_array() const;

    //! Returns a flat array of user weights cut to the ROI area.
    std::vector<double> user_weights_array() const;

private:
    void validate() const;

    //! ISimulation builder from the user to construct simulation for given set of parameters.
    simulation_builder_t m_simulation_builder;

    //! Current simulation results. Masked areas are nullified.
    std::unique_ptr<SimulationResult> m_sim_data;
    //! Experimental data cut to the ROI. Masked areas are nullified.
    std::unique_ptr<SimulationResult> m_exp_data;
    //! Weights from experimental data uncertainties. Masked areas are nullified.
    std::unique_ptr<SimulationResult> m_uncertainties;
    //! Manually defined (user) weights. Masked areas are nullified.
    std::unique_ptr<SimulationResult> m_user_weights;

    //! Raw experimental data as obtained from the user.
    std::unique_ptr<Datafield> m_raw_data;
    //! Data uncertainties as provided by the user
    std::unique_ptr<Datafield> m_raw_uncertainties;
    //! User-defined weights
    std::unique_ptr<Datafield> m_raw_user_weights;
};

#endif // BORNAGAIN_SIM_FITTING_SIMDATAPAIR_H
