package junit.test.parser.pst;

import java.awt.Color;

import junit.test.parser.TestCodeParser;
import latexDraw.parsers.pst.PSTParameters;
import latexDraw.parsers.pst.PSTParametersParser;
import latexDraw.psTricks.DviPsColors;
import latexDraw.psTricks.PSTricksConstants;

import org.junit.Before;

public class TestPSTParametersParser extends TestCodeParser
{
	@Override
	@Before
	public void setUp()
	{
		parser  = new PSTParametersParser("", null);
		parser2 = new PSTParametersParser("", null);
	}

	
	@Override
	public void testParse()
	{
		try {
			PSTParametersParser pars = (PSTParametersParser)parser;
			PSTParameters param      = new PSTParameters();
			
			pars.setParam(param);
			pars.setCode("");
			pars.parse();
			
			try {
				pars.setCode("  [");
				pars.parse();
				fail();
			}catch(Exception e) { e.printStackTrace(); }
			
			pars.setCode("[ ]");
			pars.parse();
			pars.setCode("[]");
			pars.parse();
			pars.setCode("]");
			pars.parse();
			pars.setCode("[%comment\n]");
			pars.parse();
			pars.setCode("[ \n \t    ]");
			pars.parse();
			pars.setCode("[linewidth=0.2, linewwidth=0.2]");
			pars.parse();
			pars.setCode("[linewidth=0.2 linewwidth=0.2]");
			pars.parse();
			pars.setCode("[linewidth=0.2, ,  linecolor=red]");
			pars.parse();
			assertEquals(Color.RED, param.lineColor);
			pars.setCode("[linewidth=0.2,linewidth=5]");
			pars.parse();
			assertEquals(5., param.lineWidth);
			pars.setCode("[,]");
			pars.parse();
			pars.setCode("[=2]");
			pars.parse();
			pars.setCode("[fldsjf=fdsf]");
			pars.parse();
			pars.setCode("[fldsjf=]");
			pars.parse();
			pars.setCode("[fldsjf=,]");
			pars.parse();
			pars.setCode("[=,dfs]");
			pars.parse();
			
			parse_linewidth(param);
			parse_doubleline(param);
			parse_linecolor(param);
			parse_boxsep(param);
			parse_dotsep(param);
			parse_doublesep(param);
			parse_doublecolor(param);
			parse_dimen(param);
			parse_dash(param);
			parse_fillcolor(param);
			parse_fillstyle(param);
			parse_framearc(param);
			parse_framesep(param);
			parse_gradbegin(param);
			parse_gradend(param);
			parse_gradlines(param);
			parse_gradmidpoint(param);
			parse_gradangle(param);
			parse_gangle(param);
			parse_hatchangle(param);
			parse_hatchwidth(param);
			parse_hatchsep(param);
			parse_hatchcolor(param);
			parse_linestyle(param);
			parse_shadowsize(param);
			parse_shadowangle(param);
			parse_shadowcolor(param);
		}catch(Exception e) { fail(); }
	}

	
	private void parse_shadowcolor(PSTParameters param)
	{
		try {
			Color c = new Color(120,110,120);
			DviPsColors.addUserColour(c, "mycol4");
			
			parser.setCode("[shadowcolor \t=\t white]");
			parser.parse();
			assertEquals(Color.WHITE, param.shadowCol);
			parser.setCode("[shadowcolor\r=\nmycol4]");
			parser.parse();
			assertEquals(c, param.shadowCol);
			parser.setCode("[shadowcolor\t=\tfdsfds]");
			parser.parse();
			assertEquals(c, param.shadowCol);
			parser.setCode("[shadowcolor\r=]");
			parser.parse();
			assertEquals(c, param.shadowCol);
			parser.setCode("[shadowcolor%dfsfds\n\t\r]");
			parser.parse();
			assertEquals(c, param.shadowCol);
		}catch(Exception e) { fail(); }
	}
	
	
	private void parse_shadowangle(PSTParameters param)
	{
		try {
			parser.setCode("[shadowangle \t  %       fds   \n \t   %dfsf\n \t =\t\n\r.4]");
			parser.parse();
			assertEquals(.4, param.shadowAngle);
			parser.setCode("[shadowangle%dsq\n=\t0.8]");
			parser.parse();
			assertEquals(.8, param.shadowAngle);
			parser.setCode("[shadowangle=20..]");
			parser.parse();
			assertEquals(0.8, param.shadowAngle);
			parser.setCode("[shadowangle\t=%dsq\n-20]");
			parser.parse();
			assertEquals(-20., param.shadowAngle);
			parser.setCode("[shadowangle=-.2]");
			parser.parse();
			assertEquals(-.2, param.shadowAngle);
			parser.setCode("[shadowangle\t=\t---+--0.40]");
			parser.parse();
			assertEquals(-.4, param.shadowAngle);
			parser.setCode("[shadowangle\t=\t+++--+0.80]");
			parser.parse();
			assertEquals(.8, param.shadowAngle);
		}catch(Exception e) { fail(); }
	}
	
	
	private void parse_shadowsize(PSTParameters param)
	{
		try {
			parser.setCode("[shadowsize\t=\t \r\n.4]");
			parser.parse();
			assertEquals(.4, param.shadowSize);
			parser.setCode("[shadowsize= .8\r\n\tcm]");
			parser.parse();
			assertEquals(.8, param.shadowSize);
			parser.setCode("[shadowsize=2 cm\t]");
			parser.parse();
			assertEquals(2., param.shadowSize);
			parser.setCode("[shadowsize\r%fdsf\n=\n\t 2.2%comment\r cm]");
			parser.parse();
			assertEquals(2.2, param.shadowSize);
			parser.setCode("[shadowsize=2 in]");
			parser.parse();
			assertEquals(2.*PSTricksConstants.INCH_VAL_CM, param.shadowSize);
			parser.setCode("[%fdfsd\n shadowsize =2pt ]");
			parser.parse();
			assertEquals(2./PSTricksConstants.CM_VAL_PT, param.shadowSize);
			parser.setCode("[shadowsize = 2mm]");
			parser.parse();
			assertEquals(0.2, param.shadowSize);
		}catch(Exception e) { fail(); }
	}

	
	
	private void parse_linestyle(PSTParameters param)
	{
		try {
			parser.setCode("[linestyle = test]");
			parser.parse();
			assertEquals("test", param.lineStyle);
			parser.setCode("[linestyle =]");
			parser.parse();
			assertEquals("test", param.lineStyle);
			parser.setCode("[linestyle =%fdsf\n]");
			parser.parse();
			assertEquals("test", param.lineStyle);
			parser.setCode("[linestyle =1289K3]");
			parser.parse();
			assertEquals("test", param.lineStyle);
			parser.setCode("[linestyle =0.3]");
			parser.parse();
			assertEquals("test", param.lineStyle);
			parser.setCode("[linestyle\t=\taa]");
			parser.parse();
			assertEquals("aa", param.lineStyle);
		}catch(Exception e) { fail(); }
	}
	
	
	private void parse_hatchcolor(PSTParameters param)
	{
		try {
			Color c = new Color(120,110,110);
			DviPsColors.addUserColour(c, "mycol3");
			
			parser.setCode("[hatchcolor \t=\t white]");
			parser.parse();
			assertEquals(Color.WHITE, param.hatchCol);
			parser.setCode("[hatchcolor\r=\nmycol3]");
			parser.parse();
			assertEquals(c, param.hatchCol);
			parser.setCode("[hatchcolor\t=\tfdsfds]");
			parser.parse();
			assertEquals(c, param.hatchCol);
			parser.setCode("[hatchcolor\r=]");
			parser.parse();
			assertEquals(c, param.hatchCol);
			parser.setCode("[hatchcolor%dfsfds\n\t\r]");
			parser.parse();
			assertEquals(c, param.hatchCol);
		}catch(Exception e) { fail(); }
	}
	
	
	private void parse_hatchsep(PSTParameters param)
	{
		try {
			parser.setCode("[hatchsep\t=\t \r\n.4]");
			parser.parse();
			assertEquals(.4, param.hatchSep);
			parser.setCode("[hatchsep= .8\r\n\tcm]");
			parser.parse();
			assertEquals(.8, param.hatchSep);
			parser.setCode("[hatchsep=2 cm\t]");
			parser.parse();
			assertEquals(2., param.hatchSep);
			parser.setCode("[hatchsep\r%fdsf\n=\n\t 2.2%comment\r cm]");
			parser.parse();
			assertEquals(2.2, param.hatchSep);
			parser.setCode("[hatchsep=2 in]");
			parser.parse();
			assertEquals(2.*PSTricksConstants.INCH_VAL_CM, param.hatchSep);
			parser.setCode("[hatchsep = 2 pt ]");
			parser.parse();
			assertEquals(2./PSTricksConstants.CM_VAL_PT, param.hatchSep);
			parser.setCode("[hatchsep = 2mm]");
			parser.parse();
			assertEquals(0.2, param.hatchWidth);
		}catch(Exception e) { fail(); }
	}
	
	
	private void parse_hatchwidth(PSTParameters param)
	{
		try {
			parser.setCode("[hatchwidth\t=\t \r\n.4]");
			parser.parse();
			assertEquals(.4, param.hatchWidth);
			parser.setCode("[hatchwidth= .8\r\ncm]");
			parser.parse();
			assertEquals(.8, param.hatchWidth);
			parser.setCode("[hatchwidth=2 cm]");
			parser.parse();
			assertEquals(2., param.hatchWidth);
			parser.setCode("[hatchwidth\r%fdsf\n=\n\t 2.2%comment\r cm]");
			parser.parse();
			assertEquals(2.2, param.hatchWidth);
			parser.setCode("[hatchwidth=2 in]");
			parser.parse();
			assertEquals(2.*PSTricksConstants.INCH_VAL_CM, param.hatchWidth);
			parser.setCode("[hatchwidth = 2 pt ]");
			parser.parse();
			assertEquals(2./PSTricksConstants.CM_VAL_PT, param.hatchWidth);
			parser.setCode("[hatchwidth = 2mm]");
			parser.parse();
			assertEquals(0.2, param.hatchWidth);
		}catch(Exception e) { fail(); }
	}
	
	
	
	private void parse_hatchangle(PSTParameters param)
	{
		try {
			parser.setCode("[hatchangle \t  %       fds   \n \t   %dfsf\n \t =\t\n\r.4]");
			parser.parse();
			assertEquals(.4, param.hatchAngle);
			parser.setCode("[hatchangle%dsq\n=\t0.8]");
			parser.parse();
			assertEquals(.8, param.hatchAngle);
			parser.setCode("[hatchangle=20..]");
			parser.parse();
			assertEquals(0.8, param.hatchAngle);
			parser.setCode("[hatchangle\t=%dsq\n-20]");
			parser.parse();
			assertEquals(-20., param.hatchAngle);
			parser.setCode("[hatchangle=-.2]");
			parser.parse();
			assertEquals(-.2, param.hatchAngle);
			parser.setCode("[hatchangle\t=\t---+--0.40]");
			parser.parse();
			assertEquals(-.4, param.hatchAngle);
			parser.setCode("[hatchangle\t=\t+++--+0.80]");
			parser.parse();
			assertEquals(.8, param.hatchAngle);
		}catch(Exception e) { fail(); }
	}
	
	
	
	private void parse_gangle(PSTParameters param)
	{
		try {
			parser.setCode("[gangle \t  %       fds   \n \t   %dfsf\n \t =\t\n\r.4]");
			parser.parse();
			assertEquals(.4, param.gangle);
			parser.setCode("[gangle%dsq\n=0.8]");
			parser.parse();
			assertEquals(.8, param.gangle);
			parser.setCode("[gangle=20..]");
			parser.parse();
			assertEquals(0.8, param.gangle);
			parser.setCode("[gangle\t=%dsq\n-20]");
			parser.parse();
			assertEquals(-20., param.gangle);
			parser.setCode("[gangle=-.2]");
			parser.parse();
			assertEquals(-.2, param.gangle);
			parser.setCode("[gangle\t=\t---+--0.40]");
			parser.parse();
			assertEquals(-.4, param.gangle);
			parser.setCode("[gangle\t=\t+++--+0.80]");
			parser.parse();
			assertEquals(.8, param.gangle);
		}catch(Exception e) { fail(); }
	}
	
	
	
	private void parse_gradangle(PSTParameters param)
	{
		try {
			parser.setCode("[gradangle\t%       fds   \r  %fdsfsd\n \t   %dfsf\n \t =\t\n\r.4]");
			parser.parse();
			assertEquals(.4, param.gradAngle);
			parser.setCode("[gradangle%dsq\n=0.8]");
			parser.parse();
			assertEquals(.8, param.gradAngle);
			parser.setCode("[gradangle=20]");
			parser.parse();
			assertEquals(20., param.gradAngle);
			parser.setCode("[gradangle\t=%dsq\n-20]");
			parser.parse();
			assertEquals(-20., param.gradAngle);
			parser.setCode("[gradangle=-.2]");
			parser.parse();
			assertEquals(-.2, param.gradAngle);
			parser.setCode("[gradangle\t=\t---0.40]");
			parser.parse();
			assertEquals(-.4, param.gradAngle);
			parser.setCode("[gradangle\t=\t+0.80]");
			parser.parse();
			assertEquals(.8, param.gradAngle);
		}catch(Exception e) { fail(); }
	}
	
	
	
	private void parse_gradmidpoint(PSTParameters param)
	{
		try {
			parser.setCode("[gradmidpoint\t%       fds   \r  %fdsfsd\n \t   %dfsf\n \t =\t\n\r.4]");
			parser.parse();
			assertEquals(.4, param.gradMidPoint);
			parser.setCode("[gradmidpoint=0.8]");
			parser.parse();
			assertEquals(.8, param.gradMidPoint);
			parser.setCode("[gradmidpoint=2]");
			parser.parse();
			assertEquals(0.8, param.gradMidPoint);
			parser.setCode("[gradmidpoint=20]");
			parser.parse();
			assertEquals(0.8, param.gradMidPoint);
			parser.setCode("[gradmidpoint=-2]");
			parser.parse();
			assertEquals(0.8, param.gradMidPoint);
			parser.setCode("[gradmidpoint\t=\t0.5]");
			parser.parse();
			assertEquals(0.5, param.gradMidPoint);
		}catch(Exception e) { fail(); }
	}
	
	
	private void parse_gradlines(PSTParameters param)
	{
		try {
			parser.setCode("[gradlines\t=\t4]");
			parser.parse();
			assertEquals(4, param.gradLines);
			parser.setCode("[gradlines=.8]");
			parser.parse();
			assertEquals(4, param.gradLines);
			parser.setCode("[gradlines=2]");
			parser.parse();
			assertEquals(2, param.gradLines);
			parser.setCode("[gradlines\t=\t1.4]");
			parser.parse();
			assertEquals(1, param.gradLines);
		}catch(Exception e) { fail(); }
	}
	
	
	private void parse_gradend(PSTParameters param)
	{
		try {
			Color c = new Color(110,110,110);
			DviPsColors.addUserColour(c, "mycol2");
			
			parser.setCode("[gradend \t=\t white]");
			parser.parse();
			assertEquals(Color.WHITE, param.gradEnd);
			parser.setCode("[gradend\r=\nmycol2]");
			parser.parse();
			assertEquals(c, param.gradEnd);
			parser.setCode("[gradend=fdsfds]");
			parser.parse();
			assertEquals(c, param.gradEnd);
			parser.setCode("[gradend=]");
			parser.parse();
			assertEquals(c, param.gradEnd);
			parser.setCode("[gradend%dfsfds\n\t\r]");
			parser.parse();
			assertEquals(c, param.gradEnd);
		}catch(Exception e) { fail(); }
	}
	
	
	private void parse_gradbegin(PSTParameters param)
	{
		try {
			Color c = new Color(100,100,100);
			DviPsColors.addUserColour(c, "mycol");
			
			parser.setCode("[gradbegin \t=\t white]");
			parser.parse();
			assertEquals(Color.WHITE, param.gradBegin);
			parser.setCode("[gradbegin %fdsf\r= \n  mycol]");
			parser.parse();
			assertEquals(c, param.gradBegin);
			parser.setCode("[gradbegin=fdsfds]");
			parser.parse();
			assertEquals(c, param.gradBegin);
			parser.setCode("[gradbegin=]");
			parser.parse();
			assertEquals(c, param.gradBegin);
			parser.setCode("[gradbegin%dfsfds\n\t\r]");
			parser.parse();
			assertEquals(c, param.gradBegin);
		}catch(Exception e) { fail(); }
	}
	
	
	
	private void parse_framearc(PSTParameters param)
	{
		try {
			parser.setCode("[framearc\t=\t.4]");
			parser.parse();
			assertEquals(.4, param.frameArc);
			parser.setCode("[framearc=0.8]");
			parser.parse();
			assertEquals(.8, param.frameArc);
			parser.setCode("[framearc=2]");
			parser.parse();
			assertEquals(2., param.frameArc);
			parser.setCode("[framearc\t=\t4]");
			parser.parse();
			assertEquals(4., param.frameArc);
		}catch(Exception e) { fail(); }
	}
	
	
	private void parse_framesep(PSTParameters param)
	{
		try {
			parser.setCode("[framesep\t=\t \r\n.4]");
			parser.parse();
			assertEquals(.4, param.frameSep);
			parser.setCode("[framesep= .8\r\ncm]");
			parser.parse();
			assertEquals(.8, param.frameSep);
			parser.setCode("[framesep=2 cm]");
			parser.parse();
			assertEquals(2., param.frameSep);
			parser.setCode("[framesep\r%fdsf\n=\n\t 2.2%comment\r cm]");
			parser.parse();
			assertEquals(2.2, param.frameSep);
			parser.setCode("[framesep=2 in]");
			parser.parse();
			assertEquals(2.*PSTricksConstants.INCH_VAL_CM, param.frameSep);
			parser.setCode("[framesep = 2 pt ]");
			parser.parse();
			assertEquals(2./PSTricksConstants.CM_VAL_PT, param.frameSep);
			parser.setCode("[framesep = 2mm]");
			parser.parse();
			assertEquals(0.2, param.frameSep);
		}catch(Exception e) { fail(); }
	}
	
	
	private void parse_fillcolor(PSTParameters param)
	{
		try {
			parser.setCode("[fillcolor=green]");
			parser.parse();
			assertEquals(Color.GREEN, param.fillColor);
			parser.setCode("[\n\r\t  fillcolor\t%fdsf\r=\ncyan]");
			parser.parse();
			assertEquals(Color.CYAN, param.fillColor);
			parser.setCode("[fillcolor=fdsfds]");
			parser.parse();
			assertEquals(Color.CYAN, param.fillColor);
			parser.setCode("[fillcolor =]");
			parser.parse();
			assertEquals(Color.CYAN, param.fillColor);
			parser.setCode("[fillcolor %dfsfds\n]");
			parser.parse();
			assertEquals(Color.CYAN, param.fillColor);
		}catch(Exception e) { fail(); }
	}
	
	
	
	private void parse_fillstyle(PSTParameters param)
	{
		try {
			parser.setCode("[fillstyle = test]");
			parser.parse();
			assertEquals("test", param.fillStyle);
			parser.setCode("[fillstyle = test*]");
			parser.parse();
			assertEquals("test*", param.fillStyle);
			parser.setCode("[fillstyle =]");
			parser.parse();
			assertEquals("test*", param.fillStyle);
			parser.setCode("[fillstyle =%fdsf\n]");
			parser.parse();
			assertEquals("test*", param.fillStyle);
			parser.setCode("[fillstyle =1289K3]");
			parser.parse();
			assertEquals("test*", param.fillStyle);
			parser.setCode("[fillstyle =0.3]");
			parser.parse();
			assertEquals("test*", param.fillStyle);
			parser.setCode("[fillstyle\t=\taa]");
			parser.parse();
			assertEquals("aa", param.fillStyle);
			parser.setCode("[fillstyle\t=*]");
			parser.parse();
			assertEquals("aa", param.fillStyle);
		}catch(Exception e) { fail(); }
	}
	
	
	private void parse_dash(PSTParameters param)
	{
		try {
			parser.setCode("[dash=.4 2]");
			parser.parse();
			assertEquals(.4, param.dashBlack);
			assertEquals(2., param.dashWhite);
			parser.setCode("[dash=.8 cm \n %cfds %fdsf\n %fdsf \r 4 \t cm ]");
			parser.parse();
			assertEquals(.8, param.dashBlack);
			assertEquals(4., param.dashWhite);
			parser.setCode("[dash=.8mm \t%fdsf\n %fdsf \r 4mm ]");
			parser.parse();
			assertEquals(.08, param.dashBlack);
			assertEquals(.4, param.dashWhite);
			parser.setCode("[dash=.9mm mm ]");
			parser.parse();
			assertEquals(.08, param.dashBlack);
			assertEquals(.4, param.dashWhite);
			parser.setCode("[dash=.9fd ]");
			parser.parse();
			assertEquals(.08, param.dashBlack);
			assertEquals(.4, param.dashWhite);
			parser.setCode("[dash=.d9fd ]");
			parser.parse();
			assertEquals(.08, param.dashBlack);
			assertEquals(.4, param.dashWhite);
			parser.setCode("[dash\r=\n.d9fd ]");
			parser.parse();
			assertEquals(.08, param.dashBlack);
			assertEquals(.4, param.dashWhite);
			parser.setCode("[dash=.9cm0.1cm]");
			parser.parse();
			assertEquals(.9, param.dashBlack);
			assertEquals(.1, param.dashWhite);
		}catch(Exception e) { fail(); }
	}
	
	
	private void parse_dimen(PSTParameters param)
	{
		try {
			parser.setCode("[dimen = outer]");
			parser.parse();
			assertEquals(PSTricksConstants.BORDERS_INSIDE, param.borderPos);
			parser.setCode("[dimen % fdsjif % fds\r = middle]");
			parser.parse();
			assertEquals(PSTricksConstants.BORDERS_MIDDLE, param.borderPos);
			parser.setCode("[\n %fdsfd \n \t dimen = %fds \n \n \n inner]");
			parser.parse();
			assertEquals(PSTricksConstants.BORDERS_OUTSIDE, param.borderPos);
			parser.setCode("[dimen = ]");
			parser.parse();
			assertEquals(PSTricksConstants.BORDERS_OUTSIDE, param.borderPos);
			parser.setCode("[dimen = %fdsfds \n ]");
			parser.parse();
			assertEquals(PSTricksConstants.BORDERS_OUTSIDE, param.borderPos);
			parser.setCode("[dimen = fdsfds ]");
			parser.parse();
			assertEquals(PSTricksConstants.BORDERS_OUTSIDE, param.borderPos);
			parser.setCode("[dimen = 124 ]");
			parser.parse();
			assertEquals(PSTricksConstants.BORDERS_OUTSIDE, param.borderPos);
			parser.setCode("[dimen = éà ]");
			parser.parse();
			assertEquals(PSTricksConstants.BORDERS_OUTSIDE, param.borderPos);
			parser.setCode("[dimen %dfsf \n=]");
			parser.parse();
			assertEquals(PSTricksConstants.BORDERS_OUTSIDE, param.borderPos);
		}catch(Exception e) { fail(); }
	}
	
	
	private void parse_doublecolor(PSTParameters param)
	{
		try {
			parser.setCode("[doublecolor = green]");
			parser.parse();
			assertEquals(Color.GREEN, param.dbleColor);
			parser.setCode("[doublecolor %fdsf\r= \n  cyan]");
			parser.parse();
			assertEquals(Color.CYAN, param.dbleColor);
			parser.setCode("[doublecolor=fdsfds]");
			parser.parse();
			assertEquals(Color.CYAN, param.dbleColor);
			parser.setCode("[doublecolor =]");
			parser.parse();
			assertEquals(Color.CYAN, param.dbleColor);
			parser.setCode("[doublecolor %dfsfds\n]");
			parser.parse();
			assertEquals(Color.CYAN, param.dbleColor);
		}catch(Exception e) { fail(); }
	}
	
	
	private void parse_doublesep(PSTParameters param)
	{
		try {
			parser.setCode("[doublesep=.4]");
			parser.parse();
			assertEquals(.4, param.dbleSep);
			parser.setCode("[doublesep=0.8cm]");
			parser.parse();
			assertEquals(.8, param.dbleSep);
			parser.setCode("[doublesep=2 cm]");
			parser.parse();
			assertEquals(2., param.dbleSep);
			parser.setCode("[doublesep \r %fdsf\n\t=\n\t 2.2\n%comment\r cm]");
			parser.parse();
			assertEquals(2.2, param.dbleSep);
			parser.setCode("[doublesep=2 in]");
			parser.parse();
			assertEquals(2.*PSTricksConstants.INCH_VAL_CM, param.dbleSep);
			parser.setCode("[doublesep = 2 pt ]");
			parser.parse();
			assertEquals(2./PSTricksConstants.CM_VAL_PT, param.dbleSep);
			parser.setCode("[doublesep = 2mm]");
			parser.parse();
			assertEquals(0.2, param.dbleSep);
		}catch(Exception e) { fail(); }
	}
	
	
	
	private void parse_dotsep(PSTParameters param)
	{
		try {
			parser.setCode("[dotsep=2]");
			parser.parse();
			assertEquals(2., param.dotStep);
			parser.setCode("[dotsep=2cm]");
			parser.parse();
			assertEquals(2., param.dotStep);
			parser.setCode("[dotsep=2 cm]");
			parser.parse();
			assertEquals(2., param.dotStep);
			parser.setCode("[dotsep\r\t=2.2\n%comment\r cm]");
			parser.parse();
			assertEquals(2.2, param.dotStep);
			parser.setCode("[dotsep=2 in]");
			parser.parse();
			assertEquals(2.*PSTricksConstants.INCH_VAL_CM, param.dotStep);
			parser.setCode("[dotsep=2 pt]");
			parser.parse();
			assertEquals(2./PSTricksConstants.CM_VAL_PT, param.dotStep);
			parser.setCode("[dotsep=2 mm]");
			parser.parse();
			assertEquals(0.2, param.dotStep);
		}catch(Exception e) { fail(); }
	}
	
	
	
	private void parse_boxsep(PSTParameters param)
	{
		try {
			parser.setCode("[boxsep=true]");
			parser.parse();
			assertTrue(param.boxSep);
			parser.setCode("[boxsep=false]");
			parser.parse();
			assertFalse(param.boxSep);
			parser.setCode("[boxsep=]");
			parser.parse();
			assertFalse(param.boxSep);
			parser.setCode("[boxsep=fdsfds]");
			parser.parse();
			assertFalse(param.boxSep);
			parser.setCode("[boxsep=%fdjsf\n \t true]");
			parser.parse();
			assertTrue(param.boxSep);
			parser.setCode("[boxsep= \t %fdjsf\n\r \t false]");
			parser.parse();
			assertFalse(param.boxSep);
		}catch(Exception e) { fail(); }
	}
	
	
	private void parse_linecolor(PSTParameters param)
	{
		try {
			parser.setCode("[linecolor=blue]");
			parser.parse();
			assertEquals(Color.BLUE, param.lineColor);
			parser.setCode("[linecolor=yellow]");
			parser.parse();
			assertEquals(Color.YELLOW, param.lineColor);
			parser.setCode("[linecolor=fdsfds]");
			parser.parse();
			assertEquals(Color.YELLOW, param.lineColor);
			parser.setCode("[linecolor=]");
			parser.parse();
			assertEquals(Color.YELLOW, param.lineColor);
			parser.setCode("[linecolor]");
			parser.parse();
			assertEquals(Color.YELLOW, param.lineColor);
		}catch(Exception e) { fail(); }
	}
	
	
	
	private void parse_doubleline(PSTParameters param)
	{
		try {
			parser.setCode("[doubleline=true]");
			parser.parse();
			assertTrue(param.dbleLine);
			parser.setCode("[doubleline=false]");
			parser.parse();
			assertFalse(param.dbleLine);
			parser.setCode("[doubleline=fdsfds]");
			parser.parse();
			assertFalse(param.dbleLine);
			parser.setCode("[doubleline=]");
			parser.parse();
			assertFalse(param.dbleLine);
			parser.setCode("[doubleline]");
			parser.parse();
			assertFalse(param.dbleLine);
			parser.setCode("[doubleline==_è-é]");
			parser.parse();
			assertFalse(param.dbleLine);
		}catch(Exception e) { fail(); }
	}
	
	
	private void parse_linewidth(PSTParameters param)
	{
		try {
			parser.setCode("[linewidth=0.2cm]");
			parser.parse();
			assertEquals(0.2, param.lineWidth);
			parser.setCode("[linewidth=0.2mm]");
			parser.parse();
			assertEquals(0.02, param.lineWidth);
			parser.setCode("[linewidth=0.2]");
			parser.parse();
			assertEquals(0.2, param.lineWidth);
			parser.setCode("[linewidth=0.2 pt]");
			parser.parse();
			assertEquals(0.2/PSTricksConstants.CM_VAL_PT, param.lineWidth);
			parser.setCode("[linewidth=0.2 in]");
			parser.parse();
			assertEquals(0.2*PSTricksConstants.INCH_VAL_CM, param.lineWidth);
			parser.setCode("[linewidth=0.2 in]");
			parser.parse();
			assertEquals(0.2*PSTricksConstants.INCH_VAL_CM, param.lineWidth);
			parser.setCode("[linewidth= aa ]");
			parser.parse();
			assertEquals(0.2*PSTricksConstants.INCH_VAL_CM, param.lineWidth);
			parser.setCode("[linewidth= aa in]");
			parser.parse();
			assertEquals(0.2*PSTricksConstants.INCH_VAL_CM, param.lineWidth);
			parser.setCode("[linewidth= in]");
			parser.parse();
			assertEquals(0.2*PSTricksConstants.INCH_VAL_CM, param.lineWidth);
		}catch(Exception e) { fail(); }
	}
	
	
	@Override
	public void testSkipComment()
	{
		parser.setCode("%comment\na");
		assertEquals(parser.skipComment(), "comment");
		assertEquals(parser.getPosition(), 9);
		assertEquals(parser.getLinePosition(), 2);
		parser.setCode("\\%not a comment\na");
		assertNull(parser.skipComment());
		assertEquals(parser.getPosition(), 0);
		assertEquals(parser.getLinePosition(), 1);
		parser.setCode("%comment");
		assertEquals(parser.skipComment(), "comment");
		assertEquals(parser.getPosition(), 9);
		assertEquals(parser.getLinePosition(), 2);
	}

	
	
	@Override
	public void testSkipWSP()
	{
		parser.setCode("   test");
		parser.skipWSP();
		assertEquals('t', parser.getChar());
		parser.skipWSP();
		assertEquals('t', parser.getChar());
		parser.setCode("");
		parser.skipWSP();
		assertTrue(parser.isEOC());
		parser.setCode(" \n \t \r  test");
		parser.skipWSP();
		assertEquals('t', parser.getChar());
		parser.skipWSP();
		assertEquals('t', parser.getChar());
	}
}
