/*
 * Copyright © 2012 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/** @file getuniformblockindex.c
 *
 * Tests the glGetUniformBlockIndex API: returning INVALID_INDEX for
 * inactive blocks or error conditions, and consecutive numbers from 0
 * for active blocks.
 */

#include "piglit-util-gl.h"

PIGLIT_GL_TEST_CONFIG_BEGIN

	config.supports_gl_compat_version = 10;

	config.window_visual = PIGLIT_GL_VISUAL_DOUBLE | PIGLIT_GL_VISUAL_RGBA;

PIGLIT_GL_TEST_CONFIG_END

static GLuint prog;

static const char frag_shader_text[] =
	"#extension GL_ARB_uniform_buffer_object : require\n"
	"\n"
	"uniform ub_a { vec4 a; };\n"
	"uniform ub_b { vec4 b; };\n"
	"\n"
	"void main()\n"
	"{\n"
	"	gl_FragColor = a + b;\n"
	"}\n";

void
piglit_init(int argc, char **argv)
{
	bool pass = true;
	int expected_ub_b_index;
	int index;

	piglit_require_extension("GL_ARB_uniform_buffer_object");

	prog = piglit_build_simple_program(NULL, frag_shader_text);

	index = glGetUniformBlockIndex(prog, "ub_a");
	printf("Uniform block \"ub_a\" index: 0x%08x\n", index);
	if (index == 0) {
		expected_ub_b_index = 1;
	} else if (index == 1) {
		expected_ub_b_index = 0;
	} else {
		expected_ub_b_index = 1;
		printf("  Expected 0 or 1\n");
		pass = false;
	}

	/* From the GL_ARB_uniform_buffer_object spec:
	 *
	 *     "GetUniformBlockIndex returns the uniform block index
	 *      for the uniform block named <uniformBlockName> of
	 *      <program>. If <uniformBlockName> does not identify an
	 *      active uniform block of <program>, or an error
	 *      occurred, then INVALID_INDEX is returned.  The indices
	 *      of the active uniform blocks of a program are assigned
	 *      in consecutive order, beginning with zero."
	 */
	index = glGetUniformBlockIndex(prog, "ub_b");
	printf("Uniform block \"ub_b\" index: 0x%08x\n", index);
	if (index != expected_ub_b_index) {
		printf("  Expected 0x%08x\n", expected_ub_b_index);
		pass = false;
	}

	/* Since we can't guarantee that an unused uniform block is
	 * inactive, we ask for a block name that was never even defined.
	 */
	index = glGetUniformBlockIndex(prog, "ub_c");
	printf("Uniform block \"ub_c\" index: 0x%08x\n", index);
	if (index != GL_INVALID_INDEX) {
		printf("  Expected 0x%08x\n", GL_INVALID_INDEX);
		pass = false;
	}

	if (!piglit_check_gl_error(0)) {
		pass = false;
	}

	/* From the GL_ARB_uniform_buffer_object spec:
	 *
	 *     "The error INVALID_VALUE is generated by
	 *      GetUniformIndices, GetActiveUniformsiv,
	 *      GetActiveUniformName, GetUniformBlockIndex,
	 *      GetActiveUniformBlockiv, GetActiveUniformBlockName,
	 *      and UniformBlockBinding if <program> is not a value
	 *      generated by GL.
	 */
	index = glGetUniformBlockIndex(0xd0d0, "a");
	if (!piglit_check_gl_error(GL_INVALID_VALUE)) {
		pass = false;
	} else if (index != GL_INVALID_INDEX) {
		printf("Bad program uniform index: 0x%08x\n", index);
		printf("  Expected 0x%08x\n", GL_INVALID_INDEX);
		pass = false;
	}

	piglit_report_result(pass ? PIGLIT_PASS : PIGLIT_FAIL);
}

enum piglit_result piglit_display(void)
{
	/* UNREACHED */
	return PIGLIT_FAIL;
}
