<?php
class AtmoNouvelleAquitaineBridge extends BridgeAbstract {

	const NAME = 'Atmo Nouvelle Aquitaine';
	const URI = 'https://www.atmo-nouvelleaquitaine.org';
	const DESCRIPTION = 'Fetches the latest air polution of cities in Nouvelle Aquitaine from Atmo';
	const MAINTAINER = 'floviolleau';
	const PARAMETERS = array(array(
		'cities' => array(
			'name' => 'Choisir une ville',
			'type' => 'list',
			'values' => self::CITIES
		)
	));
	const CACHE_TIMEOUT = 7200;

	private $dom;

	private function getClosest($search, $arr) {
		$closest = null;
		foreach ($arr as $key => $value) {
			if ($closest === null || abs((int)$search - $closest) > abs((int)$key - (int)$search)) {
				$closest = (int)$key;
			}
		}
		return $arr[$closest];
	}

	public function collectData() {
		$uri = self::URI . '/monair/commune/' . $this->getInput('cities');

		$html = getSimpleHTMLDOM($uri);

		$this->dom = $html->find('#block-system-main .city-prevision-map', 0);

		$message = $this->getIndexMessage() . ' ' . $this->getQualityMessage();
		$message .= ' ' . $this->getTomorrowTrendIndexMessage() . ' ' . $this->getTomorrowTrendQualityMessage();

		$item['uri'] = $uri;
		$today = date('d/m/Y');
		$item['title'] = "Bulletin de l'air du $today pour la région Nouvelle Aquitaine.";
		$item['title'] .= ' Retrouvez plus d\'informations en allant sur atmo-nouvelleaquitaine.org #QualiteAir.';
		$item['author'] = 'floviolleau';
		$item['content'] = $message;
		$item['uid'] = hash('sha256', $item['title']);

		$this->items[] = $item;
	}

	private function getIndex() {
		$index = $this->dom->find('.indice', 0)->innertext;

		if ($index == 'XX') {
			return -1;
		}

		return $index;
	}

	private function getMaxIndexText() {
		// will return '/100'
		return $this->dom->find('.pourcent', 0)->innertext;
	}

	private function getQualityText($index, $indexes) {
		if ($index == -1) {
			if (array_key_exists('no-available', $indexes)) {
				return $indexes['no-available'];
			}

			return 'Aucune donnée';
		}

		return $this->getClosest($index, $indexes);
	}

	private function getLegendIndexes() {
		$rawIndexes = $this->dom->find('.prevision-legend .prevision-legend-label');
		$indexes = array();
		for ($i = 0; $i < count($rawIndexes); $i++) {
			if ($rawIndexes[$i]->hasAttribute('data-color')) {
				$indexes[$rawIndexes[$i]->getAttribute('data-color')] = $rawIndexes[$i]->innertext;
			}
		}

		return $indexes;
	}

	private function getTomorrowTrendIndex() {
		$tomorrowTrendDomNode = $this->dom
			->find('.day-controls.raster-controls .list-raster-controls .raster-control', 2);
		$tomorrowTrendIndexNode = null;

		if ($tomorrowTrendDomNode) {
			$tomorrowTrendIndexNode = $tomorrowTrendDomNode->find('.raster-control-link', 0);
		}

		if ($tomorrowTrendIndexNode && $tomorrowTrendIndexNode->hasAttribute('data-index')) {
			$tomorrowTrendIndex = $tomorrowTrendIndexNode->getAttribute('data-index');
		} else {
			return -1;
		}

		return $tomorrowTrendIndex;
	}

	private function getTomorrowTrendQualityText($trendIndex, $indexes) {
		if ($trendIndex == -1) {
			if (array_key_exists('no-available', $indexes)) {
				return $indexes['no-available'];
			}

			return 'Aucune donnée';
		}

		return $this->getClosest($trendIndex, $indexes);
	}

	private function getIndexMessage() {
		$index = $this->getIndex();
		$maxIndexText = $this->getMaxIndexText();

		if ($index == -1) {
			return 'Aucune donnée pour l\'indice.';
		}

		return "L'indice d'aujourd'hui est $index$maxIndexText.";
	}

	private function getQualityMessage() {
		$index = $index = $this->getIndex();
		$indexes = $this->getLegendIndexes();
		$quality = $this->getQualityText($index, $indexes);

		if ($index == -1) {
			return 'Aucune donnée pour la qualité de l\'air.';
		}

		return "La qualité de l'air est $quality.";
	}

	private function getTomorrowTrendIndexMessage() {
		$trendIndex = $this->getTomorrowTrendIndex();
		$maxIndexText = $this->getMaxIndexText();

		if ($trendIndex == -1) {
			return 'Aucune donnée pour l\'indice prévu demain.';
		}

		return "L'indice prévu pour demain est $trendIndex$maxIndexText.";
	}

	private function getTomorrowTrendQualityMessage() {
		$trendIndex = $this->getTomorrowTrendIndex();
		$indexes = $this->getLegendIndexes();
		$trendQuality = $this->getTomorrowTrendQualityText($trendIndex, $indexes);

		if ($trendIndex == -1) {
			return 'Aucune donnée pour la qualité de l\'air de demain.';
		}
		return "La qualite de l'air pour demain sera $trendQuality.";
	}

	const CITIES = array(
		'Aast (64460)' => '64001',
		'Abère (64160)' => '64002',
		'Abidos (64150)' => '64003',
		'Abitain (64390)' => '64004',
		'Abjat-sur-Bandiat (24300)' => '24001',
		'Abos (64360)' => '64005',
		'Abzac (16500)' => '16001',
		'Abzac (33230)' => '33001',
		'Accous (64490)' => '64006',
		'Adilly (79200)' => '79002',
		'Adriers (86430)' => '86001',
		'Affieux (19260)' => '19001',
		'Agen (47000)' => '47001',
		'Agmé (47350)' => '47002',
		'Agnac (47800)' => '47003',
		'Agnos (64400)' => '64007',
		'Agonac (24460)' => '24002',
		'Agris (16110)' => '16003',
		'Agudelle (17500)' => '17002',
		'Ahaxe-Alciette-Bascassan (64220)' => '64008',
		'Ahetze (64210)' => '64009',
		'Ahun (23150)' => '23001',
		'Aïcirits-Camou-Suhast (64120)' => '64010',
		'Aiffres (79230)' => '79003',
		'Aignes-et-Puypéroux (16190)' => '16004',
		'Aigonnay (79370)' => '79004',
		'Aigre (16140)' => '16005',
		'Aigrefeuille-d\'Aunis (17290)' => '17003',
		'Aiguillon (47190)' => '47004',
		'Aillas (33124)' => '33002',
		'Aincille (64220)' => '64011',
		'Ainharp (64130)' => '64012',
		'Ainhice-Mongelos (64220)' => '64013',
		'Ainhoa (64250)' => '64014',
		'Aire-sur-l\'Adour (40800)' => '40001',
		'Airvault (79600)' => '79005',
		'Aix (19200)' => '19002',
		'Aixe-sur-Vienne (87700)' => '87001',
		'Ajain (23380)' => '23002',
		'Ajat (24210)' => '24004',
		'Albignac (19190)' => '19003',
		'Albussac (19380)' => '19004',
		'Alçay-Alçabéhéty-Sunharette (64470)' => '64015',
		'Aldudes (64430)' => '64016',
		'Allas-Bocage (17150)' => '17005',
		'Allas-Champagne (17500)' => '17006',
		'Allas-les-Mines (24220)' => '24006',
		'Allassac (19240)' => '19005',
		'Allemans (24600)' => '24007',
		'Allemans-du-Dropt (47800)' => '47005',
		'Alles-sur-Dordogne (24480)' => '24005',
		'Alleyrat (19200)' => '19006',
		'Alleyrat (23200)' => '23003',
		'Allez-et-Cazeneuve (47110)' => '47006',
		'Allonne (79130)' => '79007',
		'Allons (47420)' => '47007',
		'Alloue (16490)' => '16007',
		'Alos-Sibas-Abense (64470)' => '64017',
		'Altillac (19120)' => '19007',
		'Amailloux (79350)' => '79008',
		'Ambarès-et-Lagrave (33440)' => '33003',
		'Ambazac (87240)' => '87002',
		'Ambérac (16140)' => '16008',
		'Ambernac (16490)' => '16009',
		'Amberre (86110)' => '86002',
		'Ambès (33810)' => '33004',
		'Ambleville (16300)' => '16010',
		'Ambrugeat (19250)' => '19008',
		'Ambrus (47160)' => '47008',
		'Amendeuix-Oneix (64120)' => '64018',
		'Amorots-Succos (64120)' => '64019',
		'Amou (40330)' => '40002',
		'Amuré (79210)' => '79009',
		'Anais (16560)' => '16011',
		'Anais (17540)' => '17007',
		'Ance (64570)' => '64020',
		'Anché (86700)' => '86003',
		'Andernos-les-Bains (33510)' => '33005',
		'Andilly (17230)' => '17008',
		'Andiran (47170)' => '47009',
		'Andoins (64420)' => '64021',
		'Andrein (64390)' => '64022',
		'Angaïs (64510)' => '64023',
		'Angeac-Champagne (16130)' => '16012',
		'Angeac-Charente (16120)' => '16013',
		'Angeduc (16300)' => '16014',
		'Anglade (33390)' => '33006',
		'Angles-sur-l\'Anglin (86260)' => '86004',
		'Anglet (64600)' => '64024',
		'Angliers (17540)' => '17009',
		'Angliers (86330)' => '86005',
		'Angoisse (24270)' => '24008',
		'Angoulême (16000)' => '16015',
		'Angoulins (17690)' => '17010',
		'Angoumé (40990)' => '40003',
		'Angous (64190)' => '64025',
		'Angresse (40150)' => '40004',
		'Anhaux (64220)' => '64026',
		'Anlhiac (24160)' => '24009',
		'Annepont (17350)' => '17011',
		'Annesse-et-Beaulieu (24430)' => '24010',
		'Annezay (17380)' => '17012',
		'Anos (64160)' => '64027',
		'Anoye (64350)' => '64028',
		'Ansac-sur-Vienne (16500)' => '16016',
		'Antagnac (47700)' => '47010',
		'Antezant-la-Chapelle (17400)' => '17013',
		'Anthé (47370)' => '47011',
		'Antigny (86310)' => '86006',
		'Antonne-et-Trigonant (24420)' => '24011',
		'Antran (86100)' => '86007',
		'Anville (16170)' => '16017',
		'Anzême (23000)' => '23004',
		'Anzex (47700)' => '47012',
		'Aramits (64570)' => '64029',
		'Arancou (64270)' => '64031',
		'Araujuzon (64190)' => '64032',
		'Araux (64190)' => '64033',
		'Arbanats (33640)' => '33007',
		'Arbérats-Sillègue (64120)' => '64034',
		'Arbis (33760)' => '33008',
		'Arbonne (64210)' => '64035',
		'Arboucave (40320)' => '40005',
		'Arbouet-Sussaute (64120)' => '64036',
		'Arbus (64230)' => '64037',
		'Arcachon (33120)' => '33009',
		'Arçais (79210)' => '79010',
		'Arcangues (64200)' => '64038',
		'Arçay (86200)' => '86008',
		'Arces (17120)' => '17015',
		'Archiac (17520)' => '17016',
		'Archignac (24590)' => '24012',
		'Archigny (86210)' => '86009',
		'Archingeay (17380)' => '17017',
		'Arcins (33460)' => '33010',
		'Ardilleux (79110)' => '79011',
		'Ardillières (17290)' => '17018',
		'Ardin (79160)' => '79012',
		'Aren (64400)' => '64039',
		'Arengosse (40110)' => '40006',
		'Arès (33740)' => '33011',
		'Aressy (64320)' => '64041',
		'Arette (64570)' => '64040',
		'Arfeuille-Châtain (23700)' => '23005',
		'Argagnon (64300)' => '64042',
		'Argelos (40700)' => '40007',
		'Argelos (64450)' => '64043',
		'Argelouse (40430)' => '40008',
		'Argentat (19400)' => '19010',
		'Argenton (47250)' => '47013',
		'Argenton-l\'Église (79290)' => '79014',
		'Argentonnay (79150)' => '79013',
		'Arget (64410)' => '64044',
		'Arhansus (64120)' => '64045',
		'Arjuzanx (40110)' => '40009',
		'Armendarits (64640)' => '64046',
		'Armillac (47800)' => '47014',
		'Arnac-la-Poste (87160)' => '87003',
		'Arnac-Pompadour (19230)' => '19011',
		'Arnéguy (64220)' => '64047',
		'Arnos (64370)' => '64048',
		'Aroue-Ithorots-Olhaïby (64120)' => '64049',
		'Arrast-Larrebieu (64130)' => '64050',
		'Arraute-Charritte (64120)' => '64051',
		'Arrènes (23210)' => '23006',
		'Arricau-Bordes (64350)' => '64052',
		'Arrien (64420)' => '64053',
		'Arros-de-Nay (64800)' => '64054',
		'Arrosès (64350)' => '64056',
		'Ars (16130)' => '16018',
		'Ars (23480)' => '23007',
		'Ars-en-Ré (17590)' => '17019',
		'Arsac (33460)' => '33012',
		'Arsague (40330)' => '40011',
		'Artassenx (40090)' => '40012',
		'Arthenac (17520)' => '17020',
		'Arthez-d\'Armagnac (40190)' => '40013',
		'Arthez-d\'Asson (64800)' => '64058',
		'Arthez-de-Béarn (64370)' => '64057',
		'Artigueloutan (64420)' => '64059',
		'Artiguelouve (64230)' => '64060',
		'Artigues-près-Bordeaux (33370)' => '33013',
		'Artix (64170)' => '64061',
		'Arudy (64260)' => '64062',
		'Arue (40120)' => '40014',
		'Arvert (17530)' => '17021',
		'Arveyres (33500)' => '33015',
		'Arx (40310)' => '40015',
		'Arzacq-Arraziguet (64410)' => '64063',
		'Asasp-Arros (64660)' => '64064',
		'Ascain (64310)' => '64065',
		'Ascarat (64220)' => '64066',
		'Aslonnes (86340)' => '86010',
		'Asnières-en-Poitou (79170)' => '79015',
		'Asnières-la-Giraud (17400)' => '17022',
		'Asnières-sur-Blour (86430)' => '86011',
		'Asnières-sur-Nouère (16290)' => '16019',
		'Asnois (86250)' => '86012',
		'Asques (33240)' => '33016',
		'Assais-les-Jumeaux (79600)' => '79016',
		'Assat (64510)' => '64067',
		'Asson (64800)' => '64068',
		'Astaffort (47220)' => '47015',
		'Astaillac (19120)' => '19012',
		'Aste-Béon (64260)' => '64069',
		'Astis (64450)' => '64070',
		'Athos-Aspis (64390)' => '64071',
		'Aubagnan (40700)' => '40016',
		'Aubas (24290)' => '24014',
		'Aubazines (19190)' => '19013',
		'Aubertin (64290)' => '64072',
		'Aubeterre-sur-Dronne (16390)' => '16020',
		'Aubiac (33430)' => '33017',
		'Aubiac (47310)' => '47016',
		'Aubigné (79110)' => '79018',
		'Aubigny (79390)' => '79019',
		'Aubin (64230)' => '64073',
		'Aubous (64330)' => '64074',
		'Aubusson (23200)' => '23008',
		'Audaux (64190)' => '64075',
		'Audenge (33980)' => '33019',
		'Audignon (40500)' => '40017',
		'Audon (40400)' => '40018',
		'Audrix (24260)' => '24015',
		'Auga (64450)' => '64077',
		'Auge (23170)' => '23009',
		'Augé (79400)' => '79020',
		'Auge-Saint-Médard (16170)' => '16339',
		'Augères (23210)' => '23010',
		'Augignac (24300)' => '24016',
		'Augne (87120)' => '87004',
		'Aujac (17770)' => '17023',
		'Aulnay (17470)' => '17024',
		'Aulnay (86330)' => '86013',
		'Aulon (23210)' => '23011',
		'Aumagne (17770)' => '17025',
		'Aunac (16460)' => '16023',
		'Auradou (47140)' => '47017',
		'Aureil (87220)' => '87005',
		'Aureilhan (40200)' => '40019',
		'Auriac (19220)' => '19014',
		'Auriac (64450)' => '64078',
		'Auriac-du-Périgord (24290)' => '24018',
		'Auriac-sur-Dropt (47120)' => '47018',
		'Auriat (23400)' => '23012',
		'Aurice (40500)' => '40020',
		'Auriolles (33790)' => '33020',
		'Aurions-Idernes (64350)' => '64079',
		'Auros (33124)' => '33021',
		'Aussac-Vadalle (16560)' => '16024',
		'Aussevielle (64230)' => '64080',
		'Aussurucq (64130)' => '64081',
		'Auterrive (64270)' => '64082',
		'Autevielle-Saint-Martin-Bideren (64390)' => '64083',
		'Authon-Ébéon (17770)' => '17026',
		'Auzances (23700)' => '23013',
		'Availles-en-Châtellerault (86530)' => '86014',
		'Availles-Limouzine (86460)' => '86015',
		'Availles-Thouarsais (79600)' => '79022',
		'Avanton (86170)' => '86016',
		'Avensan (33480)' => '33022',
		'Avon (79800)' => '79023',
		'Avy (17800)' => '17027',
		'Aydie (64330)' => '64084',
		'Aydius (64490)' => '64085',
		'Ayen (19310)' => '19015',
		'Ayguemorte-les-Graves (33640)' => '33023',
		'Ayherre (64240)' => '64086',
		'Ayron (86190)' => '86017',
		'Aytré (17440)' => '17028',
		'Azat-Châtenet (23210)' => '23014',
		'Azat-le-Ris (87360)' => '87006',
		'Azay-le-Brûlé (79400)' => '79024',
		'Azay-sur-Thouet (79130)' => '79025',
		'Azerables (23160)' => '23015',
		'Azerat (24210)' => '24019',
		'Azur (40140)' => '40021',
		'Badefols-d\'Ans (24390)' => '24021',
		'Badefols-sur-Dordogne (24150)' => '24022',
		'Bagas (33190)' => '33024',
		'Bagnizeau (17160)' => '17029',
		'Bahus-Soubiran (40320)' => '40022',
		'Baigneaux (33760)' => '33025',
		'Baignes-Sainte-Radegonde (16360)' => '16025',
		'Baigts (40380)' => '40023',
		'Baigts-de-Béarn (64300)' => '64087',
		'Bajamont (47480)' => '47019',
		'Balansun (64300)' => '64088',
		'Balanzac (17600)' => '17030',
		'Baleix (64460)' => '64089',
		'Baleyssagues (47120)' => '47020',
		'Baliracq-Maumusson (64330)' => '64090',
		'Baliros (64510)' => '64091',
		'Balizac (33730)' => '33026',
		'Ballans (17160)' => '17031',
		'Balledent (87290)' => '87007',
		'Ballon (17290)' => '17032',
		'Balzac (16430)' => '16026',
		'Banca (64430)' => '64092',
		'Baneuil (24150)' => '24023',
		'Banize (23120)' => '23016',
		'Banos (40500)' => '40024',
		'Bar (19800)' => '19016',
		'Barbaste (47230)' => '47021',
		'Barbezières (16140)' => '16027',
		'Barbezieux-Saint-Hilaire (16300)' => '16028',
		'Barcus (64130)' => '64093',
		'Bardenac (16210)' => '16029',
		'Bardos (64520)' => '64094',
		'Bardou (24560)' => '24024',
		'Barie (33190)' => '33027',
		'Barinque (64160)' => '64095',
		'Baron (33750)' => '33028',
		'Barraute-Camu (64390)' => '64096',
		'Barret (16300)' => '16030',
		'Barro (16700)' => '16031',
		'Bars (24210)' => '24025',
		'Barsac (33720)' => '33030',
		'Barzan (17120)' => '17034',
		'Barzun (64530)' => '64097',
		'Bas-Mauco (40500)' => '40026',
		'Bascons (40090)' => '40025',
		'Bassac (16120)' => '16032',
		'Bassanne (33190)' => '33031',
		'Bassens (33530)' => '33032',
		'Bassercles (40700)' => '40027',
		'Basses (86200)' => '86018',
		'Bassignac-le-Bas (19430)' => '19017',
		'Bassignac-le-Haut (19220)' => '19018',
		'Bassillac (24330)' => '24026',
		'Bassillon-Vauzé (64350)' => '64098',
		'Bassussarry (64200)' => '64100',
		'Bastanès (64190)' => '64099',
		'Bastennes (40360)' => '40028',
		'Basville (23260)' => '23017',
		'Bats (40320)' => '40029',
		'Baudignan (40310)' => '40030',
		'Baudreix (64800)' => '64101',
		'Baurech (33880)' => '33033',
		'Bayac (24150)' => '24027',
		'Bayas (33230)' => '33034',
		'Bayers (16460)' => '16033',
		'Bayon-sur-Gironde (33710)' => '33035',
		'Bayonne (64100)' => '64102',
		'Bazac (16210)' => '16034',
		'Bazas (33430)' => '33036',
		'Bazauges (17490)' => '17035',
		'Bazelat (23160)' => '23018',
		'Bazens (47130)' => '47022',
		'Beaugas (47290)' => '47023',
		'Beaugeay (17620)' => '17036',
		'Beaulieu-sous-Parthenay (79420)' => '79029',
		'Beaulieu-sur-Dordogne (19120)' => '19019',
		'Beaulieu-sur-Sonnette (16450)' => '16035',
		'Beaumont (19390)' => '19020',
		'Beaumont (86490)' => '86019',
		'Beaumont-du-Lac (87120)' => '87009',
		'Beaumontois en Périgord (24440)' => '24028',
		'Beaupouyet (24400)' => '24029',
		'Beaupuy (47200)' => '47024',
		'Beauregard-de-Terrasson (24120)' => '24030',
		'Beauregard-et-Bassac (24140)' => '24031',
		'Beauronne (24400)' => '24032',
		'Beaussac (24340)' => '24033',
		'Beaussais-Vitré (79370)' => '79030',
		'Beautiran (33640)' => '33037',
		'Beauvais-sur-Matha (17490)' => '17037',
		'Beauville (47470)' => '47025',
		'Beauvoir-sur-Niort (79360)' => '79031',
		'Beauziac (47700)' => '47026',
		'Béceleuf (79160)' => '79032',
		'Bécheresse (16250)' => '16036',
		'Bédeille (64460)' => '64103',
		'Bedenac (17210)' => '17038',
		'Bedous (64490)' => '64104',
		'Bégaar (40400)' => '40031',
		'Bégadan (33340)' => '33038',
		'Bègles (33130)' => '33039',
		'Béguey (33410)' => '33040',
		'Béguios (64120)' => '64105',
		'Béhasque-Lapiste (64120)' => '64106',
		'Béhorléguy (64220)' => '64107',
		'Beissat (23260)' => '23019',
		'Beleymas (24140)' => '24034',
		'Belhade (40410)' => '40032',
		'Belin-Béliet (33830)' => '33042',
		'Bélis (40120)' => '40033',
		'Bellac (87300)' => '87011',
		'Bellebat (33760)' => '33043',
		'Bellechassagne (19290)' => '19021',
		'Bellefond (33760)' => '33044',
		'Bellefonds (86210)' => '86020',
		'Bellegarde-en-Marche (23190)' => '23020',
		'Belleville (79360)' => '79033',
		'Bellocq (64270)' => '64108',
		'Bellon (16210)' => '16037',
		'Belluire (17800)' => '17039',
		'Bélus (40300)' => '40034',
		'Belvès-de-Castillon (33350)' => '33045',
		'Benassay (86470)' => '86021',
		'Benayes (19510)' => '19022',
		'Bénéjacq (64800)' => '64109',
		'Bénesse-lès-Dax (40180)' => '40035',
		'Bénesse-Maremne (40230)' => '40036',
		'Benest (16350)' => '16038',
		'Bénévent-l\'Abbaye (23210)' => '23021',
		'Benon (17170)' => '17041',
		'Benquet (40280)' => '40037',
		'Bentayou-Sérée (64460)' => '64111',
		'Béost (64440)' => '64110',
		'Berbiguières (24220)' => '24036',
		'Bercloux (17770)' => '17042',
		'Bérenx (64300)' => '64112',
		'Bergerac (24100)' => '24037',
		'Bergouey (40250)' => '40038',
		'Bergouey-Viellenave (64270)' => '64113',
		'Bernac (16700)' => '16039',
		'Bernadets (64160)' => '64114',
		'Bernay-Saint-Martin (17330)' => '17043',
		'Berneuil (16480)' => '16040',
		'Berneuil (17460)' => '17044',
		'Berneuil (87300)' => '87012',
		'Bernos-Beaulac (33430)' => '33046',
		'Berrie (86120)' => '86022',
		'Berrogain-Laruns (64130)' => '64115',
		'Bersac-sur-Rivalier (87370)' => '87013',
		'Berson (33390)' => '33047',
		'Berthegon (86420)' => '86023',
		'Berthez (33124)' => '33048',
		'Bertric-Burée (24320)' => '24038',
		'Béruges (86190)' => '86024',
		'Bescat (64260)' => '64116',
		'Bésingrand (64150)' => '64117',
		'Bessac (16250)' => '16041',
		'Bessé (16140)' => '16042',
		'Besse (24550)' => '24039',
		'Bessines (79000)' => '79034',
		'Bessines-sur-Gartempe (87250)' => '87014',
		'Betbezer-d\'Armagnac (40240)' => '40039',
		'Bétête (23270)' => '23022',
		'Béthines (86310)' => '86025',
		'Bétracq (64350)' => '64118',
		'Beurlay (17250)' => '17045',
		'Beuste (64800)' => '64119',
		'Beuxes (86120)' => '86026',
		'Beychac-et-Caillau (33750)' => '33049',
		'Beylongue (40370)' => '40040',
		'Beynac (87700)' => '87015',
		'Beynac-et-Cazenac (24220)' => '24040',
		'Beynat (19190)' => '19023',
		'Beyrie-en-Béarn (64230)' => '64121',
		'Beyrie-sur-Joyeuse (64120)' => '64120',
		'Beyries (40700)' => '40041',
		'Beyssac (19230)' => '19024',
		'Beyssenac (19230)' => '19025',
		'Bézenac (24220)' => '24041',
		'Biard (86580)' => '86027',
		'Biarritz (64200)' => '64122',
		'Biarrotte (40390)' => '40042',
		'Bias (40170)' => '40043',
		'Bias (47300)' => '47027',
		'Biaudos (40390)' => '40044',
		'Bidache (64520)' => '64123',
		'Bidarray (64780)' => '64124',
		'Bidart (64210)' => '64125',
		'Bidos (64400)' => '64126',
		'Bielle (64260)' => '64127',
		'Bieujac (33210)' => '33050',
		'Biganos (33380)' => '33051',
		'Bignay (17400)' => '17046',
		'Bignoux (86800)' => '86028',
		'Bilhac (19120)' => '19026',
		'Bilhères (64260)' => '64128',
		'Billère (64140)' => '64129',
		'Bioussac (16700)' => '16044',
		'Birac (16120)' => '16045',
		'Birac (33430)' => '33053',
		'Birac-sur-Trec (47200)' => '47028',
		'Biras (24310)' => '24042',
		'Biriatou (64700)' => '64130',
		'Biron (17800)' => '17047',
		'Biron (24540)' => '24043',
		'Biron (64300)' => '64131',
		'Biscarrosse (40600)' => '40046',
		'Bizanos (64320)' => '64132',
		'Blaignac (33190)' => '33054',
		'Blaignan (33340)' => '33055',
		'Blanquefort (33290)' => '33056',
		'Blanquefort-sur-Briolance (47500)' => '47029',
		'Blanzac (87300)' => '87017',
		'Blanzac-lès-Matha (17160)' => '17048',
		'Blanzac-Porcheresse (16250)' => '16046',
		'Blanzaguet-Saint-Cybard (16320)' => '16047',
		'Blanzay (86400)' => '86029',
		'Blanzay-sur-Boutonne (17470)' => '17049',
		'Blasimon (33540)' => '33057',
		'Blaslay (86170)' => '86030',
		'Blaudeix (23140)' => '23023',
		'Blaye (33390)' => '33058',
		'Blaymont (47470)' => '47030',
		'Blésignac (33670)' => '33059',
		'Blessac (23200)' => '23024',
		'Blis-et-Born (24330)' => '24044',
		'Blond (87300)' => '87018',
		'Boé (47550)' => '47031',
		'Boeil-Bezing (64510)' => '64133',
		'Bois (17240)' => '17050',
		'Boisbreteau (16480)' => '16048',
		'Boismé (79300)' => '79038',
		'Boisné-La Tude (16320)' => '16082',
		'Boisredon (17150)' => '17052',
		'Boisse (24560)' => '24045',
		'Boisserolles (79360)' => '79039',
		'Boisseuil (87220)' => '87019',
		'Boisseuilh (24390)' => '24046',
		'Bommes (33210)' => '33060',
		'Bon-Encontre (47240)' => '47032',
		'Bonloc (64240)' => '64134',
		'Bonnac-la-Côte (87270)' => '87020',
		'Bonnat (23220)' => '23025',
		'Bonnefond (19170)' => '19027',
		'Bonnegarde (40330)' => '40047',
		'Bonnes (16390)' => '16049',
		'Bonnes (86300)' => '86031',
		'Bonnetan (33370)' => '33061',
		'Bonneuil (16120)' => '16050',
		'Bonneuil-Matours (86210)' => '86032',
		'Bonneville (16170)' => '16051',
		'Bonneville-et-Saint-Avit-de-Fumadières (24230)' => '24048',
		'Bonnut (64300)' => '64135',
		'Bonzac (33910)' => '33062',
		'Boos (40370)' => '40048',
		'Borce (64490)' => '64136',
		'Bord-Saint-Georges (23230)' => '23026',
		'Bordeaux (33000)' => '33063',
		'Bordères (64800)' => '64137',
		'Bordères-et-Lamensans (40270)' => '40049',
		'Bordes (64510)' => '64138',
		'Bords (17430)' => '17053',
		'Boresse-et-Martron (17270)' => '17054',
		'Borrèze (24590)' => '24050',
		'Bors (Canton de Baignes-Sainte-Radegonde) (16360)' => '16053',
		'Bors (Canton de Montmoreau-Saint-Cybard) (16190)' => '16052',
		'Bort-les-Orgues (19110)' => '19028',
		'Boscamnant (17360)' => '17055',
		'Bosdarros (64290)' => '64139',
		'Bosmie-l\'Aiguille (87110)' => '87021',
		'Bosmoreau-les-Mines (23400)' => '23027',
		'Bosroger (23200)' => '23028',
		'Bosset (24130)' => '24051',
		'Bossugan (33350)' => '33064',
		'Bostens (40090)' => '40050',
		'Boucau (64340)' => '64140',
		'Boudy-de-Beauregard (47290)' => '47033',
		'Boueilh-Boueilho-Lasque (64330)' => '64141',
		'Bouëx (16410)' => '16055',
		'Bougarber (64230)' => '64142',
		'Bouglon (47250)' => '47034',
		'Bougneau (17800)' => '17056',
		'Bougon (79800)' => '79042',
		'Bougue (40090)' => '40051',
		'Bouhet (17540)' => '17057',
		'Bouillac (24480)' => '24052',
		'Bouillé-Loretz (79290)' => '79043',
		'Bouillé-Saint-Paul (79290)' => '79044',
		'Bouillon (64410)' => '64143',
		'Bouin (79110)' => '79045',
		'Boulazac Isle Manoire (24750)' => '24053',
		'Bouliac (33270)' => '33065',
		'Boumourt (64370)' => '64144',
		'Bouniagues (24560)' => '24054',
		'Bourcefranc-le-Chapus (17560)' => '17058',
		'Bourdalat (40190)' => '40052',
		'Bourdeilles (24310)' => '24055',
		'Bourdelles (33190)' => '33066',
		'Bourdettes (64800)' => '64145',
		'Bouresse (86410)' => '86034',
		'Bourg (33710)' => '33067',
		'Bourg-Archambault (86390)' => '86035',
		'Bourg-Charente (16200)' => '16056',
		'Bourg-des-Maisons (24320)' => '24057',
		'Bourg-du-Bost (24600)' => '24058',
		'Bourganeuf (23400)' => '23030',
		'Bourgnac (24400)' => '24059',
		'Bourgneuf (17220)' => '17059',
		'Bourgougnague (47410)' => '47035',
		'Bourideys (33113)' => '33068',
		'Bourlens (47370)' => '47036',
		'Bournand (86120)' => '86036',
		'Bournel (47210)' => '47037',
		'Bourniquel (24150)' => '24060',
		'Bournos (64450)' => '64146',
		'Bourran (47320)' => '47038',
		'Bourriot-Bergonce (40120)' => '40053',
		'Bourrou (24110)' => '24061',
		'Boussac (23600)' => '23031',
		'Boussac-Bourg (23600)' => '23032',
		'Boussais (79600)' => '79047',
		'Boussès (47420)' => '47039',
		'Bouteilles-Saint-Sébastien (24320)' => '24062',
		'Boutenac-Touvent (17120)' => '17060',
		'Bouteville (16120)' => '16057',
		'Boutiers-Saint-Trojan (16100)' => '16058',
		'Bouzic (24250)' => '24063',
		'Brach (33480)' => '33070',
		'Bran (17210)' => '17061',
		'Branceilles (19500)' => '19029',
		'Branne (33420)' => '33071',
		'Brannens (33124)' => '33072',
		'Brantôme en Périgord (24310)' => '24064',
		'Brassempouy (40330)' => '40054',
		'Braud-et-Saint-Louis (33820)' => '33073',
		'Brax (47310)' => '47040',
		'Bresdon (17490)' => '17062',
		'Bressuire (79300)' => '79049',
		'Bretagne-de-Marsan (40280)' => '40055',
		'Bretignolles (79140)' => '79050',
		'Brettes (16240)' => '16059',
		'Breuil-la-Réorte (17700)' => '17063',
		'Breuil-Magné (17870)' => '17065',
		'Breuilaufa (87300)' => '87022',
		'Breuilh (24380)' => '24065',
		'Breuillet (17920)' => '17064',
		'Bréville (16370)' => '16060',
		'Brie (16590)' => '16061',
		'Brie (79100)' => '79054',
		'Brie-sous-Archiac (17520)' => '17066',
		'Brie-sous-Barbezieux (16300)' => '16062',
		'Brie-sous-Chalais (16210)' => '16063',
		'Brie-sous-Matha (17160)' => '17067',
		'Brie-sous-Mortagne (17120)' => '17068',
		'Brieuil-sur-Chizé (79170)' => '79055',
		'Brignac-la-Plaine (19310)' => '19030',
		'Brigueil-le-Chantre (86290)' => '86037',
		'Brigueuil (16420)' => '16064',
		'Brillac (16500)' => '16065',
		'Brion (86160)' => '86038',
		'Brion-près-Thouet (79290)' => '79056',
		'Brioux-sur-Boutonne (79170)' => '79057',
		'Briscous (64240)' => '64147',
		'Brive-la-Gaillarde (19100)' => '19031',
		'Brives-sur-Charente (17800)' => '17069',
		'Brivezac (19120)' => '19032',
		'Brizambourg (17770)' => '17070',
		'Brocas (40420)' => '40056',
		'Brossac (16480)' => '16066',
		'Brouchaud (24210)' => '24066',
		'Brouqueyran (33124)' => '33074',
		'Brousse (23700)' => '23034',
		'Bruch (47130)' => '47041',
		'Bruges (33520)' => '33075',
		'Bruges-Capbis-Mifaget (64800)' => '64148',
		'Brugnac (47260)' => '47042',
		'Brûlain (79230)' => '79058',
		'Brux (86510)' => '86039',
		'Buanes (40320)' => '40057',
		'Budelière (23170)' => '23035',
		'Budos (33720)' => '33076',
		'Bugeat (19170)' => '19033',
		'Bugnein (64190)' => '64149',
		'Bujaleuf (87460)' => '87024',
		'Bunus (64120)' => '64150',
		'Bunzac (16110)' => '16067',
		'Burgaronne (64390)' => '64151',
		'Burgnac (87800)' => '87025',
		'Burie (17770)' => '17072',
		'Buros (64160)' => '64152',
		'Burosse-Mendousse (64330)' => '64153',
		'Bussac (24350)' => '24069',
		'Bussac-Forêt (17210)' => '17074',
		'Bussac-sur-Charente (17100)' => '17073',
		'Busserolles (24360)' => '24070',
		'Bussière-Badil (24360)' => '24071',
		'Bussière-Dunoise (23320)' => '23036',
		'Bussière-Galant (87230)' => '87027',
		'Bussière-Nouvelle (23700)' => '23037',
		'Bussière-Poitevine (87320)' => '87028',
		'Bussière-Saint-Georges (23600)' => '23038',
		'Bussunarits-Sarrasquette (64220)' => '64154',
		'Bustince-Iriberry (64220)' => '64155',
		'Buxerolles (86180)' => '86041',
		'Buxeuil (37160)' => '86042',
		'Buzet-sur-Baïse (47160)' => '47043',
		'Buziet (64680)' => '64156',
		'Buzy (64260)' => '64157',
		'Cabanac-et-Villagrains (33650)' => '33077',
		'Cabara (33420)' => '33078',
		'Cabariot (17430)' => '17075',
		'Cabidos (64410)' => '64158',
		'Cachen (40120)' => '40058',
		'Cadarsac (33750)' => '33079',
		'Cadaujac (33140)' => '33080',
		'Cadillac (33410)' => '33081',
		'Cadillac-en-Fronsadais (33240)' => '33082',
		'Cadillon (64330)' => '64159',
		'Cagnotte (40300)' => '40059',
		'Cahuzac (47330)' => '47044',
		'Calès (24150)' => '24073',
		'Calignac (47600)' => '47045',
		'Callen (40430)' => '40060',
		'Calonges (47430)' => '47046',
		'Calviac-en-Périgord (24370)' => '24074',
		'Camarsac (33750)' => '33083',
		'Cambes (33880)' => '33084',
		'Cambes (47350)' => '47047',
		'Camblanes-et-Meynac (33360)' => '33085',
		'Cambo-les-Bains (64250)' => '64160',
		'Came (64520)' => '64161',
		'Camiac-et-Saint-Denis (33420)' => '33086',
		'Camiran (33190)' => '33087',
		'Camou-Cihigue (64470)' => '64162',
		'Campagnac-lès-Quercy (24550)' => '24075',
		'Campagne (24260)' => '24076',
		'Campagne (40090)' => '40061',
		'Campet-et-Lamolère (40090)' => '40062',
		'Camps-Saint-Mathurin-Léobazel (19430)' => '19034',
		'Camps-sur-l\'Isle (33660)' => '33088',
		'Campsegret (24140)' => '24077',
		'Campugnan (33390)' => '33089',
		'Cancon (47290)' => '47048',
		'Candresse (40180)' => '40063',
		'Canéjan (33610)' => '33090',
		'Canenx-et-Réaut (40090)' => '40064',
		'Cantenac (33460)' => '33091',
		'Cantillac (24530)' => '24079',
		'Cantois (33760)' => '33092',
		'Capbreton (40130)' => '40065',
		'Capdrot (24540)' => '24080',
		'Capian (33550)' => '33093',
		'Caplong (33220)' => '33094',
		'Captieux (33840)' => '33095',
		'Carbon-Blanc (33560)' => '33096',
		'Carcans (33121)' => '33097',
		'Carcarès-Sainte-Croix (40400)' => '40066',
		'Carcen-Ponson (40400)' => '40067',
		'Cardan (33410)' => '33098',
		'Cardesse (64360)' => '64165',
		'Carignan-de-Bordeaux (33360)' => '33099',
		'Carlux (24370)' => '24081',
		'Caro (64220)' => '64166',
		'Carrère (64160)' => '64167',
		'Carresse-Cassaber (64270)' => '64168',
		'Cars (33390)' => '33100',
		'Carsac-Aillac (24200)' => '24082',
		'Carsac-de-Gurson (24610)' => '24083',
		'Cartelègue (33390)' => '33101',
		'Carves (24170)' => '24084',
		'Cassen (40380)' => '40068',
		'Casseneuil (47440)' => '47049',
		'Casseuil (33190)' => '33102',
		'Cassignas (47340)' => '47050',
		'Castagnède (64270)' => '64170',
		'Castaignos-Souslens (40700)' => '40069',
		'Castandet (40270)' => '40070',
		'Casteide-Cami (64170)' => '64171',
		'Casteide-Candau (64370)' => '64172',
		'Casteide-Doat (64460)' => '64173',
		'Castel-Sarrazin (40330)' => '40074',
		'Castelculier (47240)' => '47051',
		'Casteljaloux (47700)' => '47052',
		'Castella (47340)' => '47053',
		'Castelmoron-d\'Albret (33540)' => '33103',
		'Castelmoron-sur-Lot (47260)' => '47054',
		'Castelnau-Chalosse (40360)' => '40071',
		'Castelnau-de-Médoc (33480)' => '33104',
		'Castelnau-sur-Gupie (47180)' => '47056',
		'Castelnau-Tursan (40320)' => '40072',
		'Castelnaud-de-Gratecambe (47290)' => '47055',
		'Castelnaud-la-Chapelle (24250)' => '24086',
		'Castelner (40700)' => '40073',
		'Castels (24220)' => '24087',
		'Castelviel (33540)' => '33105',
		'Castéra-Loubix (64460)' => '64174',
		'Castet (64260)' => '64175',
		'Castetbon (64190)' => '64176',
		'Castétis (64300)' => '64177',
		'Castetnau-Camblong (64190)' => '64178',
		'Castetner (64300)' => '64179',
		'Castetpugon (64330)' => '64180',
		'Castets (40260)' => '40075',
		'Castets-en-Dorthe (33210)' => '33106',
		'Castillon (Canton d\'Arthez-de-Béarn) (64370)' => '64181',
		'Castillon (Canton de Lembeye) (64350)' => '64182',
		'Castillon-de-Castets (33210)' => '33107',
		'Castillon-la-Bataille (33350)' => '33108',
		'Castillonnès (47330)' => '47057',
		'Castres-Gironde (33640)' => '33109',
		'Caubeyres (47160)' => '47058',
		'Caubios-Loos (64230)' => '64183',
		'Caubon-Saint-Sauveur (47120)' => '47059',
		'Caudecoste (47220)' => '47060',
		'Caudrot (33490)' => '33111',
		'Caumont (33540)' => '33112',
		'Caumont-sur-Garonne (47430)' => '47061',
		'Cauna (40500)' => '40076',
		'Caunay (79190)' => '79060',
		'Cauneille (40300)' => '40077',
		'Caupenne (40250)' => '40078',
		'Cause-de-Clérans (24150)' => '24088',
		'Cauvignac (33690)' => '33113',
		'Cauzac (47470)' => '47062',
		'Cavarc (47330)' => '47063',
		'Cavignac (33620)' => '33114',
		'Cazalis (33113)' => '33115',
		'Cazalis (40700)' => '40079',
		'Cazats (33430)' => '33116',
		'Cazaugitat (33790)' => '33117',
		'Cazères-sur-l\'Adour (40270)' => '40080',
		'Cazideroque (47370)' => '47064',
		'Cazoulès (24370)' => '24089',
		'Ceaux-en-Couhé (86700)' => '86043',
		'Ceaux-en-Loudun (86200)' => '86044',
		'Celle-Lévescault (86600)' => '86045',
		'Cellefrouin (16260)' => '16068',
		'Celles (17520)' => '17076',
		'Celles (24600)' => '24090',
		'Celles-sur-Belle (79370)' => '79061',
		'Cellettes (16230)' => '16069',
		'Cénac (33360)' => '33118',
		'Cénac-et-Saint-Julien (24250)' => '24091',
		'Cendrieux (24380)' => '24092',
		'Cenon (33150)' => '33119',
		'Cenon-sur-Vienne (86530)' => '86046',
		'Cercles (24320)' => '24093',
		'Cercoux (17270)' => '17077',
		'Cère (40090)' => '40081',
		'Cerizay (79140)' => '79062',
		'Cernay (86140)' => '86047',
		'Cérons (33720)' => '33120',
		'Cersay (79290)' => '79063',
		'Cescau (64170)' => '64184',
		'Cessac (33760)' => '33121',
		'Cestas (33610)' => '33122',
		'Cette-Eygun (64490)' => '64185',
		'Ceyroux (23210)' => '23042',
		'Cézac (33620)' => '33123',
		'Chabanais (16150)' => '16070',
		'Chabournay (86380)' => '86048',
		'Chabrac (16150)' => '16071',
		'Chabrignac (19350)' => '19035',
		'Chadenac (17800)' => '17078',
		'Chadurie (16250)' => '16072',
		'Chail (79500)' => '79064',
		'Chaillac-sur-Vienne (87200)' => '87030',
		'Chaillevette (17890)' => '17079',
		'Chalagnac (24380)' => '24094',
		'Chalais (16210)' => '16073',
		'Chalais (24800)' => '24095',
		'Chalais (86200)' => '86049',
		'Chalandray (86190)' => '86050',
		'Challignac (16300)' => '16074',
		'Châlus (87230)' => '87032',
		'Chamadelle (33230)' => '33124',
		'Chamberaud (23480)' => '23043',
		'Chamberet (19370)' => '19036',
		'Chambon (17290)' => '17080',
		'Chambon-Sainte-Croix (23220)' => '23044',
		'Chambon-sur-Voueize (23170)' => '23045',
		'Chambonchard (23110)' => '23046',
		'Chamborand (23240)' => '23047',
		'Chamboret (87140)' => '87033',
		'Chamboulive (19450)' => '19037',
		'Chameyrat (19330)' => '19038',
		'Chamouillac (17130)' => '17081',
		'Champagnac (17500)' => '17082',
		'Champagnac-de-Belair (24530)' => '24096',
		'Champagnac-la-Noaille (19320)' => '19039',
		'Champagnac-la-Prune (19320)' => '19040',
		'Champagnac-la-Rivière (87150)' => '87034',
		'Champagnat (23190)' => '23048',
		'Champagne (17620)' => '17083',
		'Champagne-et-Fontaine (24320)' => '24097',
		'Champagné-le-Sec (86510)' => '86051',
		'Champagne-Mouton (16350)' => '16076',
		'Champagné-Saint-Hilaire (86160)' => '86052',
		'Champagne-Vigny (16250)' => '16075',
		'Champagnolles (17240)' => '17084',
		'Champcevinel (24750)' => '24098',
		'Champdeniers-Saint-Denis (79220)' => '79066',
		'Champdolent (17430)' => '17085',
		'Champeaux-et-la-Chapelle-Pommier (24340)' => '24099',
		'Champigny-le-Sec (86170)' => '86053',
		'Champmillon (16290)' => '16077',
		'Champnétery (87400)' => '87035',
		'Champniers (16430)' => '16078',
		'Champniers (86400)' => '86054',
		'Champniers-et-Reilhac (24360)' => '24100',
		'Champs-Romain (24470)' => '24101',
		'Champsac (87230)' => '87036',
		'Champsanglard (23220)' => '23049',
		'Chanac-les-Mines (19150)' => '19041',
		'Chancelade (24650)' => '24102',
		'Chaniers (17610)' => '17086',
		'Chantecorps (79340)' => '79068',
		'Chanteix (19330)' => '19042',
		'Chanteloup (79320)' => '79069',
		'Chantemerle-sur-la-Soie (17380)' => '17087',
		'Chantérac (24190)' => '24104',
		'Chantillac (16360)' => '16079',
		'Chapdeuil (24320)' => '24105',
		'Chapelle-Spinasse (19300)' => '19046',
		'Chapelle-Viviers (86300)' => '86059',
		'Chaptelat (87270)' => '87038',
		'Chard (23700)' => '23053',
		'Charmé (16140)' => '16083',
		'Charrais (86170)' => '86060',
		'Charras (16380)' => '16084',
		'Charre (64190)' => '64186',
		'Charritte-de-Bas (64130)' => '64187',
		'Charron (17230)' => '17091',
		'Charron (23700)' => '23054',
		'Charroux (86250)' => '86061',
		'Chartrier-Ferrière (19600)' => '19047',
		'Chartuzac (17130)' => '17092',
		'Chassaignes (24600)' => '24114',
		'Chasseneuil-du-Poitou (86360)' => '86062',
		'Chasseneuil-sur-Bonnieure (16260)' => '16085',
		'Chassenon (16150)' => '16086',
		'Chassiecq (16350)' => '16087',
		'Chassors (16200)' => '16088',
		'Chasteaux (19600)' => '19049',
		'Chatain (86250)' => '86063',
		'Château-Chervix (87380)' => '87039',
		'Château-Garnier (86350)' => '86064',
		'Château-l\'Évêque (24460)' => '24115',
		'Château-Larcher (86370)' => '86065',
		'Châteaubernard (16100)' => '16089',
		'Châteauneuf-la-Forêt (87130)' => '87040',
		'Châteauneuf-sur-Charente (16120)' => '16090',
		'Châteauponsac (87290)' => '87041',
		'Châtelaillon-Plage (17340)' => '17094',
		'Châtelard (23700)' => '23055',
		'Châtellerault (86100)' => '86066',
		'Châtelus-le-Marcheix (23430)' => '23056',
		'Châtelus-Malvaleix (23270)' => '23057',
		'Chatenet (17210)' => '17095',
		'Châtignac (16480)' => '16091',
		'Châtillon (86700)' => '86067',
		'Châtillon-sur-Thouet (79200)' => '79080',
		'Châtres (24120)' => '24116',
		'Chauffour-sur-Vell (19500)' => '19050',
		'Chaumeil (19390)' => '19051',
		'Chaunac (17130)' => '17096',
		'Chaunay (86510)' => '86068',
		'Chauray (79180)' => '79081',
		'Chauvigny (86300)' => '86070',
		'Chavagnac (24120)' => '24117',
		'Chavanac (19290)' => '19052',
		'Chavanat (23250)' => '23060',
		'Chaveroche (19200)' => '19053',
		'Chazelles (16380)' => '16093',
		'Chef-Boutonne (79110)' => '79083',
		'Cheissoux (87460)' => '87043',
		'Chenac-Saint-Seurin-d\'Uzet (17120)' => '17098',
		'Chenailler-Mascheix (19120)' => '19054',
		'Chenay (79120)' => '79084',
		'Cheneché (86380)' => '86071',
		'Chénérailles (23130)' => '23061',
		'Chenevelles (86450)' => '86072',
		'Chéniers (23220)' => '23062',
		'Chenommet (16460)' => '16094',
		'Chenon (16460)' => '16095',
		'Chepniers (17210)' => '17099',
		'Chérac (17610)' => '17100',
		'Chéraute (64130)' => '64188',
		'Cherbonnières (17470)' => '17101',
		'Chérigné (79170)' => '79085',
		'Chermignac (17460)' => '17102',
		'Chéronnac (87600)' => '87044',
		'Cherval (24320)' => '24119',
		'Cherveix-Cubas (24390)' => '24120',
		'Cherves (86170)' => '86073',
		'Cherves-Châtelars (16310)' => '16096',
		'Cherves-Richemont (16370)' => '16097',
		'Chervettes (17380)' => '17103',
		'Cherveux (79410)' => '79086',
		'Chevanceaux (17210)' => '17104',
		'Chey (79120)' => '79087',
		'Chiché (79350)' => '79088',
		'Chillac (16480)' => '16099',
		'Chirac (16150)' => '16100',
		'Chirac-Bellevue (19160)' => '19055',
		'Chiré-en-Montreuil (86190)' => '86074',
		'Chives (17510)' => '17105',
		'Chizé (79170)' => '79090',
		'Chouppes (86110)' => '86075',
		'Chourgnac (24640)' => '24121',
		'Ciboure (64500)' => '64189',
		'Cierzac (17520)' => '17106',
		'Cieux (87520)' => '87045',
		'Ciré-d\'Aunis (17290)' => '17107',
		'Cirières (79140)' => '79091',
		'Cissac-Médoc (33250)' => '33125',
		'Cissé (86170)' => '86076',
		'Civaux (86320)' => '86077',
		'Civrac-de-Blaye (33920)' => '33126',
		'Civrac-en-Médoc (33340)' => '33128',
		'Civrac-sur-Dordogne (33350)' => '33127',
		'Civray (86400)' => '86078',
		'Cladech (24170)' => '24122',
		'Clairac (47320)' => '47065',
		'Clairavaux (23500)' => '23063',
		'Claix (16440)' => '16101',
		'Clam (17500)' => '17108',
		'Claracq (64330)' => '64190',
		'Classun (40320)' => '40082',
		'Clavé (79420)' => '79092',
		'Clavette (17220)' => '17109',
		'Clèdes (40320)' => '40083',
		'Clérac (17270)' => '17110',
		'Clergoux (19320)' => '19056',
		'Clermont (40180)' => '40084',
		'Clermont-d\'Excideuil (24160)' => '24124',
		'Clermont-de-Beauregard (24140)' => '24123',
		'Clermont-Dessous (47130)' => '47066',
		'Clermont-Soubiran (47270)' => '47067',
		'Clessé (79350)' => '79094',
		'Cleyrac (33540)' => '33129',
		'Clion (17240)' => '17111',
		'Cloué (86600)' => '86080',
		'Clugnat (23270)' => '23064',
		'Clussais-la-Pommeraie (79190)' => '79095',
		'Coarraze (64800)' => '64191',
		'Cocumont (47250)' => '47068',
		'Cognac (16100)' => '16102',
		'Cognac-la-Forêt (87310)' => '87046',
		'Coimères (33210)' => '33130',
		'Coirac (33540)' => '33131',
		'Coivert (17330)' => '17114',
		'Colayrac-Saint-Cirq (47450)' => '47069',
		'Collonges-la-Rouge (19500)' => '19057',
		'Colombier (24560)' => '24126',
		'Colombiers (17460)' => '17115',
		'Colombiers (86490)' => '86081',
		'Colondannes (23800)' => '23065',
		'Coly (24120)' => '24127',
		'Comberanche-et-Épeluche (24600)' => '24128',
		'Combiers (16320)' => '16103',
		'Combrand (79140)' => '79096',
		'Combressol (19250)' => '19058',
		'Commensacq (40210)' => '40085',
		'Compreignac (87140)' => '87047',
		'Comps (33710)' => '33132',
		'Concèze (19350)' => '19059',
		'Conchez-de-Béarn (64330)' => '64192',
		'Condac (16700)' => '16104',
		'Condat-sur-Ganaveix (19140)' => '19060',
		'Condat-sur-Trincou (24530)' => '24129',
		'Condat-sur-Vézère (24570)' => '24130',
		'Condat-sur-Vienne (87920)' => '87048',
		'Condéon (16360)' => '16105',
		'Condezaygues (47500)' => '47070',
		'Confolens (16500)' => '16106',
		'Confolent-Port-Dieu (19200)' => '19167',
		'Conne-de-Labarde (24560)' => '24132',
		'Connezac (24300)' => '24131',
		'Consac (17150)' => '17116',
		'Contré (17470)' => '17117',
		'Corbère-Abères (64350)' => '64193',
		'Corgnac-sur-l\'Isle (24800)' => '24134',
		'Corignac (17130)' => '17118',
		'Corme-Écluse (17600)' => '17119',
		'Corme-Royal (17600)' => '17120',
		'Cornil (19150)' => '19061',
		'Cornille (24750)' => '24135',
		'Corrèze (19800)' => '19062',
		'Coslédaà-Lube-Boast (64160)' => '64194',
		'Cosnac (19360)' => '19063',
		'Coubeyrac (33890)' => '33133',
		'Coubjours (24390)' => '24136',
		'Coublucq (64410)' => '64195',
		'Coudures (40500)' => '40086',
		'Couffy-sur-Sarsonne (19340)' => '19064',
		'Couhé (86700)' => '86082',
		'Coulaures (24420)' => '24137',
		'Coulgens (16560)' => '16107',
		'Coulombiers (86600)' => '86083',
		'Coulon (79510)' => '79100',
		'Coulonges (16330)' => '16108',
		'Coulonges (17800)' => '17122',
		'Coulonges (86290)' => '86084',
		'Coulonges-sur-l\'Autize (79160)' => '79101',
		'Coulonges-Thouarsais (79330)' => '79102',
		'Coulounieix-Chamiers (24660)' => '24138',
		'Coulx (47260)' => '47071',
		'Couquèques (33340)' => '33134',
		'Courant (17330)' => '17124',
		'Courbiac (47370)' => '47072',
		'Courbillac (16200)' => '16109',
		'Courcelles (17400)' => '17125',
		'Courcerac (17160)' => '17126',
		'Courcôme (16240)' => '16110',
		'Courçon (17170)' => '17127',
		'Courcoury (17100)' => '17128',
		'Courgeac (16190)' => '16111',
		'Courlac (16210)' => '16112',
		'Courlay (79440)' => '79103',
		'Courpiac (33760)' => '33135',
		'Courpignac (17130)' => '17129',
		'Cours (47360)' => '47073',
		'Cours (79220)' => '79104',
		'Cours-de-Monségur (33580)' => '33136',
		'Cours-de-Pile (24520)' => '24140',
		'Cours-les-Bains (33690)' => '33137',
		'Coursac (24430)' => '24139',
		'Courteix (19340)' => '19065',
		'Coussac-Bonneval (87500)' => '87049',
		'Coussay (86110)' => '86085',
		'Coussay-les-Bois (86270)' => '86086',
		'Couthures-sur-Garonne (47180)' => '47074',
		'Coutières (79340)' => '79105',
		'Coutras (33230)' => '33138',
		'Couture (16460)' => '16114',
		'Couture-d\'Argenson (79110)' => '79106',
		'Coutures (24320)' => '24141',
		'Coutures (33580)' => '33139',
		'Coux (17130)' => '17130',
		'Coux et Bigaroque-Mouzens (24220)' => '24142',
		'Couze-et-Saint-Front (24150)' => '24143',
		'Couzeix (87270)' => '87050',
		'Cozes (17120)' => '17131',
		'Cramchaban (17170)' => '17132',
		'Craon (86110)' => '86087',
		'Cravans (17260)' => '17133',
		'Crazannes (17350)' => '17134',
		'Créon (33670)' => '33140',
		'Créon-d\'Armagnac (40240)' => '40087',
		'Cressac-Saint-Genis (16250)' => '16115',
		'Cressat (23140)' => '23068',
		'Cressé (17160)' => '17135',
		'Creyssac (24350)' => '24144',
		'Creysse (24100)' => '24145',
		'Creyssensac-et-Pissot (24380)' => '24146',
		'Crézières (79110)' => '79107',
		'Criteuil-la-Magdeleine (16300)' => '16116',
		'Crocq (23260)' => '23069',
		'Croignon (33750)' => '33141',
		'Croix-Chapeau (17220)' => '17136',
		'Cromac (87160)' => '87053',
		'Crouseilles (64350)' => '64196',
		'Croutelle (86240)' => '86088',
		'Crozant (23160)' => '23070',
		'Croze (23500)' => '23071',
		'Cubjac (24640)' => '24147',
		'Cublac (19520)' => '19066',
		'Cubnezais (33620)' => '33142',
		'Cubzac-les-Ponts (33240)' => '33143',
		'Cudos (33430)' => '33144',
		'Cuhon (86110)' => '86089',
		'Cunèges (24240)' => '24148',
		'Cuq (47220)' => '47076',
		'Cuqueron (64360)' => '64197',
		'Curac (16210)' => '16117',
		'Curçay-sur-Dive (86120)' => '86090',
		'Curemonte (19500)' => '19067',
		'Cursan (33670)' => '33145',
		'Curzay-sur-Vonne (86600)' => '86091',
		'Cussac (87150)' => '87054',
		'Cussac-Fort-Médoc (33460)' => '33146',
		'Cuzorn (47500)' => '47077',
		'Daglan (24250)' => '24150',
		'Daignac (33420)' => '33147',
		'Damazan (47160)' => '47078',
		'Dampierre-sur-Boutonne (17470)' => '17138',
		'Dampniat (19360)' => '19068',
		'Dangé-Saint-Romain (86220)' => '86092',
		'Darazac (19220)' => '19069',
		'Dardenac (33420)' => '33148',
		'Darnac (87320)' => '87055',
		'Darnets (19300)' => '19070',
		'Daubèze (33540)' => '33149',
		'Dausse (47140)' => '47079',
		'Davignac (19250)' => '19071',
		'Dax (40100)' => '40088',
		'Denguin (64230)' => '64198',
		'Dercé (86420)' => '86093',
		'Deviat (16190)' => '16118',
		'Dévillac (47210)' => '47080',
		'Dienné (86410)' => '86094',
		'Dieulivol (33580)' => '33150',
		'Dignac (16410)' => '16119',
		'Dinsac (87210)' => '87056',
		'Dirac (16410)' => '16120',
		'Dissay (86130)' => '86095',
		'Diusse (64330)' => '64199',
		'Doazit (40700)' => '40089',
		'Doazon (64370)' => '64200',
		'Doeuil-sur-le-Mignon (17330)' => '17139',
		'Dognen (64190)' => '64201',
		'Doissat (24170)' => '24151',
		'Dolmayrac (47110)' => '47081',
		'Dolus-d\'Oléron (17550)' => '17140',
		'Domeyrot (23140)' => '23072',
		'Domezain-Berraute (64120)' => '64202',
		'Domme (24250)' => '24152',
		'Dompierre-les-Églises (87190)' => '87057',
		'Dompierre-sur-Charente (17610)' => '17141',
		'Dompierre-sur-Mer (17139)' => '17142',
		'Domps (87120)' => '87058',
		'Dondas (47470)' => '47082',
		'Donnezac (33860)' => '33151',
		'Dontreix (23700)' => '23073',
		'Donzac (33410)' => '33152',
		'Donzacq (40360)' => '40090',
		'Donzenac (19270)' => '19072',
		'Douchapt (24350)' => '24154',
		'Doudrac (47210)' => '47083',
		'Doulezon (33350)' => '33153',
		'Doumy (64450)' => '64203',
		'Dournazac (87230)' => '87060',
		'Doussay (86140)' => '86096',
		'Douville (24140)' => '24155',
		'Doux (79390)' => '79108',
		'Douzains (47330)' => '47084',
		'Douzat (16290)' => '16121',
		'Douzillac (24190)' => '24157',
		'Droux (87190)' => '87061',
		'Duhort-Bachen (40800)' => '40091',
		'Dumes (40500)' => '40092',
		'Dun-le-Palestel (23800)' => '23075',
		'Durance (47420)' => '47085',
		'Duras (47120)' => '47086',
		'Dussac (24270)' => '24158',
		'Eaux-Bonnes (64440)' => '64204',
		'Ébréon (16140)' => '16122',
		'Échallat (16170)' => '16123',
		'Échebrune (17800)' => '17145',
		'Échillais (17620)' => '17146',
		'Échiré (79410)' => '79109',
		'Échourgnac (24410)' => '24159',
		'Écoyeux (17770)' => '17147',
		'Écuras (16220)' => '16124',
		'Écurat (17810)' => '17148',
		'Édon (16320)' => '16125',
		'Égletons (19300)' => '19073',
		'Église-Neuve-d\'Issac (24400)' => '24161',
		'Église-Neuve-de-Vergt (24380)' => '24160',
		'Empuré (16240)' => '16127',
		'Engayrac (47470)' => '47087',
		'Ensigné (79170)' => '79111',
		'Épannes (79270)' => '79112',
		'Épargnes (17120)' => '17152',
		'Épenède (16490)' => '16128',
		'Éraville (16120)' => '16129',
		'Escalans (40310)' => '40093',
		'Escassefort (47350)' => '47088',
		'Escaudes (33840)' => '33155',
		'Escaunets (65500)' => '65160',
		'Esclottes (47120)' => '47089',
		'Escoire (24420)' => '24162',
		'Escos (64270)' => '64205',
		'Escot (64490)' => '64206',
		'Escou (64870)' => '64207',
		'Escoubès (64160)' => '64208',
		'Escource (40210)' => '40094',
		'Escoussans (33760)' => '33156',
		'Escout (64870)' => '64209',
		'Escurès (64350)' => '64210',
		'Eslourenties-Daban (64420)' => '64211',
		'Esnandes (17137)' => '17153',
		'Espagnac (19150)' => '19075',
		'Espartignac (19140)' => '19076',
		'Espéchède (64160)' => '64212',
		'Espelette (64250)' => '64213',
		'Espès-Undurein (64130)' => '64214',
		'Espiens (47600)' => '47090',
		'Espiet (33420)' => '33157',
		'Espiute (64390)' => '64215',
		'Espoey (64420)' => '64216',
		'Esquiule (64400)' => '64217',
		'Esse (16500)' => '16131',
		'Essouvert (17400)' => '17277',
		'Estérençuby (64220)' => '64218',
		'Estialescq (64290)' => '64219',
		'Estibeaux (40290)' => '40095',
		'Estigarde (40240)' => '40096',
		'Estillac (47310)' => '47091',
		'Estivals (19600)' => '19077',
		'Estivaux (19410)' => '19078',
		'Estos (64400)' => '64220',
		'Étagnac (16150)' => '16132',
		'Étaules (17750)' => '17155',
		'Étauliers (33820)' => '33159',
		'Etcharry (64120)' => '64221',
		'Etchebar (64470)' => '64222',
		'Étouars (24360)' => '24163',
		'Étriac (16250)' => '16133',
		'Etsaut (64490)' => '64223',
		'Eugénie-les-Bains (40320)' => '40097',
		'Évaux-les-Bains (23110)' => '23076',
		'Excideuil (24160)' => '24164',
		'Exideuil (16150)' => '16134',
		'Exireuil (79400)' => '79114',
		'Exoudun (79800)' => '79115',
		'Expiremont (17130)' => '17156',
		'Eybouleuf (87400)' => '87062',
		'Eyburie (19140)' => '19079',
		'Eygurande (19340)' => '19080',
		'Eygurande-et-Gardedeuil (24700)' => '24165',
		'Eyjeaux (87220)' => '87063',
		'Eyliac (24330)' => '24166',
		'Eymet (24500)' => '24167',
		'Eymouthiers (16220)' => '16135',
		'Eymoutiers (87120)' => '87064',
		'Eynesse (33220)' => '33160',
		'Eyrans (33390)' => '33161',
		'Eyrein (19800)' => '19081',
		'Eyres-Moncube (40500)' => '40098',
		'Eysines (33320)' => '33162',
		'Eysus (64400)' => '64224',
		'Eyvirat (24460)' => '24170',
		'Eyzerac (24800)' => '24171',
		'Faleyras (33760)' => '33163',
		'Fals (47220)' => '47092',
		'Fanlac (24290)' => '24174',
		'Fargues (33210)' => '33164',
		'Fargues (40500)' => '40099',
		'Fargues-Saint-Hilaire (33370)' => '33165',
		'Fargues-sur-Ourbise (47700)' => '47093',
		'Fauguerolles (47400)' => '47094',
		'Fauillet (47400)' => '47095',
		'Faurilles (24560)' => '24176',
		'Faux (24560)' => '24177',
		'Faux-la-Montagne (23340)' => '23077',
		'Faux-Mazuras (23400)' => '23078',
		'Favars (19330)' => '19082',
		'Faye-l\'Abbesse (79350)' => '79116',
		'Faye-sur-Ardin (79160)' => '79117',
		'Féas (64570)' => '64225',
		'Felletin (23500)' => '23079',
		'Fénery (79450)' => '79118',
		'Féniers (23100)' => '23080',
		'Fenioux (17350)' => '17157',
		'Fenioux (79160)' => '79119',
		'Ferrensac (47330)' => '47096',
		'Ferrières (17170)' => '17158',
		'Festalemps (24410)' => '24178',
		'Feugarolles (47230)' => '47097',
		'Feuillade (16380)' => '16137',
		'Feyt (19340)' => '19083',
		'Feytiat (87220)' => '87065',
		'Fichous-Riumayou (64410)' => '64226',
		'Fieux (47600)' => '47098',
		'Firbeix (24450)' => '24180',
		'Flaugeac (24240)' => '24181',
		'Flaujagues (33350)' => '33168',
		'Flavignac (87230)' => '87066',
		'Flayat (23260)' => '23081',
		'Fléac (16730)' => '16138',
		'Fléac-sur-Seugne (17800)' => '17159',
		'Fleix (86300)' => '86098',
		'Fleurac (16200)' => '16139',
		'Fleurac (24580)' => '24183',
		'Fleurat (23320)' => '23082',
		'Fleuré (86340)' => '86099',
		'Floirac (17120)' => '17160',
		'Floirac (33270)' => '33167',
		'Florimont-Gaumier (24250)' => '24184',
		'Floudès (33190)' => '33169',
		'Folles (87250)' => '87067',
		'Fomperron (79340)' => '79121',
		'Fongrave (47260)' => '47099',
		'Fonroque (24500)' => '24186',
		'Fontaine-Chalendray (17510)' => '17162',
		'Fontaine-le-Comte (86240)' => '86100',
		'Fontaines-d\'Ozillac (17500)' => '17163',
		'Fontanières (23110)' => '23083',
		'Fontclaireau (16230)' => '16140',
		'Fontcouverte (17100)' => '17164',
		'Fontenet (17400)' => '17165',
		'Fontenille (16230)' => '16141',
		'Fontenille-Saint-Martin-d\'Entraigues (79110)' => '79122',
		'Fontet (33190)' => '33170',
		'Forges (17290)' => '17166',
		'Forgès (19380)' => '19084',
		'Fors (79230)' => '79125',
		'Fossemagne (24210)' => '24188',
		'Fossès-et-Baleyssac (33190)' => '33171',
		'Fougueyrolles (33220)' => '24189',
		'Foulayronnes (47510)' => '47100',
		'Fouleix (24380)' => '24190',
		'Fouquebrune (16410)' => '16143',
		'Fouqueure (16140)' => '16144',
		'Fouras (17450)' => '17168',
		'Fourques-sur-Garonne (47200)' => '47101',
		'Fours (33390)' => '33172',
		'Foussignac (16200)' => '16145',
		'Fraisse (24130)' => '24191',
		'Francescas (47600)' => '47102',
		'François (79260)' => '79128',
		'Francs (33570)' => '33173',
		'Fransèches (23480)' => '23086',
		'Fréchou (47600)' => '47103',
		'Frégimont (47360)' => '47104',
		'Frespech (47140)' => '47105',
		'Fresselines (23450)' => '23087',
		'Fressines (79370)' => '79129',
		'Fromental (87250)' => '87068',
		'Fronsac (33126)' => '33174',
		'Frontenac (33760)' => '33175',
		'Frontenay-Rohan-Rohan (79270)' => '79130',
		'Frozes (86190)' => '86102',
		'Fumel (47500)' => '47106',
		'Gaas (40350)' => '40101',
		'Gabarnac (33410)' => '33176',
		'Gabarret (40310)' => '40102',
		'Gabaston (64160)' => '64227',
		'Gabat (64120)' => '64228',
		'Gabillou (24210)' => '24192',
		'Gageac-et-Rouillac (24240)' => '24193',
		'Gaillan-en-Médoc (33340)' => '33177',
		'Gaillères (40090)' => '40103',
		'Gajac (33430)' => '33178',
		'Gajoubert (87330)' => '87069',
		'Galapian (47190)' => '47107',
		'Galgon (33133)' => '33179',
		'Gamarde-les-Bains (40380)' => '40104',
		'Gamarthe (64220)' => '64229',
		'Gan (64290)' => '64230',
		'Gans (33430)' => '33180',
		'Garat (16410)' => '16146',
		'Gardegan-et-Tourtirac (33350)' => '33181',
		'Gardères (65320)' => '65185',
		'Gardes-le-Pontaroux (16320)' => '16147',
		'Gardonne (24680)' => '24194',
		'Garein (40420)' => '40105',
		'Garindein (64130)' => '64231',
		'Garlède-Mondebat (64450)' => '64232',
		'Garlin (64330)' => '64233',
		'Garos (64410)' => '64234',
		'Garrey (40180)' => '40106',
		'Garris (64120)' => '64235',
		'Garrosse (40110)' => '40107',
		'Gartempe (23320)' => '23088',
		'Gastes (40160)' => '40108',
		'Gaugeac (24540)' => '24195',
		'Gaujac (47200)' => '47108',
		'Gaujacq (40330)' => '40109',
		'Gauriac (33710)' => '33182',
		'Gauriaguet (33240)' => '33183',
		'Gavaudun (47150)' => '47109',
		'Gayon (64350)' => '64236',
		'Geaune (40320)' => '40110',
		'Geay (17250)' => '17171',
		'Geay (79330)' => '79131',
		'Gelos (64110)' => '64237',
		'Geloux (40090)' => '40111',
		'Gémozac (17260)' => '17172',
		'Genac-Bignac (16170)' => '16148',
		'Gençay (86160)' => '86103',
		'Générac (33920)' => '33184',
		'Génis (24160)' => '24196',
		'Génissac (33420)' => '33185',
		'Genneton (79150)' => '79132',
		'Genouillac (16270)' => '16149',
		'Genouillac (23350)' => '23089',
		'Genouillé (17430)' => '17174',
		'Genouillé (86250)' => '86104',
		'Gensac (33890)' => '33186',
		'Gensac-la-Pallue (16130)' => '16150',
		'Genté (16130)' => '16151',
		'Gentioux-Pigerolles (23340)' => '23090',
		'Ger (64530)' => '64238',
		'Gerderest (64160)' => '64239',
		'Gère-Bélesten (64260)' => '64240',
		'Germignac (17520)' => '17175',
		'Germond-Rouvre (79220)' => '79133',
		'Géronce (64400)' => '64241',
		'Gestas (64190)' => '64242',
		'Géus-d\'Arzacq (64370)' => '64243',
		'Geüs-d\'Oloron (64400)' => '64244',
		'Gibourne (17160)' => '17176',
		'Gibret (40380)' => '40112',
		'Gimel-les-Cascades (19800)' => '19085',
		'Gimeux (16130)' => '16152',
		'Ginestet (24130)' => '24197',
		'Gioux (23500)' => '23091',
		'Gironde-sur-Dropt (33190)' => '33187',
		'Giscos (33840)' => '33188',
		'Givrezac (17260)' => '17178',
		'Gizay (86340)' => '86105',
		'Glandon (87500)' => '87071',
		'Glanges (87380)' => '87072',
		'Glénay (79330)' => '79134',
		'Glénic (23380)' => '23092',
		'Glénouze (86200)' => '86106',
		'Goès (64400)' => '64245',
		'Gomer (64420)' => '64246',
		'Gond-Pontouvre (16160)' => '16154',
		'Gondeville (16200)' => '16153',
		'Gontaud-de-Nogaret (47400)' => '47110',
		'Goos (40180)' => '40113',
		'Gornac (33540)' => '33189',
		'Gorre (87310)' => '87073',
		'Gotein-Libarrenx (64130)' => '64247',
		'Goualade (33840)' => '33190',
		'Gouex (86320)' => '86107',
		'Goulles (19430)' => '19086',
		'Gourbera (40990)' => '40114',
		'Gourdon-Murat (19170)' => '19087',
		'Gourgé (79200)' => '79135',
		'Gournay-Loizé (79110)' => '79136',
		'Gours (33660)' => '33191',
		'Gourville (16170)' => '16156',
		'Gourvillette (17490)' => '17180',
		'Gousse (40465)' => '40115',
		'Gout-Rossignol (24320)' => '24199',
		'Gouts (40400)' => '40116',
		'Gouzon (23230)' => '23093',
		'Gradignan (33170)' => '33192',
		'Grand-Brassac (24350)' => '24200',
		'Grandjean (17350)' => '17181',
		'Grandsaigne (19300)' => '19088',
		'Granges-d\'Ans (24390)' => '24202',
		'Granges-sur-Lot (47260)' => '47111',
		'Granzay-Gript (79360)' => '79137',
		'Grassac (16380)' => '16158',
		'Grateloup-Saint-Gayrand (47400)' => '47112',
		'Graves-Saint-Amant (16120)' => '16297',
		'Grayan-et-l\'Hôpital (33590)' => '33193',
		'Grayssas (47270)' => '47113',
		'Grenade-sur-l\'Adour (40270)' => '40117',
		'Grézac (17120)' => '17183',
		'Grèzes (24120)' => '24204',
		'Grézet-Cavagnan (47250)' => '47114',
		'Grézillac (33420)' => '33194',
		'Grignols (24110)' => '24205',
		'Grignols (33690)' => '33195',
		'Grives (24170)' => '24206',
		'Groléjac (24250)' => '24207',
		'Gros-Chastang (19320)' => '19089',
		'Grun-Bordas (24380)' => '24208',
		'Guéret (23000)' => '23096',
		'Guérin (47250)' => '47115',
		'Guesnes (86420)' => '86109',
		'Guéthary (64210)' => '64249',
		'Guiche (64520)' => '64250',
		'Guillac (33420)' => '33196',
		'Guillos (33720)' => '33197',
		'Guimps (16300)' => '16160',
		'Guinarthe-Parenties (64390)' => '64251',
		'Guitinières (17500)' => '17187',
		'Guîtres (33230)' => '33198',
		'Guizengeard (16480)' => '16161',
		'Gujan-Mestras (33470)' => '33199',
		'Gumond (19320)' => '19090',
		'Gurat (16320)' => '16162',
		'Gurmençon (64400)' => '64252',
		'Gurs (64190)' => '64253',
		'Habas (40290)' => '40118',
		'Hagetaubin (64370)' => '64254',
		'Hagetmau (40700)' => '40119',
		'Haimps (17160)' => '17188',
		'Haims (86310)' => '86110',
		'Halsou (64480)' => '64255',
		'Hanc (79110)' => '79140',
		'Hasparren (64240)' => '64256',
		'Hastingues (40300)' => '40120',
		'Hauriet (40250)' => '40121',
		'Haut-de-Bosdarros (64800)' => '64257',
		'Haut-Mauco (40280)' => '40122',
		'Hautefage (19400)' => '19091',
		'Hautefage-la-Tour (47340)' => '47117',
		'Hautefaye (24300)' => '24209',
		'Hautefort (24390)' => '24210',
		'Hautesvignes (47400)' => '47118',
		'Haux (33550)' => '33201',
		'Haux (64470)' => '64258',
		'Hélette (64640)' => '64259',
		'Hendaye (64700)' => '64260',
		'Herm (40990)' => '40123',
		'Herré (40310)' => '40124',
		'Herrère (64680)' => '64261',
		'Heugas (40180)' => '40125',
		'Hiers-Brouage (17320)' => '17189',
		'Hiersac (16290)' => '16163',
		'Hiesse (16490)' => '16164',
		'Higuères-Souye (64160)' => '64262',
		'Hinx (40180)' => '40126',
		'Hontanx (40190)' => '40127',
		'Horsarrieu (40700)' => '40128',
		'Hosta (64120)' => '64265',
		'Hostens (33125)' => '33202',
		'Houeillès (47420)' => '47119',
		'Houlette (16200)' => '16165',
		'Hours (64420)' => '64266',
		'Hourtin (33990)' => '33203',
		'Hure (33190)' => '33204',
		'Ibarrolle (64120)' => '64267',
		'Idaux-Mendy (64130)' => '64268',
		'Idron (64320)' => '64269',
		'Igon (64800)' => '64270',
		'Iholdy (64640)' => '64271',
		'Île-d\'Aix (17123)' => '17004',
		'Ilharre (64120)' => '64272',
		'Illats (33720)' => '33205',
		'Ingrandes (86220)' => '86111',
		'Irais (79600)' => '79141',
		'Irissarry (64780)' => '64273',
		'Irouléguy (64220)' => '64274',
		'Isle (87170)' => '87075',
		'Isle-Saint-Georges (33640)' => '33206',
		'Ispoure (64220)' => '64275',
		'Issac (24400)' => '24211',
		'Issigeac (24560)' => '24212',
		'Issor (64570)' => '64276',
		'Issoudun-Létrieix (23130)' => '23097',
		'Isturits (64240)' => '64277',
		'Iteuil (86240)' => '86113',
		'Itxassou (64250)' => '64279',
		'Izeste (64260)' => '64280',
		'Izon (33450)' => '33207',
		'Jabreilles-les-Bordes (87370)' => '87076',
		'Jalesches (23270)' => '23098',
		'Janailhac (87800)' => '87077',
		'Janaillat (23250)' => '23099',
		'Jardres (86800)' => '86114',
		'Jarnac (16200)' => '16167',
		'Jarnac-Champagne (17520)' => '17192',
		'Jarnages (23140)' => '23100',
		'Jasses (64190)' => '64281',
		'Jatxou (64480)' => '64282',
		'Jau-Dignac-et-Loirac (33590)' => '33208',
		'Jauldes (16560)' => '16168',
		'Jaunay-Clan (86130)' => '86115',
		'Jaure (24140)' => '24213',
		'Javerdat (87520)' => '87078',
		'Javerlhac-et-la-Chapelle-Saint-Robert (24300)' => '24214',
		'Javrezac (16100)' => '16169',
		'Jaxu (64220)' => '64283',
		'Jayac (24590)' => '24215',
		'Jazeneuil (86600)' => '86116',
		'Jazennes (17260)' => '17196',
		'Jonzac (17500)' => '17197',
		'Josse (40230)' => '40129',
		'Jouac (87890)' => '87080',
		'Jouhet (86500)' => '86117',
		'Jouillat (23220)' => '23101',
		'Jourgnac (87800)' => '87081',
		'Journet (86290)' => '86118',
		'Journiac (24260)' => '24217',
		'Joussé (86350)' => '86119',
		'Jugazan (33420)' => '33209',
		'Jugeals-Nazareth (19500)' => '19093',
		'Juicq (17770)' => '17198',
		'Juignac (16190)' => '16170',
		'Juillac (19350)' => '19094',
		'Juillac (33890)' => '33210',
		'Juillac-le-Coq (16130)' => '16171',
		'Juillé (16230)' => '16173',
		'Juillé (79170)' => '79142',
		'Julienne (16200)' => '16174',
		'Jumilhac-le-Grand (24630)' => '24218',
		'Jurançon (64110)' => '64284',
		'Juscorps (79230)' => '79144',
		'Jusix (47180)' => '47120',
		'Jussas (17130)' => '17199',
		'Juxue (64120)' => '64285',
		'L\'Absie (79240)' => '79001',
		'L\'Église-aux-Bois (19170)' => '19074',
		'L\'Éguille (17600)' => '17151',
		'L\'Hôpital-d\'Orion (64270)' => '64263',
		'L\'Hôpital-Saint-Blaise (64130)' => '64264',
		'L\'Houmeau (17137)' => '17190',
		'L\'Isle-d\'Espagnac (16340)' => '16166',
		'L\'Isle-Jourdain (86150)' => '86112',
		'La Bachellerie (24210)' => '24020',
		'La Barde (17360)' => '17033',
		'La Bastide-Clairence (64240)' => '64289',
		'La Bataille (79110)' => '79027',
		'La Bazeuge (87210)' => '87008',
		'La Boissière-d\'Ans (24640)' => '24047',
		'La Boissière-en-Gâtine (79310)' => '79040',
		'La Brède (33650)' => '33213',
		'La Brée-les-Bains (17840)' => '17486',
		'La Brionne (23000)' => '23033',
		'La Brousse (17160)' => '17071',
		'La Bussière (86310)' => '86040',
		'La Cassagne (24120)' => '24085',
		'La Celle-Dunoise (23800)' => '23039',
		'La Celle-sous-Gouzon (23230)' => '23040',
		'La Cellette (23350)' => '23041',
		'La Chapelle (16140)' => '16081',
		'La Chapelle-Aubareil (24290)' => '24106',
		'La Chapelle-aux-Brocs (19360)' => '19043',
		'La Chapelle-aux-Saints (19120)' => '19044',
		'La Chapelle-Baloue (23160)' => '23050',
		'La Chapelle-Bâton (79220)' => '79070',
		'La Chapelle-Bâton (86250)' => '86055',
		'La Chapelle-Bertrand (79200)' => '79071',
		'La Chapelle-des-Pots (17100)' => '17089',
		'La Chapelle-Faucher (24530)' => '24107',
		'La Chapelle-Gonaguet (24350)' => '24108',
		'La Chapelle-Grésignac (24320)' => '24109',
		'La Chapelle-Montabourlet (24320)' => '24110',
		'La Chapelle-Montbrandeix (87440)' => '87037',
		'La Chapelle-Montmoreau (24300)' => '24111',
		'La Chapelle-Montreuil (86470)' => '86056',
		'La Chapelle-Moulière (86210)' => '86058',
		'La Chapelle-Pouilloux (79190)' => '79074',
		'La Chapelle-Saint-Étienne (79240)' => '79075',
		'La Chapelle-Saint-Géraud (19430)' => '19045',
		'La Chapelle-Saint-Jean (24390)' => '24113',
		'La Chapelle-Saint-Laurent (79430)' => '79076',
		'La Chapelle-Saint-Martial (23250)' => '23051',
		'La Chapelle-Taillefert (23000)' => '23052',
		'La Chapelle-Thireuil (79160)' => '79077',
		'La Chaussade (23200)' => '23059',
		'La Chaussée (86330)' => '86069',
		'La Chèvrerie (16240)' => '16098',
		'La Clisse (17600)' => '17112',
		'La Clotte (17360)' => '17113',
		'La Coquille (24450)' => '24133',
		'La Couarde (79800)' => '79098',
		'La Couarde-sur-Mer (17670)' => '17121',
		'La Couronne (16400)' => '16113',
		'La Courtine (23100)' => '23067',
		'La Crèche (79260)' => '79048',
		'La Croisille-sur-Briance (87130)' => '87051',
		'La Croix-Blanche (47340)' => '47075',
		'La Croix-Comtesse (17330)' => '17137',
		'La Croix-sur-Gartempe (87210)' => '87052',
		'La Dornac (24120)' => '24153',
		'La Douze (24330)' => '24156',
		'La Faye (16700)' => '16136',
		'La Ferrière-Airoux (86160)' => '86097',
		'La Ferrière-en-Parthenay (79390)' => '79120',
		'La Feuillade (24120)' => '24179',
		'La Flotte (17630)' => '17161',
		'La Force (24130)' => '24222',
		'La Forêt-de-Tessé (16240)' => '16142',
		'La Forêt-du-Temple (23360)' => '23084',
		'La Forêt-sur-Sèvre (79380)' => '79123',
		'La Foye-Monjault (79360)' => '79127',
		'La Frédière (17770)' => '17169',
		'La Genétouze (17360)' => '17173',
		'La Geneytouse (87400)' => '87070',
		'La Gonterie-Boulouneix (24310)' => '24198',
		'La Grève-sur-Mignon (17170)' => '17182',
		'La Grimaudière (86330)' => '86108',
		'La Gripperie-Saint-Symphorien (17620)' => '17184',
		'La Jard (17460)' => '17191',
		'La Jarne (17220)' => '17193',
		'La Jarrie (17220)' => '17194',
		'La Jarrie-Audouin (17330)' => '17195',
		'La Jemaye (24410)' => '24216',
		'La Jonchère-Saint-Maurice (87340)' => '87079',
		'La Laigne (17170)' => '17201',
		'La Lande-de-Fronsac (33240)' => '33219',
		'La Magdeleine (16240)' => '16197',
		'La Mazière-aux-Bons-Hommes (23260)' => '23129',
		'La Meyze (87800)' => '87096',
		'La Mothe-Saint-Héray (79800)' => '79184',
		'La Nouaille (23500)' => '23144',
		'La Péruse (16270)' => '16259',
		'La Petite-Boissière (79700)' => '79207',
		'La Peyratte (79200)' => '79208',
		'La Porcherie (87380)' => '87120',
		'La Pouge (23250)' => '23157',
		'La Puye (86260)' => '86202',
		'La Réole (33190)' => '33352',
		'La Réunion (47700)' => '47222',
		'La Rivière (33126)' => '33356',
		'La Roche-Canillac (19320)' => '19174',
		'La Roche-Chalais (24490)' => '24354',
		'La Roche-l\'Abeille (87800)' => '87127',
		'La Roche-Posay (86270)' => '86207',
		'La Roche-Rigault (86200)' => '86079',
		'La Rochebeaucourt-et-Argentine (24340)' => '24353',
		'La Rochefoucauld (16110)' => '16281',
		'La Rochelle (17000)' => '17300',
		'La Rochénard (79270)' => '79229',
		'La Rochette (16110)' => '16282',
		'La Ronde (17170)' => '17303',
		'La Roque-Gageac (24250)' => '24355',
		'La Roquille (33220)' => '33360',
		'La Saunière (23000)' => '23169',
		'La Sauve (33670)' => '33505',
		'La Sauvetat-de-Savères (47270)' => '47289',
		'La Sauvetat-du-Dropt (47800)' => '47290',
		'La Sauvetat-sur-Lède (47150)' => '47291',
		'La Serre-Bussière-Vieille (23190)' => '23172',
		'La Souterraine (23300)' => '23176',
		'La Tâche (16260)' => '16377',
		'La Teste-de-Buch (33260)' => '33529',
		'La Tour-Blanche (24320)' => '24554',
		'La Tremblade (17390)' => '17452',
		'La Trimouille (86290)' => '86273',
		'La Vallée (17250)' => '17455',
		'La Vergne (17400)' => '17465',
		'La Villedieu (17470)' => '17471',
		'La Villedieu (23340)' => '23264',
		'La Villedieu-du-Clain (86340)' => '86290',
		'La Villeneuve (23260)' => '23265',
		'La Villetelle (23260)' => '23266',
		'Laà-Mondrans (64300)' => '64286',
		'Laàs (64390)' => '64287',
		'Labarde (33460)' => '33211',
		'Labastide-Castel-Amouroux (47250)' => '47121',
		'Labastide-Cézéracq (64170)' => '64288',
		'Labastide-Chalosse (40700)' => '40130',
		'Labastide-d\'Armagnac (40240)' => '40131',
		'Labastide-Monréjeau (64170)' => '64290',
		'Labastide-Villefranche (64270)' => '64291',
		'Labatmale (64530)' => '64292',
		'Labatut (40300)' => '40132',
		'Labatut (64460)' => '64293',
		'Labenne (40530)' => '40133',
		'Labescau (33690)' => '33212',
		'Labets-Biscay (64120)' => '64294',
		'Labeyrie (64300)' => '64295',
		'Labouheyre (40210)' => '40134',
		'Labretonie (47350)' => '47122',
		'Labrit (40420)' => '40135',
		'Lacadée (64300)' => '64296',
		'Lacajunte (40320)' => '40136',
		'Lacanau (33680)' => '33214',
		'Lacapelle-Biron (47150)' => '47123',
		'Lacarre (64220)' => '64297',
		'Lacarry-Arhan-Charritte-de-Haut (64470)' => '64298',
		'Lacaussade (47150)' => '47124',
		'Lacelle (19170)' => '19095',
		'Lacépède (47360)' => '47125',
		'Lachaise (16300)' => '16176',
		'Lachapelle (47350)' => '47126',
		'Lacommande (64360)' => '64299',
		'Lacq (64170)' => '64300',
		'Lacquy (40120)' => '40137',
		'Lacrabe (40700)' => '40138',
		'Lacropte (24380)' => '24220',
		'Ladapeyre (23270)' => '23102',
		'Ladaux (33760)' => '33215',
		'Ladignac-le-Long (87500)' => '87082',
		'Ladignac-sur-Rondelles (19150)' => '19096',
		'Ladiville (16120)' => '16177',
		'Lados (33124)' => '33216',
		'Lafage-sur-Sombre (19320)' => '19097',
		'Lafat (23800)' => '23103',
		'Lafitte-sur-Lot (47320)' => '47127',
		'Lafox (47240)' => '47128',
		'Lagarde-Enval (19150)' => '19098',
		'Lagarde-sur-le-Né (16300)' => '16178',
		'Lagarrigue (47190)' => '47129',
		'Lageon (79200)' => '79145',
		'Lagleygeolle (19500)' => '19099',
		'Laglorieuse (40090)' => '40139',
		'Lagor (64150)' => '64301',
		'Lagorce (33230)' => '33218',
		'Lagord (17140)' => '17200',
		'Lagos (64800)' => '64302',
		'Lagrange (40240)' => '40140',
		'Lagraulière (19700)' => '19100',
		'Lagruère (47400)' => '47130',
		'Laguenne (19150)' => '19101',
		'Laguinge-Restoue (64470)' => '64303',
		'Lagupie (47180)' => '47131',
		'Lahonce (64990)' => '64304',
		'Lahontan (64270)' => '64305',
		'Lahosse (40250)' => '40141',
		'Lahourcade (64150)' => '64306',
		'Lalande-de-Pomerol (33500)' => '33222',
		'Lalandusse (47330)' => '47132',
		'Lalinde (24150)' => '24223',
		'Lalongue (64350)' => '64307',
		'Lalonquette (64450)' => '64308',
		'Laluque (40465)' => '40142',
		'Lamarque (33460)' => '33220',
		'Lamayou (64460)' => '64309',
		'Lamazière-Basse (19160)' => '19102',
		'Lamazière-Haute (19340)' => '19103',
		'Lamongerie (19510)' => '19104',
		'Lamontjoie (47310)' => '47133',
		'Lamonzie-Montastruc (24520)' => '24224',
		'Lamonzie-Saint-Martin (24680)' => '24225',
		'Lamothe (40250)' => '40143',
		'Lamothe-Landerron (33190)' => '33221',
		'Lamothe-Montravel (24230)' => '24226',
		'Landerrouat (33790)' => '33223',
		'Landerrouet-sur-Ségur (33540)' => '33224',
		'Landes (17380)' => '17202',
		'Landiras (33720)' => '33225',
		'Landrais (17290)' => '17203',
		'Langoiran (33550)' => '33226',
		'Langon (33210)' => '33227',
		'Lanne-en-Barétous (64570)' => '64310',
		'Lannecaube (64350)' => '64311',
		'Lanneplaà (64300)' => '64312',
		'Lannes (47170)' => '47134',
		'Lanouaille (24270)' => '24227',
		'Lanquais (24150)' => '24228',
		'Lansac (33710)' => '33228',
		'Lantabat (64640)' => '64313',
		'Lanteuil (19190)' => '19105',
		'Lanton (33138)' => '33229',
		'Laparade (47260)' => '47135',
		'Laperche (47800)' => '47136',
		'Lapleau (19550)' => '19106',
		'Laplume (47310)' => '47137',
		'Lapouyade (33620)' => '33230',
		'Laprade (16390)' => '16180',
		'Larbey (40250)' => '40144',
		'Larceveau-Arros-Cibits (64120)' => '64314',
		'Larche (19600)' => '19107',
		'Largeasse (79240)' => '79147',
		'Laroche-près-Feyt (19340)' => '19108',
		'Laroin (64110)' => '64315',
		'Laroque (33410)' => '33231',
		'Laroque-Timbaut (47340)' => '47138',
		'Larrau (64560)' => '64316',
		'Larressore (64480)' => '64317',
		'Larreule (64410)' => '64318',
		'Larribar-Sorhapuru (64120)' => '64319',
		'Larrivière-Saint-Savin (40270)' => '40145',
		'Lartigue (33840)' => '33232',
		'Laruns (64440)' => '64320',
		'Laruscade (33620)' => '33233',
		'Larzac (24170)' => '24230',
		'Lascaux (19130)' => '19109',
		'Lasclaveries (64450)' => '64321',
		'Lasse (64220)' => '64322',
		'Lasserre (47600)' => '47139',
		'Lasserre (64350)' => '64323',
		'Lasseube (64290)' => '64324',
		'Lasseubetat (64290)' => '64325',
		'Lathus-Saint-Rémy (86390)' => '86120',
		'Latillé (86190)' => '86121',
		'Latresne (33360)' => '33234',
		'Latrille (40800)' => '40146',
		'Latronche (19160)' => '19110',
		'Laugnac (47360)' => '47140',
		'Laurède (40250)' => '40147',
		'Lauret (40320)' => '40148',
		'Laurière (87370)' => '87083',
		'Laussou (47150)' => '47141',
		'Lauthiers (86300)' => '86122',
		'Lauzun (47410)' => '47142',
		'Laval-sur-Luzège (19550)' => '19111',
		'Lavalade (24540)' => '24231',
		'Lavardac (47230)' => '47143',
		'Lavaufranche (23600)' => '23104',
		'Lavaur (24550)' => '24232',
		'Lavausseau (86470)' => '86123',
		'Lavaveix-les-Mines (23150)' => '23105',
		'Lavazan (33690)' => '33235',
		'Lavergne (47800)' => '47144',
		'Laveyssière (24130)' => '24233',
		'Lavignac (87230)' => '87084',
		'Lavoux (86800)' => '86124',
		'Lay-Lamidou (64190)' => '64326',
		'Layrac (47390)' => '47145',
		'Le Barp (33114)' => '33029',
		'Le Beugnon (79130)' => '79035',
		'Le Bois-Plage-en-Ré (17580)' => '17051',
		'Le Bouchage (16350)' => '16054',
		'Le Bourdeix (24300)' => '24056',
		'Le Bourdet (79210)' => '79046',
		'Le Bourg-d\'Hem (23220)' => '23029',
		'Le Bouscat (33110)' => '33069',
		'Le Breuil-Bernard (79320)' => '79051',
		'Le Bugue (24260)' => '24067',
		'Le Buis (87140)' => '87023',
		'Le Buisson-de-Cadouin (24480)' => '24068',
		'Le Busseau (79240)' => '79059',
		'Le Chalard (87500)' => '87031',
		'Le Change (24640)' => '24103',
		'Le Chastang (19190)' => '19048',
		'Le Château-d\'Oléron (17480)' => '17093',
		'Le Châtenet-en-Dognon (87400)' => '87042',
		'Le Chauchet (23130)' => '23058',
		'Le Chay (17600)' => '17097',
		'Le Chillou (79600)' => '79089',
		'Le Compas (23700)' => '23066',
		'Le Donzeil (23480)' => '23074',
		'Le Dorat (87210)' => '87059',
		'Le Douhet (17100)' => '17143',
		'Le Fieu (33230)' => '33166',
		'Le Fleix (24130)' => '24182',
		'Le Fouilloux (17270)' => '17167',
		'Le Frêche (40190)' => '40100',
		'Le Gicq (17160)' => '17177',
		'Le Grand-Bourg (23240)' => '23095',
		'Le Grand-Madieu (16450)' => '16157',
		'Le Grand-Village-Plage (17370)' => '17485',
		'Le Gua (17600)' => '17185',
		'Le Gué-d\'Alleré (17540)' => '17186',
		'Le Haillan (33185)' => '33200',
		'Le Jardin (19300)' => '19092',
		'Le Lardin-Saint-Lazare (24570)' => '24229',
		'Le Leuy (40250)' => '40153',
		'Le Lindois (16310)' => '16188',
		'Le Lonzac (19470)' => '19118',
		'Le Mas-d\'Agenais (47430)' => '47159',
		'Le Mas-d\'Artige (23100)' => '23125',
		'Le Monteil-au-Vicomte (23460)' => '23134',
		'Le Mung (17350)' => '17252',
		'Le Nizan (33430)' => '33305',
		'Le Palais-sur-Vienne (87410)' => '87113',
		'Le Passage (47520)' => '47201',
		'Le Pescher (19190)' => '19163',
		'Le Pian-Médoc (33290)' => '33322',
		'Le Pian-sur-Garonne (33490)' => '33323',
		'Le Pin (17210)' => '17276',
		'Le Pin (79140)' => '79210',
		'Le Pizou (24700)' => '24329',
		'Le Porge (33680)' => '33333',
		'Le Pout (33670)' => '33335',
		'Le Puy (33580)' => '33345',
		'Le Retail (79130)' => '79226',
		'Le Rochereau (86170)' => '86208',
		'Le Sen (40420)' => '40297',
		'Le Seure (17770)' => '17426',
		'Le Taillan-Médoc (33320)' => '33519',
		'Le Tallud (79200)' => '79322',
		'Le Tâtre (16360)' => '16380',
		'Le Teich (33470)' => '33527',
		'Le Temple (33680)' => '33528',
		'Le Temple-sur-Lot (47110)' => '47306',
		'Le Thou (17290)' => '17447',
		'Le Tourne (33550)' => '33534',
		'Le Tuzan (33125)' => '33536',
		'Le Vanneau-Irleau (79270)' => '79337',
		'Le Verdon-sur-Mer (33123)' => '33544',
		'Le Vert (79170)' => '79346',
		'Le Vieux-Cérier (16350)' => '16403',
		'Le Vigeant (86150)' => '86289',
		'Le Vigen (87110)' => '87205',
		'Le Vignau (40270)' => '40329',
		'Lecumberry (64220)' => '64327',
		'Lédat (47300)' => '47146',
		'Ledeuix (64400)' => '64328',
		'Lée (64320)' => '64329',
		'Lées-Athas (64490)' => '64330',
		'Lège-Cap-Ferret (33950)' => '33236',
		'Léguillac-de-Cercles (24340)' => '24235',
		'Léguillac-de-l\'Auche (24110)' => '24236',
		'Leigné-les-Bois (86450)' => '86125',
		'Leigné-sur-Usseau (86230)' => '86127',
		'Leignes-sur-Fontaine (86300)' => '86126',
		'Lembeye (64350)' => '64331',
		'Lembras (24100)' => '24237',
		'Lème (64450)' => '64332',
		'Lempzours (24800)' => '24238',
		'Lencloître (86140)' => '86128',
		'Lencouacq (40120)' => '40149',
		'Léogeats (33210)' => '33237',
		'Léognan (33850)' => '33238',
		'Léon (40550)' => '40150',
		'Léoville (17500)' => '17204',
		'Lépaud (23170)' => '23106',
		'Lépinas (23150)' => '23107',
		'Léren (64270)' => '64334',
		'Lerm-et-Musset (33840)' => '33239',
		'Les Adjots (16700)' => '16002',
		'Les Alleuds (79190)' => '79006',
		'Les Angles-sur-Corrèze (19000)' => '19009',
		'Les Artigues-de-Lussac (33570)' => '33014',
		'Les Billanges (87340)' => '87016',
		'Les Billaux (33500)' => '33052',
		'Les Cars (87230)' => '87029',
		'Les Éduts (17510)' => '17149',
		'Les Églises-d\'Argenteuil (17400)' => '17150',
		'Les Églisottes-et-Chalaures (33230)' => '33154',
		'Les Essards (16210)' => '16130',
		'Les Essards (17250)' => '17154',
		'Les Esseintes (33190)' => '33158',
		'Les Eyzies-de-Tayac-Sireuil (24620)' => '24172',
		'Les Farges (24290)' => '24175',
		'Les Forges (79340)' => '79124',
		'Les Fosses (79360)' => '79126',
		'Les Gonds (17100)' => '17179',
		'Les Gours (16140)' => '16155',
		'Les Grands-Chézeaux (87160)' => '87074',
		'Les Graulges (24340)' => '24203',
		'Les Groseillers (79220)' => '79139',
		'Les Lèches (24400)' => '24234',
		'Les Lèves-et-Thoumeyragues (33220)' => '33242',
		'Les Mars (23700)' => '23123',
		'Les Mathes (17570)' => '17225',
		'Les Métairies (16200)' => '16220',
		'Les Nouillers (17380)' => '17266',
		'Les Ormes (86220)' => '86183',
		'Les Peintures (33230)' => '33315',
		'Les Pins (16260)' => '16261',
		'Les Portes-en-Ré (17880)' => '17286',
		'Les Salles-de-Castillon (33350)' => '33499',
		'Les Salles-Lavauguyon (87440)' => '87189',
		'Les Touches-de-Périgny (17160)' => '17451',
		'Les Trois-Moutiers (86120)' => '86274',
		'Lescar (64230)' => '64335',
		'Lescun (64490)' => '64336',
		'Lesgor (40400)' => '40151',
		'Lésignac-Durand (16310)' => '16183',
		'Lésigny (86270)' => '86129',
		'Lesparre-Médoc (33340)' => '33240',
		'Lesperon (40260)' => '40152',
		'Lespielle (64350)' => '64337',
		'Lespourcy (64160)' => '64338',
		'Lessac (16500)' => '16181',
		'Lestards (19170)' => '19112',
		'Lestelle-Bétharram (64800)' => '64339',
		'Lesterps (16420)' => '16182',
		'Lestiac-sur-Garonne (33550)' => '33241',
		'Leugny (86220)' => '86130',
		'Lévignac-de-Guyenne (47120)' => '47147',
		'Lévignacq (40170)' => '40154',
		'Leyrat (23600)' => '23108',
		'Leyritz-Moncassin (47700)' => '47148',
		'Lezay (79120)' => '79148',
		'Lhommaizé (86410)' => '86131',
		'Lhoumois (79390)' => '79149',
		'Libourne (33500)' => '33243',
		'Lichans-Sunhar (64470)' => '64340',
		'Lichères (16460)' => '16184',
		'Lichos (64130)' => '64341',
		'Licq-Athérey (64560)' => '64342',
		'Liginiac (19160)' => '19113',
		'Liglet (86290)' => '86132',
		'Lignan-de-Bazas (33430)' => '33244',
		'Lignan-de-Bordeaux (33360)' => '33245',
		'Lignareix (19200)' => '19114',
		'Ligné (16140)' => '16185',
		'Ligneyrac (19500)' => '19115',
		'Lignières-Sonneville (16130)' => '16186',
		'Ligueux (33220)' => '33246',
		'Ligugé (86240)' => '86133',
		'Limalonges (79190)' => '79150',
		'Limendous (64420)' => '64343',
		'Limeuil (24510)' => '24240',
		'Limeyrat (24210)' => '24241',
		'Limoges (87000)' => '87085',
		'Linard (23220)' => '23109',
		'Linards (87130)' => '87086',
		'Linars (16730)' => '16187',
		'Linazay (86400)' => '86134',
		'Liniers (86800)' => '86135',
		'Linxe (40260)' => '40155',
		'Liorac-sur-Louyre (24520)' => '24242',
		'Liourdres (19120)' => '19116',
		'Lioux-les-Monges (23700)' => '23110',
		'Liposthey (40410)' => '40156',
		'Lisle (24350)' => '24243',
		'Lissac-sur-Couze (19600)' => '19117',
		'Listrac-de-Durèze (33790)' => '33247',
		'Listrac-Médoc (33480)' => '33248',
		'Lit-et-Mixe (40170)' => '40157',
		'Livron (64530)' => '64344',
		'Lizant (86400)' => '86136',
		'Lizières (23240)' => '23111',
		'Lohitzun-Oyhercq (64120)' => '64345',
		'Loire-les-Marais (17870)' => '17205',
		'Loiré-sur-Nie (17470)' => '17206',
		'Loix (17111)' => '17207',
		'Lolme (24540)' => '24244',
		'Lombia (64160)' => '64346',
		'Lonçon (64410)' => '64347',
		'Londigny (16700)' => '16189',
		'Longèves (17230)' => '17208',
		'Longré (16240)' => '16190',
		'Longueville (47200)' => '47150',
		'Lonnes (16230)' => '16191',
		'Lons (64140)' => '64348',
		'Lonzac (17520)' => '17209',
		'Lorignac (17240)' => '17210',
		'Lorigné (79190)' => '79152',
		'Lormont (33310)' => '33249',
		'Losse (40240)' => '40158',
		'Lostanges (19500)' => '19119',
		'Loubejac (24550)' => '24245',
		'Loubens (33190)' => '33250',
		'Loubès-Bernac (47120)' => '47151',
		'Loubieng (64300)' => '64349',
		'Loubigné (79110)' => '79153',
		'Loubillé (79110)' => '79154',
		'Louchats (33125)' => '33251',
		'Loudun (86200)' => '86137',
		'Louer (40380)' => '40159',
		'Lougratte (47290)' => '47152',
		'Louhossoa (64250)' => '64350',
		'Louignac (19310)' => '19120',
		'Louin (79600)' => '79156',
		'Loulay (17330)' => '17211',
		'Loupes (33370)' => '33252',
		'Loupiac (33410)' => '33253',
		'Loupiac-de-la-Réole (33190)' => '33254',
		'Lourdios-Ichère (64570)' => '64351',
		'Lourdoueix-Saint-Pierre (23360)' => '23112',
		'Lourenties (64420)' => '64352',
		'Lourquen (40250)' => '40160',
		'Louvie-Juzon (64260)' => '64353',
		'Louvie-Soubiron (64440)' => '64354',
		'Louvigny (64410)' => '64355',
		'Louzac-Saint-André (16100)' => '16193',
		'Louzignac (17160)' => '17212',
		'Louzy (79100)' => '79157',
		'Lozay (17330)' => '17213',
		'Lubbon (40240)' => '40161',
		'Lubersac (19210)' => '19121',
		'Luc-Armau (64350)' => '64356',
		'Lucarré (64350)' => '64357',
		'Lucbardez-et-Bargues (40090)' => '40162',
		'Lucgarier (64420)' => '64358',
		'Luchapt (86430)' => '86138',
		'Luchat (17600)' => '17214',
		'Luché-sur-Brioux (79170)' => '79158',
		'Luché-Thouarsais (79330)' => '79159',
		'Lucmau (33840)' => '33255',
		'Lucq-de-Béarn (64360)' => '64359',
		'Ludon-Médoc (33290)' => '33256',
		'Lüe (40210)' => '40163',
		'Lugaignac (33420)' => '33257',
		'Lugasson (33760)' => '33258',
		'Luglon (40630)' => '40165',
		'Lugon-et-l\'Île-du-Carnay (33240)' => '33259',
		'Lugos (33830)' => '33260',
		'Lunas (24130)' => '24246',
		'Lupersat (23190)' => '23113',
		'Lupsault (16140)' => '16194',
		'Luquet (65320)' => '65292',
		'Lurbe-Saint-Christau (64660)' => '64360',
		'Lusignac (24320)' => '24247',
		'Lusignan (86600)' => '86139',
		'Lusignan-Petit (47360)' => '47154',
		'Lussac (16450)' => '16195',
		'Lussac (17500)' => '17215',
		'Lussac (33570)' => '33261',
		'Lussac-les-Châteaux (86320)' => '86140',
		'Lussac-les-Églises (87360)' => '87087',
		'Lussagnet (40270)' => '40166',
		'Lussagnet-Lusson (64160)' => '64361',
		'Lussant (17430)' => '17216',
		'Lussas-et-Nontronneau (24300)' => '24248',
		'Lussat (23170)' => '23114',
		'Lusseray (79170)' => '79160',
		'Luxé (16230)' => '16196',
		'Luxe-Sumberraute (64120)' => '64362',
		'Luxey (40430)' => '40167',
		'Luzay (79100)' => '79161',
		'Lys (64260)' => '64363',
		'Macau (33460)' => '33262',
		'Macaye (64240)' => '64364',
		'Macqueville (17490)' => '17217',
		'Madaillan (47360)' => '47155',
		'Madirac (33670)' => '33263',
		'Madranges (19470)' => '19122',
		'Magescq (40140)' => '40168',
		'Magnac-Bourg (87380)' => '87088',
		'Magnac-Laval (87190)' => '87089',
		'Magnac-Lavalette-Villars (16320)' => '16198',
		'Magnac-sur-Touvre (16600)' => '16199',
		'Magnat-l\'Étrange (23260)' => '23115',
		'Magné (79460)' => '79162',
		'Magné (86160)' => '86141',
		'Mailhac-sur-Benaize (87160)' => '87090',
		'Maillas (40120)' => '40169',
		'Maillé (86190)' => '86142',
		'Maillères (40120)' => '40170',
		'Maine-de-Boixe (16230)' => '16200',
		'Mainsat (23700)' => '23116',
		'Mainxe (16200)' => '16202',
		'Mainzac (16380)' => '16203',
		'Mairé (86270)' => '86143',
		'Mairé-Levescault (79190)' => '79163',
		'Maison-Feyne (23800)' => '23117',
		'Maisonnais-sur-Tardoire (87440)' => '87091',
		'Maisonnay (79500)' => '79164',
		'Maisonneuve (86170)' => '86144',
		'Maisonnisses (23150)' => '23118',
		'Maisontiers (79600)' => '79165',
		'Malaussanne (64410)' => '64365',
		'Malaville (16120)' => '16204',
		'Malemort (19360)' => '19123',
		'Malleret (23260)' => '23119',
		'Malleret-Boussac (23600)' => '23120',
		'Malval (23220)' => '23121',
		'Manaurie (24620)' => '24249',
		'Mano (40410)' => '40171',
		'Manot (16500)' => '16205',
		'Mansac (19520)' => '19124',
		'Mansat-la-Courrière (23400)' => '23122',
		'Mansle (16230)' => '16206',
		'Mant (40700)' => '40172',
		'Manzac-sur-Vern (24110)' => '24251',
		'Marans (17230)' => '17218',
		'Maransin (33230)' => '33264',
		'Marc-la-Tour (19150)' => '19127',
		'Marçay (86370)' => '86145',
		'Marcellus (47200)' => '47156',
		'Marcenais (33620)' => '33266',
		'Marcheprime (33380)' => '33555',
		'Marcillac (33860)' => '33267',
		'Marcillac-la-Croisille (19320)' => '19125',
		'Marcillac-la-Croze (19500)' => '19126',
		'Marcillac-Lanville (16140)' => '16207',
		'Marcillac-Saint-Quentin (24200)' => '24252',
		'Marennes (17320)' => '17219',
		'Mareuil (16170)' => '16208',
		'Mareuil (24340)' => '24253',
		'Margaux (33460)' => '33268',
		'Margerides (19200)' => '19128',
		'Margueron (33220)' => '33269',
		'Marignac (17800)' => '17220',
		'Marigny (79360)' => '79166',
		'Marigny-Brizay (86380)' => '86146',
		'Marigny-Chemereau (86370)' => '86147',
		'Marillac-le-Franc (16110)' => '16209',
		'Marimbault (33430)' => '33270',
		'Marions (33690)' => '33271',
		'Marmande (47200)' => '47157',
		'Marmont-Pachas (47220)' => '47158',
		'Marnac (24220)' => '24254',
		'Marnay (86160)' => '86148',
		'Marnes (79600)' => '79167',
		'Marpaps (40330)' => '40173',
		'Marquay (24620)' => '24255',
		'Marsac (16570)' => '16210',
		'Marsac (23210)' => '23124',
		'Marsac-sur-l\'Isle (24430)' => '24256',
		'Marsais (17700)' => '17221',
		'Marsalès (24540)' => '24257',
		'Marsaneix (24750)' => '24258',
		'Marsas (33620)' => '33272',
		'Marsilly (17137)' => '17222',
		'Martaizé (86330)' => '86149',
		'Marthon (16380)' => '16211',
		'Martignas-sur-Jalle (33127)' => '33273',
		'Martillac (33650)' => '33274',
		'Martres (33760)' => '33275',
		'Marval (87440)' => '87092',
		'Masbaraud-Mérignat (23400)' => '23126',
		'Mascaraàs-Haron (64330)' => '64366',
		'Maslacq (64300)' => '64367',
		'Masléon (87130)' => '87093',
		'Masparraute (64120)' => '64368',
		'Maspie-Lalonquère-Juillacq (64350)' => '64369',
		'Masquières (47370)' => '47160',
		'Massac (17490)' => '17223',
		'Massais (79150)' => '79168',
		'Masseilles (33690)' => '33276',
		'Massels (47140)' => '47161',
		'Masseret (19510)' => '19129',
		'Massignac (16310)' => '16212',
		'Massognes (86170)' => '86150',
		'Massoulès (47140)' => '47162',
		'Massugas (33790)' => '33277',
		'Matha (17160)' => '17224',
		'Maucor (64160)' => '64370',
		'Maulay (86200)' => '86151',
		'Mauléon (79700)' => '79079',
		'Mauléon-Licharre (64130)' => '64371',
		'Mauprévoir (86460)' => '86152',
		'Maure (64460)' => '64372',
		'Maurens (24140)' => '24259',
		'Mauriac (33540)' => '33278',
		'Mauries (40320)' => '40174',
		'Maurrin (40270)' => '40175',
		'Maussac (19250)' => '19130',
		'Mautes (23190)' => '23127',
		'Mauvezin-d\'Armagnac (40240)' => '40176',
		'Mauvezin-sur-Gupie (47200)' => '47163',
		'Mauzac-et-Grand-Castang (24150)' => '24260',
		'Mauzé-sur-le-Mignon (79210)' => '79170',
		'Mauzé-Thouarsais (79100)' => '79171',
		'Mauzens-et-Miremont (24260)' => '24261',
		'Mayac (24420)' => '24262',
		'Maylis (40250)' => '40177',
		'Mazeirat (23150)' => '23128',
		'Mazeray (17400)' => '17226',
		'Mazères (33210)' => '33279',
		'Mazères-Lezons (64110)' => '64373',
		'Mazerolles (16310)' => '16213',
		'Mazerolles (17800)' => '17227',
		'Mazerolles (40090)' => '40178',
		'Mazerolles (64230)' => '64374',
		'Mazerolles (86320)' => '86153',
		'Mazeuil (86110)' => '86154',
		'Mazeyrolles (24550)' => '24263',
		'Mazières (16270)' => '16214',
		'Mazières-en-Gâtine (79310)' => '79172',
		'Mazières-Naresse (47210)' => '47164',
		'Mazières-sur-Béronne (79500)' => '79173',
		'Mazion (33390)' => '33280',
		'Méasnes (23360)' => '23130',
		'Médillac (16210)' => '16215',
		'Médis (17600)' => '17228',
		'Mées (40990)' => '40179',
		'Méharin (64120)' => '64375',
		'Meilhac (87800)' => '87094',
		'Meilhan (40400)' => '40180',
		'Meilhan-sur-Garonne (47180)' => '47165',
		'Meilhards (19510)' => '19131',
		'Meillon (64510)' => '64376',
		'Melle (79500)' => '79174',
		'Melleran (79190)' => '79175',
		'Mendionde (64240)' => '64377',
		'Menditte (64130)' => '64378',
		'Mendive (64220)' => '64379',
		'Ménesplet (24700)' => '24264',
		'Ménigoute (79340)' => '79176',
		'Ménoire (19190)' => '19132',
		'Mensignac (24350)' => '24266',
		'Méracq (64410)' => '64380',
		'Mercoeur (19430)' => '19133',
		'Mérignac (16200)' => '16216',
		'Mérignac (17210)' => '17229',
		'Mérignac (33700)' => '33281',
		'Mérignas (33350)' => '33282',
		'Mérinchal (23420)' => '23131',
		'Méritein (64190)' => '64381',
		'Merlines (19340)' => '19134',
		'Merpins (16100)' => '16217',
		'Meschers-sur-Gironde (17132)' => '17230',
		'Mescoules (24240)' => '24267',
		'Mesnac (16370)' => '16218',
		'Mesplède (64370)' => '64382',
		'Messac (17130)' => '17231',
		'Messanges (40660)' => '40181',
		'Messé (79120)' => '79177',
		'Messemé (86200)' => '86156',
		'Mesterrieux (33540)' => '33283',
		'Mestes (19200)' => '19135',
		'Meursac (17120)' => '17232',
		'Meux (17500)' => '17233',
		'Meuzac (87380)' => '87095',
		'Meymac (19250)' => '19136',
		'Meyrals (24220)' => '24268',
		'Meyrignac-l\'Église (19800)' => '19137',
		'Meyssac (19500)' => '19138',
		'Mézin (47170)' => '47167',
		'Mézos (40170)' => '40182',
		'Mialet (24450)' => '24269',
		'Mialos (64410)' => '64383',
		'Mignaloux-Beauvoir (86550)' => '86157',
		'Migné-Auxances (86440)' => '86158',
		'Migré (17330)' => '17234',
		'Migron (17770)' => '17235',
		'Milhac-d\'Auberoche (24330)' => '24270',
		'Milhac-de-Nontron (24470)' => '24271',
		'Millac (86150)' => '86159',
		'Millevaches (19290)' => '19139',
		'Mimbaste (40350)' => '40183',
		'Mimizan (40200)' => '40184',
		'Minzac (24610)' => '24272',
		'Mios (33380)' => '33284',
		'Miossens-Lanusse (64450)' => '64385',
		'Mirambeau (17150)' => '17236',
		'Miramont-de-Guyenne (47800)' => '47168',
		'Miramont-Sensacq (40320)' => '40185',
		'Mirebeau (86110)' => '86160',
		'Mirepeix (64800)' => '64386',
		'Missé (79100)' => '79178',
		'Misson (40290)' => '40186',
		'Moëze (17780)' => '17237',
		'Moirax (47310)' => '47169',
		'Moissannes (87400)' => '87099',
		'Molières (24480)' => '24273',
		'Moliets-et-Maa (40660)' => '40187',
		'Momas (64230)' => '64387',
		'Mombrier (33710)' => '33285',
		'Momuy (40700)' => '40188',
		'Momy (64350)' => '64388',
		'Monassut-Audiracq (64160)' => '64389',
		'Monbahus (47290)' => '47170',
		'Monbalen (47340)' => '47171',
		'Monbazillac (24240)' => '24274',
		'Moncaup (64350)' => '64390',
		'Moncaut (47310)' => '47172',
		'Moncayolle-Larrory-Mendibieu (64130)' => '64391',
		'Monceaux-sur-Dordogne (19400)' => '19140',
		'Moncla (64330)' => '64392',
		'Monclar (47380)' => '47173',
		'Moncontour (86330)' => '86161',
		'Moncoutant (79320)' => '79179',
		'Moncrabeau (47600)' => '47174',
		'Mondion (86230)' => '86162',
		'Monein (64360)' => '64393',
		'Monestier (24240)' => '24276',
		'Monestier-Merlines (19340)' => '19141',
		'Monestier-Port-Dieu (19110)' => '19142',
		'Monfaucon (24130)' => '24277',
		'Monflanquin (47150)' => '47175',
		'Mongaillard (47230)' => '47176',
		'Mongauzy (33190)' => '33287',
		'Monget (40700)' => '40189',
		'Monheurt (47160)' => '47177',
		'Monmadalès (24560)' => '24278',
		'Monmarvès (24560)' => '24279',
		'Monpazier (24540)' => '24280',
		'Monpezat (64350)' => '64394',
		'Monplaisant (24170)' => '24293',
		'Monprimblanc (33410)' => '33288',
		'Mons (16140)' => '16221',
		'Mons (17160)' => '17239',
		'Monsac (24440)' => '24281',
		'Monsaguel (24560)' => '24282',
		'Monsec (24340)' => '24283',
		'Monségur (33580)' => '33289',
		'Monségur (40700)' => '40190',
		'Monségur (47150)' => '47178',
		'Monségur (64460)' => '64395',
		'Monsempron-Libos (47500)' => '47179',
		'Mont (64300)' => '64396',
		'Mont-de-Marsan (40000)' => '40192',
		'Mont-Disse (64330)' => '64401',
		'Montagnac-d\'Auberoche (24210)' => '24284',
		'Montagnac-la-Crempse (24140)' => '24285',
		'Montagnac-sur-Auvignon (47600)' => '47180',
		'Montagnac-sur-Lède (47150)' => '47181',
		'Montagne (33570)' => '33290',
		'Montagoudin (33190)' => '33291',
		'Montagrier (24350)' => '24286',
		'Montagut (64410)' => '64397',
		'Montaignac-Saint-Hippolyte (19300)' => '19143',
		'Montaigut-le-Blanc (23320)' => '23132',
		'Montalembert (79190)' => '79180',
		'Montamisé (86360)' => '86163',
		'Montaner (64460)' => '64398',
		'Montardon (64121)' => '64399',
		'Montastruc (47380)' => '47182',
		'Montauriol (47330)' => '47183',
		'Montaut (24560)' => '24287',
		'Montaut (40500)' => '40191',
		'Montaut (47210)' => '47184',
		'Montaut (64800)' => '64400',
		'Montayral (47500)' => '47185',
		'Montazeau (24230)' => '24288',
		'Montboucher (23400)' => '23133',
		'Montboyer (16620)' => '16222',
		'Montbron (16220)' => '16223',
		'Montcaret (24230)' => '24289',
		'Montégut (40190)' => '40193',
		'Montemboeuf (16310)' => '16225',
		'Montendre (17130)' => '17240',
		'Montesquieu (47130)' => '47186',
		'Monteton (47120)' => '47187',
		'Montferrand-du-Périgord (24440)' => '24290',
		'Montfort (64190)' => '64403',
		'Montfort-en-Chalosse (40380)' => '40194',
		'Montgaillard (40500)' => '40195',
		'Montgibaud (19210)' => '19144',
		'Montguyon (17270)' => '17241',
		'Monthoiron (86210)' => '86164',
		'Montignac (24290)' => '24291',
		'Montignac (33760)' => '33292',
		'Montignac-Charente (16330)' => '16226',
		'Montignac-de-Lauzun (47800)' => '47188',
		'Montignac-le-Coq (16390)' => '16227',
		'Montignac-Toupinerie (47350)' => '47189',
		'Montigné (16170)' => '16228',
		'Montils (17800)' => '17242',
		'Montjean (16240)' => '16229',
		'Montlieu-la-Garde (17210)' => '17243',
		'Montmérac (16300)' => '16224',
		'Montmoreau-Saint-Cybard (16190)' => '16230',
		'Montmorillon (86500)' => '86165',
		'Montory (64470)' => '64404',
		'Montpellier-de-Médillan (17260)' => '17244',
		'Montpeyroux (24610)' => '24292',
		'Montpezat (47360)' => '47190',
		'Montpon-Ménestérol (24700)' => '24294',
		'Montpouillan (47200)' => '47191',
		'Montravers (79140)' => '79183',
		'Montrem (24110)' => '24295',
		'Montreuil-Bonnin (86470)' => '86166',
		'Montrol-Sénard (87330)' => '87100',
		'Montrollet (16420)' => '16231',
		'Montroy (17220)' => '17245',
		'Monts-sur-Guesnes (86420)' => '86167',
		'Montsoué (40500)' => '40196',
		'Montussan (33450)' => '33293',
		'Monviel (47290)' => '47192',
		'Moragne (17430)' => '17246',
		'Morcenx (40110)' => '40197',
		'Morganx (40700)' => '40198',
		'Morizès (33190)' => '33294',
		'Morlaàs (64160)' => '64405',
		'Morlanne (64370)' => '64406',
		'Mornac (16600)' => '16232',
		'Mornac-sur-Seudre (17113)' => '17247',
		'Mortagne-sur-Gironde (17120)' => '17248',
		'Mortemart (87330)' => '87101',
		'Mortiers (17500)' => '17249',
		'Morton (86120)' => '86169',
		'Mortroux (23220)' => '23136',
		'Mosnac (16120)' => '16233',
		'Mosnac (17240)' => '17250',
		'Mougon (79370)' => '79185',
		'Mouguerre (64990)' => '64407',
		'Mouhous (64330)' => '64408',
		'Mouillac (33240)' => '33295',
		'Mouleydier (24520)' => '24296',
		'Moulidars (16290)' => '16234',
		'Mouliets-et-Villemartin (33350)' => '33296',
		'Moulin-Neuf (24700)' => '24297',
		'Moulinet (47290)' => '47193',
		'Moulis-en-Médoc (33480)' => '33297',
		'Moulismes (86500)' => '86170',
		'Moulon (33420)' => '33298',
		'Moumour (64400)' => '64409',
		'Mourens (33410)' => '33299',
		'Mourenx (64150)' => '64410',
		'Mourioux-Vieilleville (23210)' => '23137',
		'Mouscardès (40290)' => '40199',
		'Moussac (86150)' => '86171',
		'Moustey (40410)' => '40200',
		'Moustier (47800)' => '47194',
		'Moustier-Ventadour (19300)' => '19145',
		'Mouterre-Silly (86200)' => '86173',
		'Mouterre-sur-Blourde (86430)' => '86172',
		'Mouthiers-sur-Boëme (16440)' => '16236',
		'Moutier-d\'Ahun (23150)' => '23138',
		'Moutier-Malcard (23220)' => '23139',
		'Moutier-Rozeille (23200)' => '23140',
		'Moutiers-sous-Chantemerle (79320)' => '79188',
		'Mouton (16460)' => '16237',
		'Moutonneau (16460)' => '16238',
		'Mouzon (16310)' => '16239',
		'Mugron (40250)' => '40201',
		'Muron (17430)' => '17253',
		'Musculdy (64130)' => '64411',
		'Mussidan (24400)' => '24299',
		'Nabas (64190)' => '64412',
		'Nabinaud (16390)' => '16240',
		'Nabirat (24250)' => '24300',
		'Nachamps (17380)' => '17254',
		'Nadaillac (24590)' => '24301',
		'Nailhac (24390)' => '24302',
		'Naillat (23800)' => '23141',
		'Naintré (86530)' => '86174',
		'Nalliers (86310)' => '86175',
		'Nanclars (16230)' => '16241',
		'Nancras (17600)' => '17255',
		'Nanteuil (79400)' => '79189',
		'Nanteuil-Auriac-de-Bourzac (24320)' => '24303',
		'Nanteuil-en-Vallée (16700)' => '16242',
		'Nantheuil (24800)' => '24304',
		'Nanthiat (24800)' => '24305',
		'Nantiat (87140)' => '87103',
		'Nantillé (17770)' => '17256',
		'Narcastet (64510)' => '64413',
		'Narp (64190)' => '64414',
		'Narrosse (40180)' => '40202',
		'Nassiet (40330)' => '40203',
		'Nastringues (24230)' => '24306',
		'Naujac-sur-Mer (33990)' => '33300',
		'Naujan-et-Postiac (33420)' => '33301',
		'Naussannes (24440)' => '24307',
		'Navailles-Angos (64450)' => '64415',
		'Navarrenx (64190)' => '64416',
		'Naves (19460)' => '19146',
		'Nay (64800)' => '64417',
		'Néac (33500)' => '33302',
		'Nedde (87120)' => '87104',
		'Négrondes (24460)' => '24308',
		'Néoux (23200)' => '23142',
		'Nérac (47600)' => '47195',
		'Nerbis (40250)' => '40204',
		'Nercillac (16200)' => '16243',
		'Néré (17510)' => '17257',
		'Nérigean (33750)' => '33303',
		'Nérignac (86150)' => '86176',
		'Nersac (16440)' => '16244',
		'Nespouls (19600)' => '19147',
		'Neuffons (33580)' => '33304',
		'Neuillac (17520)' => '17258',
		'Neulles (17500)' => '17259',
		'Neuvic (19160)' => '19148',
		'Neuvic (24190)' => '24309',
		'Neuvic-Entier (87130)' => '87105',
		'Neuvicq (17270)' => '17260',
		'Neuvicq-le-Château (17490)' => '17261',
		'Neuville (19380)' => '19149',
		'Neuville-de-Poitou (86170)' => '86177',
		'Neuvy-Bouin (79130)' => '79190',
		'Nexon (87800)' => '87106',
		'Nicole (47190)' => '47196',
		'Nieuil (16270)' => '16245',
		'Nieuil-l\'Espoir (86340)' => '86178',
		'Nieul (87510)' => '87107',
		'Nieul-le-Virouil (17150)' => '17263',
		'Nieul-lès-Saintes (17810)' => '17262',
		'Nieul-sur-Mer (17137)' => '17264',
		'Nieulle-sur-Seudre (17600)' => '17265',
		'Niort (79000)' => '79191',
		'Noailhac (19500)' => '19150',
		'Noaillac (33190)' => '33306',
		'Noaillan (33730)' => '33307',
		'Noailles (19600)' => '19151',
		'Noguères (64150)' => '64418',
		'Nomdieu (47600)' => '47197',
		'Nonac (16190)' => '16246',
		'Nonards (19120)' => '19152',
		'Nonaville (16120)' => '16247',
		'Nontron (24300)' => '24311',
		'Noth (23300)' => '23143',
		'Notre-Dame-de-Sanilhac (24660)' => '24312',
		'Nouaillé-Maupertuis (86340)' => '86180',
		'Nouhant (23170)' => '23145',
		'Nouic (87330)' => '87108',
		'Nousse (40380)' => '40205',
		'Nousty (64420)' => '64419',
		'Nouzerines (23600)' => '23146',
		'Nouzerolles (23360)' => '23147',
		'Nouziers (23350)' => '23148',
		'Nuaillé-d\'Aunis (17540)' => '17267',
		'Nuaillé-sur-Boutonne (17470)' => '17268',
		'Nueil-les-Aubiers (79250)' => '79195',
		'Nueil-sous-Faye (86200)' => '86181',
		'Objat (19130)' => '19153',
		'Oeyregave (40300)' => '40206',
		'Oeyreluy (40180)' => '40207',
		'Ogenne-Camptort (64190)' => '64420',
		'Ogeu-les-Bains (64680)' => '64421',
		'Oiron (79100)' => '79196',
		'Oloron-Sainte-Marie (64400)' => '64422',
		'Omet (33410)' => '33308',
		'Onard (40380)' => '40208',
		'Ondres (40440)' => '40209',
		'Onesse-Laharie (40110)' => '40210',
		'Oraàs (64390)' => '64423',
		'Oradour (16140)' => '16248',
		'Oradour-Fanais (16500)' => '16249',
		'Oradour-Saint-Genest (87210)' => '87109',
		'Oradour-sur-Glane (87520)' => '87110',
		'Oradour-sur-Vayres (87150)' => '87111',
		'Orches (86230)' => '86182',
		'Ordiarp (64130)' => '64424',
		'Ordonnac (33340)' => '33309',
		'Orègue (64120)' => '64425',
		'Orgedeuil (16220)' => '16250',
		'Orgnac-sur-Vézère (19410)' => '19154',
		'Origne (33113)' => '33310',
		'Orignolles (17210)' => '17269',
		'Orin (64400)' => '64426',
		'Oriolles (16480)' => '16251',
		'Orion (64390)' => '64427',
		'Orist (40300)' => '40211',
		'Orival (16210)' => '16252',
		'Orliac (24170)' => '24313',
		'Orliac-de-Bar (19390)' => '19155',
		'Orliaguet (24370)' => '24314',
		'Oroux (79390)' => '79197',
		'Orriule (64390)' => '64428',
		'Orsanco (64120)' => '64429',
		'Orthevielle (40300)' => '40212',
		'Orthez (64300)' => '64430',
		'Orx (40230)' => '40213',
		'Os-Marsillon (64150)' => '64431',
		'Ossages (40290)' => '40214',
		'Ossas-Suhare (64470)' => '64432',
		'Osse-en-Aspe (64490)' => '64433',
		'Ossenx (64190)' => '64434',
		'Osserain-Rivareyte (64390)' => '64435',
		'Ossès (64780)' => '64436',
		'Ostabat-Asme (64120)' => '64437',
		'Ouillon (64160)' => '64438',
		'Ousse (64320)' => '64439',
		'Ousse-Suzan (40110)' => '40215',
		'Ouzilly (86380)' => '86184',
		'Oyré (86220)' => '86186',
		'Ozenx-Montestrucq (64300)' => '64440',
		'Ozillac (17500)' => '17270',
		'Ozourt (40380)' => '40216',
		'Pageas (87230)' => '87112',
		'Pagolle (64120)' => '64441',
		'Paillé (17470)' => '17271',
		'Paillet (33550)' => '33311',
		'Pailloles (47440)' => '47198',
		'Paizay-le-Chapt (79170)' => '79198',
		'Paizay-le-Sec (86300)' => '86187',
		'Paizay-le-Tort (79500)' => '79199',
		'Paizay-Naudouin-Embourie (16240)' => '16253',
		'Palazinges (19190)' => '19156',
		'Palisse (19160)' => '19157',
		'Palluaud (16390)' => '16254',
		'Pamplie (79220)' => '79200',
		'Pamproux (79800)' => '79201',
		'Panazol (87350)' => '87114',
		'Pandrignes (19150)' => '19158',
		'Parbayse (64360)' => '64442',
		'Parcoul-Chenaud (24410)' => '24316',
		'Pardaillan (47120)' => '47199',
		'Pardies (64150)' => '64443',
		'Pardies-Piétat (64800)' => '64444',
		'Parempuyre (33290)' => '33312',
		'Parentis-en-Born (40160)' => '40217',
		'Parleboscq (40310)' => '40218',
		'Parranquet (47210)' => '47200',
		'Parsac-Rimondeix (23140)' => '23149',
		'Parthenay (79200)' => '79202',
		'Parzac (16450)' => '16255',
		'Pas-de-Jeu (79100)' => '79203',
		'Passirac (16480)' => '16256',
		'Pau (64000)' => '64445',
		'Pauillac (33250)' => '33314',
		'Paulhiac (47150)' => '47202',
		'Paulin (24590)' => '24317',
		'Paunat (24510)' => '24318',
		'Paussac-et-Saint-Vivien (24310)' => '24319',
		'Payré (86700)' => '86188',
		'Payros-Cazautets (40320)' => '40219',
		'Payroux (86350)' => '86189',
		'Pays de Belvès (24170)' => '24035',
		'Payzac (24270)' => '24320',
		'Pazayac (24120)' => '24321',
		'Pécorade (40320)' => '40220',
		'Pellegrue (33790)' => '33316',
		'Penne-d\'Agenais (47140)' => '47203',
		'Pensol (87440)' => '87115',
		'Péré (17700)' => '17272',
		'Péret-Bel-Air (19300)' => '19159',
		'Pérignac (16250)' => '16258',
		'Pérignac (17800)' => '17273',
		'Périgné (79170)' => '79204',
		'Périgny (17180)' => '17274',
		'Périgueux (24000)' => '24322',
		'Périssac (33240)' => '33317',
		'Pérols-sur-Vézère (19170)' => '19160',
		'Perpezac-le-Blanc (19310)' => '19161',
		'Perpezac-le-Noir (19410)' => '19162',
		'Perquie (40190)' => '40221',
		'Pers (79190)' => '79205',
		'Persac (86320)' => '86190',
		'Pessac (33600)' => '33318',
		'Pessac-sur-Dordogne (33890)' => '33319',
		'Pessines (17810)' => '17275',
		'Petit-Bersac (24600)' => '24323',
		'Petit-Palais-et-Cornemps (33570)' => '33320',
		'Peujard (33240)' => '33321',
		'Pey (40300)' => '40222',
		'Peyrabout (23000)' => '23150',
		'Peyrat-de-Bellac (87300)' => '87116',
		'Peyrat-la-Nonière (23130)' => '23151',
		'Peyrat-le-Château (87470)' => '87117',
		'Peyre (40700)' => '40223',
		'Peyrehorade (40300)' => '40224',
		'Peyrelevade (19290)' => '19164',
		'Peyrelongue-Abos (64350)' => '64446',
		'Peyrière (47350)' => '47204',
		'Peyrignac (24210)' => '24324',
		'Peyrilhac (87510)' => '87118',
		'Peyrillac-et-Millac (24370)' => '24325',
		'Peyrissac (19260)' => '19165',
		'Peyzac-le-Moustier (24620)' => '24326',
		'Pezuls (24510)' => '24327',
		'Philondenx (40320)' => '40225',
		'Piégut-Pluviers (24360)' => '24328',
		'Pierre-Buffière (87260)' => '87119',
		'Pierrefitte (19450)' => '19166',
		'Pierrefitte (23130)' => '23152',
		'Pierrefitte (79330)' => '79209',
		'Piets-Plasence-Moustrou (64410)' => '64447',
		'Pillac (16390)' => '16260',
		'Pimbo (40320)' => '40226',
		'Pindères (47700)' => '47205',
		'Pindray (86500)' => '86191',
		'Pinel-Hauterive (47380)' => '47206',
		'Pineuilh (33220)' => '33324',
		'Pionnat (23140)' => '23154',
		'Pioussay (79110)' => '79211',
		'Pisany (17600)' => '17278',
		'Pissos (40410)' => '40227',
		'Plaisance (24560)' => '24168',
		'Plaisance (86500)' => '86192',
		'Plassac (17240)' => '17279',
		'Plassac (33390)' => '33325',
		'Plassac-Rouffiac (16250)' => '16263',
		'Plassay (17250)' => '17280',
		'Plazac (24580)' => '24330',
		'Pleine-Selve (33820)' => '33326',
		'Pleumartin (86450)' => '86193',
		'Pleuville (16490)' => '16264',
		'Pliboux (79190)' => '79212',
		'Podensac (33720)' => '33327',
		'Poey-d\'Oloron (64400)' => '64449',
		'Poey-de-Lescar (64230)' => '64448',
		'Poitiers (86000)' => '86194',
		'Polignac (17210)' => '17281',
		'Pomarez (40360)' => '40228',
		'Pomerol (33500)' => '33328',
		'Pommiers-Moulons (17130)' => '17282',
		'Pompaire (79200)' => '79213',
		'Pompéjac (33730)' => '33329',
		'Pompiey (47230)' => '47207',
		'Pompignac (33370)' => '33330',
		'Pompogne (47420)' => '47208',
		'Pomport (24240)' => '24331',
		'Pomps (64370)' => '64450',
		'Pondaurat (33190)' => '33331',
		'Pons (17800)' => '17283',
		'Ponson-Debat-Pouts (64460)' => '64451',
		'Ponson-Dessus (64460)' => '64452',
		'Pont-du-Casse (47480)' => '47209',
		'Pont-l\'Abbé-d\'Arnoult (17250)' => '17284',
		'Pontacq (64530)' => '64453',
		'Pontarion (23250)' => '23155',
		'Pontcharraud (23260)' => '23156',
		'Pontenx-les-Forges (40200)' => '40229',
		'Ponteyraud (24410)' => '24333',
		'Pontiacq-Viellepinte (64460)' => '64454',
		'Pontonx-sur-l\'Adour (40465)' => '40230',
		'Pontours (24150)' => '24334',
		'Porchères (33660)' => '33332',
		'Port-d\'Envaux (17350)' => '17285',
		'Port-de-Lanne (40300)' => '40231',
		'Port-de-Piles (86220)' => '86195',
		'Port-des-Barques (17730)' => '17484',
		'Port-Sainte-Foy-et-Ponchapt (33220)' => '24335',
		'Port-Sainte-Marie (47130)' => '47210',
		'Portet (64330)' => '64455',
		'Portets (33640)' => '33334',
		'Pouançay (86120)' => '86196',
		'Pouant (86200)' => '86197',
		'Poudenas (47170)' => '47211',
		'Poudenx (40700)' => '40232',
		'Pouffonds (79500)' => '79214',
		'Pougne-Hérisson (79130)' => '79215',
		'Pouillac (17210)' => '17287',
		'Pouillé (86800)' => '86198',
		'Pouillon (40350)' => '40233',
		'Pouliacq (64410)' => '64456',
		'Poullignac (16190)' => '16267',
		'Poursac (16700)' => '16268',
		'Poursay-Garnaud (17400)' => '17288',
		'Poursiugues-Boucoue (64410)' => '64457',
		'Poussanges (23500)' => '23158',
		'Poussignac (47700)' => '47212',
		'Pouydesseaux (40120)' => '40234',
		'Poyanne (40380)' => '40235',
		'Poyartin (40380)' => '40236',
		'Pradines (19170)' => '19168',
		'Prahecq (79230)' => '79216',
		'Prailles (79370)' => '79217',
		'Pranzac (16110)' => '16269',
		'Prats-de-Carlux (24370)' => '24336',
		'Prats-du-Périgord (24550)' => '24337',
		'Prayssas (47360)' => '47213',
		'Préchac (33730)' => '33336',
		'Préchacq-Josbaig (64190)' => '64458',
		'Préchacq-les-Bains (40465)' => '40237',
		'Préchacq-Navarrenx (64190)' => '64459',
		'Précilhon (64400)' => '64460',
		'Préguillac (17460)' => '17289',
		'Preignac (33210)' => '33337',
		'Pressac (86460)' => '86200',
		'Pressignac (16150)' => '16270',
		'Pressignac-Vicq (24150)' => '24338',
		'Pressigny (79390)' => '79218',
		'Preyssac-d\'Excideuil (24160)' => '24339',
		'Priaires (79210)' => '79219',
		'Prignac (17160)' => '17290',
		'Prignac-en-Médoc (33340)' => '33338',
		'Prignac-et-Marcamps (33710)' => '33339',
		'Prigonrieux (24130)' => '24340',
		'Prin-Deyrançon (79210)' => '79220',
		'Prinçay (86420)' => '86201',
		'Prissé-la-Charrière (79360)' => '79078',
		'Proissans (24200)' => '24341',
		'Puch-d\'Agenais (47160)' => '47214',
		'Pugnac (33710)' => '33341',
		'Pugny (79320)' => '79222',
		'Puihardy (79160)' => '79223',
		'Puilboreau (17138)' => '17291',
		'Puisseguin (33570)' => '33342',
		'Pujo-le-Plan (40190)' => '40238',
		'Pujols (33350)' => '33344',
		'Pujols (47300)' => '47215',
		'Pujols-sur-Ciron (33210)' => '33343',
		'Puy-d\'Arnac (19120)' => '19169',
		'Puy-du-Lac (17380)' => '17292',
		'Puy-Malsignat (23130)' => '23159',
		'Puybarban (33190)' => '33346',
		'Puymiclan (47350)' => '47216',
		'Puymirol (47270)' => '47217',
		'Puymoyen (16400)' => '16271',
		'Puynormand (33660)' => '33347',
		'Puyol-Cazalet (40320)' => '40239',
		'Puyoô (64270)' => '64461',
		'Puyravault (17700)' => '17293',
		'Puyréaux (16230)' => '16272',
		'Puyrenier (24340)' => '24344',
		'Puyrolland (17380)' => '17294',
		'Puysserampion (47800)' => '47218',
		'Queaux (86150)' => '86203',
		'Queyrac (33340)' => '33348',
		'Queyssac (24140)' => '24345',
		'Queyssac-les-Vignes (19120)' => '19170',
		'Quinçay (86190)' => '86204',
		'Quinsac (24530)' => '24346',
		'Quinsac (33360)' => '33349',
		'Raix (16240)' => '16273',
		'Ramous (64270)' => '64462',
		'Rampieux (24440)' => '24347',
		'Rancogne (16110)' => '16274',
		'Rancon (87290)' => '87121',
		'Ranton (86200)' => '86205',
		'Ranville-Breuillaud (16140)' => '16275',
		'Raslay (86120)' => '86206',
		'Rauzan (33420)' => '33350',
		'Rayet (47210)' => '47219',
		'Razac-d\'Eymet (24500)' => '24348',
		'Razac-de-Saussignac (24240)' => '24349',
		'Razac-sur-l\'Isle (24430)' => '24350',
		'Razès (87640)' => '87122',
		'Razimet (47160)' => '47220',
		'Réaup-Lisse (47170)' => '47221',
		'Réaux sur Trèfle (17500)' => '17295',
		'Rébénacq (64260)' => '64463',
		'Reffannes (79420)' => '79225',
		'Reignac (16360)' => '16276',
		'Reignac (33860)' => '33351',
		'Rempnat (87120)' => '87123',
		'Renung (40270)' => '40240',
		'Réparsac (16200)' => '16277',
		'Rétaud (17460)' => '17296',
		'Reterre (23110)' => '23160',
		'Retjons (40120)' => '40164',
		'Reygade (19430)' => '19171',
		'Ribagnac (24240)' => '24351',
		'Ribarrouy (64330)' => '64464',
		'Ribérac (24600)' => '24352',
		'Rilhac-Lastours (87800)' => '87124',
		'Rilhac-Rancon (87570)' => '87125',
		'Rilhac-Treignac (19260)' => '19172',
		'Rilhac-Xaintrie (19220)' => '19173',
		'Rimbez-et-Baudiets (40310)' => '40242',
		'Rimons (33580)' => '33353',
		'Riocaud (33220)' => '33354',
		'Rion-des-Landes (40370)' => '40243',
		'Rions (33410)' => '33355',
		'Rioux (17460)' => '17298',
		'Rioux-Martin (16210)' => '16279',
		'Riupeyrous (64160)' => '64465',
		'Rivedoux-Plage (17940)' => '17297',
		'Rivehaute (64190)' => '64466',
		'Rives (47210)' => '47223',
		'Rivière-Saas-et-Gourby (40180)' => '40244',
		'Rivières (16110)' => '16280',
		'Roaillan (33210)' => '33357',
		'Roche-le-Peyroux (19160)' => '19175',
		'Rochechouart (87600)' => '87126',
		'Rochefort (17300)' => '17299',
		'Roches (23270)' => '23162',
		'Roches-Prémarie-Andillé (86340)' => '86209',
		'Roiffé (86120)' => '86210',
		'Rom (79120)' => '79230',
		'Romagne (33760)' => '33358',
		'Romagne (86700)' => '86211',
		'Romans (79260)' => '79231',
		'Romazières (17510)' => '17301',
		'Romegoux (17250)' => '17302',
		'Romestaing (47250)' => '47224',
		'Ronsenac (16320)' => '16283',
		'Rontignon (64110)' => '64467',
		'Roquebrune (33580)' => '33359',
		'Roquefort (40120)' => '40245',
		'Roquefort (47310)' => '47225',
		'Roquiague (64130)' => '64468',
		'Rosiers-d\'Égletons (19300)' => '19176',
		'Rosiers-de-Juillac (19350)' => '19177',
		'Rouffiac (16210)' => '16284',
		'Rouffiac (17800)' => '17304',
		'Rouffignac (17130)' => '17305',
		'Rouffignac-de-Sigoulès (24240)' => '24357',
		'Rouffignac-Saint-Cernin-de-Reilhac (24580)' => '24356',
		'Rougnac (16320)' => '16285',
		'Rougnat (23700)' => '23164',
		'Rouillac (16170)' => '16286',
		'Rouillé (86480)' => '86213',
		'Roullet-Saint-Estèphe (16440)' => '16287',
		'Roumagne (47800)' => '47226',
		'Roumazières-Loubert (16270)' => '16192',
		'Roussac (87140)' => '87128',
		'Roussines (16310)' => '16289',
		'Rouzède (16220)' => '16290',
		'Royan (17200)' => '17306',
		'Royère-de-Vassivière (23460)' => '23165',
		'Royères (87400)' => '87129',
		'Roziers-Saint-Georges (87130)' => '87130',
		'Ruch (33350)' => '33361',
		'Rudeau-Ladosse (24340)' => '24221',
		'Ruelle-sur-Touvre (16600)' => '16291',
		'Ruffec (16700)' => '16292',
		'Ruffiac (47700)' => '47227',
		'Sablonceaux (17600)' => '17307',
		'Sablons (33910)' => '33362',
		'Sabres (40630)' => '40246',
		'Sadillac (24500)' => '24359',
		'Sadirac (33670)' => '33363',
		'Sadroc (19270)' => '19178',
		'Sagelat (24170)' => '24360',
		'Sagnat (23800)' => '23166',
		'Saillac (19500)' => '19179',
		'Saillans (33141)' => '33364',
		'Saillat-sur-Vienne (87720)' => '87131',
		'Saint Aulaye-Puymangou (24410)' => '24376',
		'Saint Maurice Étusson (79150)' => '79280',
		'Saint-Abit (64800)' => '64469',
		'Saint-Adjutory (16310)' => '16293',
		'Saint-Agnant (17620)' => '17308',
		'Saint-Agnant-de-Versillat (23300)' => '23177',
		'Saint-Agnant-près-Crocq (23260)' => '23178',
		'Saint-Agne (24520)' => '24361',
		'Saint-Agnet (40800)' => '40247',
		'Saint-Aignan (33126)' => '33365',
		'Saint-Aigulin (17360)' => '17309',
		'Saint-Alpinien (23200)' => '23179',
		'Saint-Amand (23200)' => '23180',
		'Saint-Amand-de-Coly (24290)' => '24364',
		'Saint-Amand-de-Vergt (24380)' => '24365',
		'Saint-Amand-Jartoudeix (23400)' => '23181',
		'Saint-Amand-le-Petit (87120)' => '87132',
		'Saint-Amand-Magnazeix (87290)' => '87133',
		'Saint-Amand-sur-Sèvre (79700)' => '79235',
		'Saint-Amant-de-Boixe (16330)' => '16295',
		'Saint-Amant-de-Bonnieure (16230)' => '16296',
		'Saint-Amant-de-Montmoreau (16190)' => '16294',
		'Saint-Amant-de-Nouère (16170)' => '16298',
		'Saint-André-d\'Allas (24200)' => '24366',
		'Saint-André-de-Cubzac (33240)' => '33366',
		'Saint-André-de-Double (24190)' => '24367',
		'Saint-André-de-Lidon (17260)' => '17310',
		'Saint-André-de-Seignanx (40390)' => '40248',
		'Saint-André-du-Bois (33490)' => '33367',
		'Saint-André-et-Appelles (33220)' => '33369',
		'Saint-André-sur-Sèvre (79380)' => '79236',
		'Saint-Androny (33390)' => '33370',
		'Saint-Angeau (16230)' => '16300',
		'Saint-Angel (19200)' => '19180',
		'Saint-Antoine-Cumond (24410)' => '24368',
		'Saint-Antoine-d\'Auberoche (24330)' => '24369',
		'Saint-Antoine-de-Breuilh (24230)' => '24370',
		'Saint-Antoine-de-Ficalba (47340)' => '47228',
		'Saint-Antoine-du-Queyret (33790)' => '33372',
		'Saint-Antoine-sur-l\'Isle (33660)' => '33373',
		'Saint-Aquilin (24110)' => '24371',
		'Saint-Armou (64160)' => '64470',
		'Saint-Astier (24110)' => '24372',
		'Saint-Astier (47120)' => '47229',
		'Saint-Aubin (40250)' => '40249',
		'Saint-Aubin (47150)' => '47230',
		'Saint-Aubin-de-Blaye (33820)' => '33374',
		'Saint-Aubin-de-Branne (33420)' => '33375',
		'Saint-Aubin-de-Cadelech (24500)' => '24373',
		'Saint-Aubin-de-Lanquais (24560)' => '24374',
		'Saint-Aubin-de-Médoc (33160)' => '33376',
		'Saint-Aubin-de-Nabirat (24250)' => '24375',
		'Saint-Aubin-du-Plain (79300)' => '79238',
		'Saint-Aubin-le-Cloud (79450)' => '79239',
		'Saint-Augustin (17570)' => '17311',
		'Saint-Augustin (19390)' => '19181',
		'Saint-Aulaire (19130)' => '19182',
		'Saint-Aulais-la-Chapelle (16300)' => '16301',
		'Saint-Auvent (87310)' => '87135',
		'Saint-Avit (16210)' => '16302',
		'Saint-Avit (40090)' => '40250',
		'Saint-Avit (47350)' => '47231',
		'Saint-Avit-de-Soulège (33220)' => '33377',
		'Saint-Avit-de-Tardes (23200)' => '23182',
		'Saint-Avit-de-Vialard (24260)' => '24377',
		'Saint-Avit-le-Pauvre (23480)' => '23183',
		'Saint-Avit-Rivière (24540)' => '24378',
		'Saint-Avit-Saint-Nazaire (33220)' => '33378',
		'Saint-Avit-Sénieur (24440)' => '24379',
		'Saint-Barbant (87330)' => '87136',
		'Saint-Bard (23260)' => '23184',
		'Saint-Barthélemy (40390)' => '40251',
		'Saint-Barthélemy-d\'Agenais (47350)' => '47232',
		'Saint-Barthélemy-de-Bellegarde (24700)' => '24380',
		'Saint-Barthélemy-de-Bussière (24360)' => '24381',
		'Saint-Bazile (87150)' => '87137',
		'Saint-Bazile-de-la-Roche (19320)' => '19183',
		'Saint-Bazile-de-Meyssac (19500)' => '19184',
		'Saint-Benoît (86280)' => '86214',
		'Saint-Boès (64300)' => '64471',
		'Saint-Bonnet (16300)' => '16303',
		'Saint-Bonnet-Avalouze (19150)' => '19185',
		'Saint-Bonnet-Briance (87260)' => '87138',
		'Saint-Bonnet-de-Bellac (87300)' => '87139',
		'Saint-Bonnet-Elvert (19380)' => '19186',
		'Saint-Bonnet-l\'Enfantier (19410)' => '19188',
		'Saint-Bonnet-la-Rivière (19130)' => '19187',
		'Saint-Bonnet-les-Tours-de-Merle (19430)' => '19189',
		'Saint-Bonnet-près-Bort (19200)' => '19190',
		'Saint-Bonnet-sur-Gironde (17150)' => '17312',
		'Saint-Brice (16100)' => '16304',
		'Saint-Brice (33540)' => '33379',
		'Saint-Brice-sur-Vienne (87200)' => '87140',
		'Saint-Bris-des-Bois (17770)' => '17313',
		'Saint-Caprais-de-Blaye (33820)' => '33380',
		'Saint-Caprais-de-Bordeaux (33880)' => '33381',
		'Saint-Caprais-de-Lerm (47270)' => '47234',
		'Saint-Capraise-d\'Eymet (24500)' => '24383',
		'Saint-Capraise-de-Lalinde (24150)' => '24382',
		'Saint-Cassien (24540)' => '24384',
		'Saint-Castin (64160)' => '64472',
		'Saint-Cernin-de-l\'Herm (24550)' => '24386',
		'Saint-Cernin-de-Labarde (24560)' => '24385',
		'Saint-Cernin-de-Larche (19600)' => '19191',
		'Saint-Césaire (17770)' => '17314',
		'Saint-Chabrais (23130)' => '23185',
		'Saint-Chamant (19380)' => '19192',
		'Saint-Chamassy (24260)' => '24388',
		'Saint-Christoly-de-Blaye (33920)' => '33382',
		'Saint-Christoly-Médoc (33340)' => '33383',
		'Saint-Christophe (16420)' => '16306',
		'Saint-Christophe (17220)' => '17315',
		'Saint-Christophe (23000)' => '23186',
		'Saint-Christophe (86230)' => '86217',
		'Saint-Christophe-de-Double (33230)' => '33385',
		'Saint-Christophe-des-Bardes (33330)' => '33384',
		'Saint-Christophe-sur-Roc (79220)' => '79241',
		'Saint-Cibard (33570)' => '33386',
		'Saint-Ciers-Champagne (17520)' => '17316',
		'Saint-Ciers-d\'Abzac (33910)' => '33387',
		'Saint-Ciers-de-Canesse (33710)' => '33388',
		'Saint-Ciers-du-Taillon (17240)' => '17317',
		'Saint-Ciers-sur-Bonnieure (16230)' => '16307',
		'Saint-Ciers-sur-Gironde (33820)' => '33389',
		'Saint-Cirgues-la-Loutre (19220)' => '19193',
		'Saint-Cirq (24260)' => '24389',
		'Saint-Clair (86330)' => '86218',
		'Saint-Claud (16450)' => '16308',
		'Saint-Clément (19700)' => '19194',
		'Saint-Clément-des-Baleines (17590)' => '17318',
		'Saint-Colomb-de-Lauzun (47410)' => '47235',
		'Saint-Côme (33430)' => '33391',
		'Saint-Coutant (16350)' => '16310',
		'Saint-Coutant (79120)' => '79243',
		'Saint-Coutant-le-Grand (17430)' => '17320',
		'Saint-Crépin (17380)' => '17321',
		'Saint-Crépin-d\'Auberoche (24330)' => '24390',
		'Saint-Crépin-de-Richemont (24310)' => '24391',
		'Saint-Crépin-et-Carlucet (24590)' => '24392',
		'Saint-Cricq-Chalosse (40700)' => '40253',
		'Saint-Cricq-du-Gave (40300)' => '40254',
		'Saint-Cricq-Villeneuve (40190)' => '40255',
		'Saint-Cybardeaux (16170)' => '16312',
		'Saint-Cybranet (24250)' => '24395',
		'Saint-Cyprien (19130)' => '19195',
		'Saint-Cyprien (24220)' => '24396',
		'Saint-Cyr (86130)' => '86219',
		'Saint-Cyr (87310)' => '87141',
		'Saint-Cyr-du-Doret (17170)' => '17322',
		'Saint-Cyr-la-Lande (79100)' => '79244',
		'Saint-Cyr-la-Roche (19130)' => '19196',
		'Saint-Cyr-les-Champagnes (24270)' => '24397',
		'Saint-Denis-d\'Oléron (17650)' => '17323',
		'Saint-Denis-de-Pile (33910)' => '33393',
		'Saint-Denis-des-Murs (87400)' => '87142',
		'Saint-Dizant-du-Bois (17150)' => '17324',
		'Saint-Dizant-du-Gua (17240)' => '17325',
		'Saint-Dizier-la-Tour (23130)' => '23187',
		'Saint-Dizier-les-Domaines (23270)' => '23188',
		'Saint-Dizier-Leyrenne (23400)' => '23189',
		'Saint-Domet (23190)' => '23190',
		'Saint-Dos (64270)' => '64474',
		'Saint-Éloi (23000)' => '23191',
		'Saint-Éloy-les-Tuileries (19210)' => '19198',
		'Saint-Émilion (33330)' => '33394',
		'Saint-Esteben (64640)' => '64476',
		'Saint-Estèphe (24360)' => '24398',
		'Saint-Estèphe (33180)' => '33395',
		'Saint-Étienne-aux-Clos (19200)' => '19199',
		'Saint-Étienne-d\'Orthe (40300)' => '40256',
		'Saint-Étienne-de-Baïgorry (64430)' => '64477',
		'Saint-Étienne-de-Fougères (47380)' => '47239',
		'Saint-Étienne-de-Fursac (23290)' => '23192',
		'Saint-Étienne-de-Lisse (33330)' => '33396',
		'Saint-Étienne-de-Puycorbier (24400)' => '24399',
		'Saint-Étienne-de-Villeréal (47210)' => '47240',
		'Saint-Étienne-la-Cigogne (79360)' => '79247',
		'Saint-Étienne-la-Geneste (19160)' => '19200',
		'Saint-Eugène (17520)' => '17326',
		'Saint-Eutrope (16190)' => '16314',
		'Saint-Eutrope-de-Born (47210)' => '47241',
		'Saint-Exupéry (33190)' => '33398',
		'Saint-Exupéry-les-Roches (19200)' => '19201',
		'Saint-Faust (64110)' => '64478',
		'Saint-Félix (16480)' => '16315',
		'Saint-Félix (17330)' => '17327',
		'Saint-Félix-de-Bourdeilles (24340)' => '24403',
		'Saint-Félix-de-Foncaude (33540)' => '33399',
		'Saint-Félix-de-Reillac-et-Mortemart (24260)' => '24404',
		'Saint-Félix-de-Villadeix (24510)' => '24405',
		'Saint-Ferme (33580)' => '33400',
		'Saint-Fiel (23000)' => '23195',
		'Saint-Fort-sur-Gironde (17240)' => '17328',
		'Saint-Fort-sur-le-Né (16130)' => '16316',
		'Saint-Fraigne (16140)' => '16317',
		'Saint-Fréjoux (19200)' => '19204',
		'Saint-Frion (23500)' => '23196',
		'Saint-Front (16460)' => '16318',
		'Saint-Front-d\'Alemps (24460)' => '24408',
		'Saint-Front-de-Pradoux (24400)' => '24409',
		'Saint-Front-la-Rivière (24300)' => '24410',
		'Saint-Front-sur-Lémance (47500)' => '47242',
		'Saint-Front-sur-Nizonne (24300)' => '24411',
		'Saint-Froult (17780)' => '17329',
		'Saint-Gaudent (86400)' => '86220',
		'Saint-Gein (40190)' => '40259',
		'Saint-Gelais (79410)' => '79249',
		'Saint-Génard (79500)' => '79251',
		'Saint-Gence (87510)' => '87143',
		'Saint-Généroux (79600)' => '79252',
		'Saint-Genès-de-Blaye (33390)' => '33405',
		'Saint-Genès-de-Castillon (33350)' => '33406',
		'Saint-Genès-de-Fronsac (33240)' => '33407',
		'Saint-Genès-de-Lombaud (33670)' => '33408',
		'Saint-Genest-d\'Ambière (86140)' => '86221',
		'Saint-Genest-sur-Roselle (87260)' => '87144',
		'Saint-Geniès (24590)' => '24412',
		'Saint-Geniez-ô-Merle (19220)' => '19205',
		'Saint-Genis-d\'Hiersac (16570)' => '16320',
		'Saint-Genis-de-Saintonge (17240)' => '17331',
		'Saint-Genis-du-Bois (33760)' => '33409',
		'Saint-Georges (16700)' => '16321',
		'Saint-Georges (47370)' => '47328',
		'Saint-Georges-Antignac (17240)' => '17332',
		'Saint-Georges-Blancaneix (24130)' => '24413',
		'Saint-Georges-d\'Oléron (17190)' => '17337',
		'Saint-Georges-de-Didonne (17110)' => '17333',
		'Saint-Georges-de-Longuepierre (17470)' => '17334',
		'Saint-Georges-de-Montclard (24140)' => '24414',
		'Saint-Georges-de-Noisné (79400)' => '79253',
		'Saint-Georges-de-Rex (79210)' => '79254',
		'Saint-Georges-des-Agoûts (17150)' => '17335',
		'Saint-Georges-des-Coteaux (17810)' => '17336',
		'Saint-Georges-du-Bois (17700)' => '17338',
		'Saint-Georges-la-Pouge (23250)' => '23197',
		'Saint-Georges-lès-Baillargeaux (86130)' => '86222',
		'Saint-Georges-les-Landes (87160)' => '87145',
		'Saint-Georges-Nigremont (23500)' => '23198',
		'Saint-Geours-d\'Auribat (40380)' => '40260',
		'Saint-Geours-de-Maremne (40230)' => '40261',
		'Saint-Géraud (47120)' => '47245',
		'Saint-Géraud-de-Corps (24700)' => '24415',
		'Saint-Germain (86310)' => '86223',
		'Saint-Germain-Beaupré (23160)' => '23199',
		'Saint-Germain-d\'Esteuil (33340)' => '33412',
		'Saint-Germain-de-Belvès (24170)' => '24416',
		'Saint-Germain-de-Grave (33490)' => '33411',
		'Saint-Germain-de-la-Rivière (33240)' => '33414',
		'Saint-Germain-de-Longue-Chaume (79200)' => '79255',
		'Saint-Germain-de-Lusignan (17500)' => '17339',
		'Saint-Germain-de-Marencennes (17700)' => '17340',
		'Saint-Germain-de-Montbron (16380)' => '16323',
		'Saint-Germain-de-Vibrac (17500)' => '17341',
		'Saint-Germain-des-Prés (24160)' => '24417',
		'Saint-Germain-du-Puch (33750)' => '33413',
		'Saint-Germain-du-Salembre (24190)' => '24418',
		'Saint-Germain-du-Seudre (17240)' => '17342',
		'Saint-Germain-et-Mons (24520)' => '24419',
		'Saint-Germain-Lavolps (19290)' => '19206',
		'Saint-Germain-les-Belles (87380)' => '87146',
		'Saint-Germain-les-Vergnes (19330)' => '19207',
		'Saint-Germier (79340)' => '79256',
		'Saint-Gervais (33240)' => '33415',
		'Saint-Gervais-les-Trois-Clochers (86230)' => '86224',
		'Saint-Géry (24400)' => '24420',
		'Saint-Geyrac (24330)' => '24421',
		'Saint-Gilles-les-Forêts (87130)' => '87147',
		'Saint-Girons-d\'Aiguevives (33920)' => '33416',
		'Saint-Girons-en-Béarn (64300)' => '64479',
		'Saint-Gladie-Arrive-Munein (64390)' => '64480',
		'Saint-Goin (64400)' => '64481',
		'Saint-Gor (40120)' => '40262',
		'Saint-Gourson (16700)' => '16325',
		'Saint-Goussaud (23430)' => '23200',
		'Saint-Grégoire-d\'Ardennes (17240)' => '17343',
		'Saint-Groux (16230)' => '16326',
		'Saint-Hilaire-Bonneval (87260)' => '87148',
		'Saint-Hilaire-d\'Estissac (24140)' => '24422',
		'Saint-Hilaire-de-la-Noaille (33190)' => '33418',
		'Saint-Hilaire-de-Lusignan (47450)' => '47246',
		'Saint-Hilaire-de-Villefranche (17770)' => '17344',
		'Saint-Hilaire-du-Bois (17500)' => '17345',
		'Saint-Hilaire-du-Bois (33540)' => '33419',
		'Saint-Hilaire-Foissac (19550)' => '19208',
		'Saint-Hilaire-la-Palud (79210)' => '79257',
		'Saint-Hilaire-la-Plaine (23150)' => '23201',
		'Saint-Hilaire-la-Treille (87190)' => '87149',
		'Saint-Hilaire-le-Château (23250)' => '23202',
		'Saint-Hilaire-les-Courbes (19170)' => '19209',
		'Saint-Hilaire-les-Places (87800)' => '87150',
		'Saint-Hilaire-Luc (19160)' => '19210',
		'Saint-Hilaire-Peyroux (19560)' => '19211',
		'Saint-Hilaire-Taurieux (19400)' => '19212',
		'Saint-Hippolyte (17430)' => '17346',
		'Saint-Hippolyte (33330)' => '33420',
		'Saint-Jacques-de-Thouars (79100)' => '79258',
		'Saint-Jal (19700)' => '19213',
		'Saint-Jammes (64160)' => '64482',
		'Saint-Jean-d\'Angély (17400)' => '17347',
		'Saint-Jean-d\'Angle (17620)' => '17348',
		'Saint-Jean-d\'Ataux (24190)' => '24424',
		'Saint-Jean-d\'Estissac (24140)' => '24426',
		'Saint-Jean-d\'Eyraud (24140)' => '24427',
		'Saint-Jean-d\'Illac (33127)' => '33422',
		'Saint-Jean-de-Blaignac (33420)' => '33421',
		'Saint-Jean-de-Côle (24800)' => '24425',
		'Saint-Jean-de-Duras (47120)' => '47247',
		'Saint-Jean-de-Lier (40380)' => '40263',
		'Saint-Jean-de-Liversay (17170)' => '17349',
		'Saint-Jean-de-Luz (64500)' => '64483',
		'Saint-Jean-de-Marsacq (40230)' => '40264',
		'Saint-Jean-de-Sauves (86330)' => '86225',
		'Saint-Jean-de-Thouars (79100)' => '79259',
		'Saint-Jean-de-Thurac (47270)' => '47248',
		'Saint-Jean-le-Vieux (64220)' => '64484',
		'Saint-Jean-Ligoure (87260)' => '87151',
		'Saint-Jean-Pied-de-Port (64220)' => '64485',
		'Saint-Jean-Poudge (64330)' => '64486',
		'Saint-Jory-de-Chalais (24800)' => '24428',
		'Saint-Jory-las-Bloux (24160)' => '24429',
		'Saint-Jouin-de-Marnes (79600)' => '79260',
		'Saint-Jouin-de-Milly (79380)' => '79261',
		'Saint-Jouvent (87510)' => '87152',
		'Saint-Julien-aux-Bois (19220)' => '19214',
		'Saint-Julien-Beychevelle (33250)' => '33423',
		'Saint-Julien-d\'Armagnac (40240)' => '40265',
		'Saint-Julien-d\'Eymet (24500)' => '24433',
		'Saint-Julien-de-Crempse (24140)' => '24431',
		'Saint-Julien-de-l\'Escap (17400)' => '17350',
		'Saint-Julien-de-Lampon (24370)' => '24432',
		'Saint-Julien-en-Born (40170)' => '40266',
		'Saint-Julien-l\'Ars (86800)' => '86226',
		'Saint-Julien-la-Genête (23110)' => '23203',
		'Saint-Julien-le-Châtel (23130)' => '23204',
		'Saint-Julien-le-Pèlerin (19430)' => '19215',
		'Saint-Julien-le-Petit (87460)' => '87153',
		'Saint-Julien-le-Vendômois (19210)' => '19216',
		'Saint-Julien-Maumont (19500)' => '19217',
		'Saint-Julien-près-Bort (19110)' => '19218',
		'Saint-Junien (87200)' => '87154',
		'Saint-Junien-la-Bregère (23400)' => '23205',
		'Saint-Junien-les-Combes (87300)' => '87155',
		'Saint-Just (24320)' => '24434',
		'Saint-Just-Ibarre (64120)' => '64487',
		'Saint-Just-le-Martel (87590)' => '87156',
		'Saint-Just-Luzac (17320)' => '17351',
		'Saint-Justin (40240)' => '40267',
		'Saint-Laon (86200)' => '86227',
		'Saint-Laurent (23000)' => '23206',
		'Saint-Laurent (47130)' => '47249',
		'Saint-Laurent-Bretagne (64160)' => '64488',
		'Saint-Laurent-d\'Arce (33240)' => '33425',
		'Saint-Laurent-de-Belzagot (16190)' => '16328',
		'Saint-Laurent-de-Céris (16450)' => '16329',
		'Saint-Laurent-de-Cognac (16100)' => '16330',
		'Saint-Laurent-de-Gosse (40390)' => '40268',
		'Saint-Laurent-de-Jourdes (86410)' => '86228',
		'Saint-Laurent-de-la-Barrière (17380)' => '17352',
		'Saint-Laurent-de-la-Prée (17450)' => '17353',
		'Saint-Laurent-des-Combes (16480)' => '16331',
		'Saint-Laurent-des-Combes (33330)' => '33426',
		'Saint-Laurent-des-Hommes (24400)' => '24436',
		'Saint-Laurent-des-Vignes (24100)' => '24437',
		'Saint-Laurent-du-Bois (33540)' => '33427',
		'Saint-Laurent-du-Plan (33190)' => '33428',
		'Saint-Laurent-la-Vallée (24170)' => '24438',
		'Saint-Laurent-les-Églises (87240)' => '87157',
		'Saint-Laurent-Médoc (33112)' => '33424',
		'Saint-Laurent-sur-Gorre (87310)' => '87158',
		'Saint-Laurs (79160)' => '79263',
		'Saint-Léger (16250)' => '16332',
		'Saint-Léger (17800)' => '17354',
		'Saint-Léger (47160)' => '47250',
		'Saint-Léger-Bridereix (23300)' => '23207',
		'Saint-Léger-de-Balson (33113)' => '33429',
		'Saint-Léger-de-la-Martinière (79500)' => '79264',
		'Saint-Léger-de-Montbrillais (86120)' => '86229',
		'Saint-Léger-de-Montbrun (79100)' => '79265',
		'Saint-Léger-la-Montagne (87340)' => '87159',
		'Saint-Léger-le-Guérétois (23000)' => '23208',
		'Saint-Léger-Magnazeix (87190)' => '87160',
		'Saint-Léomer (86290)' => '86230',
		'Saint-Léon (33670)' => '33431',
		'Saint-Léon (47160)' => '47251',
		'Saint-Léon-d\'Issigeac (24560)' => '24441',
		'Saint-Léon-sur-l\'Isle (24110)' => '24442',
		'Saint-Léon-sur-Vézère (24290)' => '24443',
		'Saint-Léonard-de-Noblat (87400)' => '87161',
		'Saint-Lin (79420)' => '79267',
		'Saint-Lon-les-Mines (40300)' => '40269',
		'Saint-Loubert (33210)' => '33432',
		'Saint-Loubès (33450)' => '33433',
		'Saint-Loubouer (40320)' => '40270',
		'Saint-Louis-de-Montferrand (33440)' => '33434',
		'Saint-Louis-en-l\'Isle (24400)' => '24444',
		'Saint-Loup (17380)' => '17356',
		'Saint-Loup (23130)' => '23209',
		'Saint-Loup-Lamairé (79600)' => '79268',
		'Saint-Macaire (33490)' => '33435',
		'Saint-Macoux (86400)' => '86231',
		'Saint-Magne (33125)' => '33436',
		'Saint-Magne-de-Castillon (33350)' => '33437',
		'Saint-Maigrin (17520)' => '17357',
		'Saint-Maime-de-Péreyrol (24380)' => '24459',
		'Saint-Maixant (23200)' => '23210',
		'Saint-Maixant (33490)' => '33438',
		'Saint-Maixent-de-Beugné (79160)' => '79269',
		'Saint-Maixent-l\'École (79400)' => '79270',
		'Saint-Mandé-sur-Brédoire (17470)' => '17358',
		'Saint-Marc-à-Frongier (23200)' => '23211',
		'Saint-Marc-à-Loubaud (23460)' => '23212',
		'Saint-Marc-la-Lande (79310)' => '79271',
		'Saint-Marcel-du-Périgord (24510)' => '24445',
		'Saint-Marcory (24540)' => '24446',
		'Saint-Mard (17700)' => '17359',
		'Saint-Marien (23600)' => '23213',
		'Saint-Mariens (33620)' => '33439',
		'Saint-Martial (16190)' => '16334',
		'Saint-Martial (17330)' => '17361',
		'Saint-Martial (33490)' => '33440',
		'Saint-Martial-d\'Albarède (24160)' => '24448',
		'Saint-Martial-d\'Artenset (24700)' => '24449',
		'Saint-Martial-de-Gimel (19150)' => '19220',
		'Saint-Martial-de-Mirambeau (17150)' => '17362',
		'Saint-Martial-de-Nabirat (24250)' => '24450',
		'Saint-Martial-de-Valette (24300)' => '24451',
		'Saint-Martial-de-Vitaterne (17500)' => '17363',
		'Saint-Martial-Entraygues (19400)' => '19221',
		'Saint-Martial-le-Mont (23150)' => '23214',
		'Saint-Martial-le-Vieux (23100)' => '23215',
		'Saint-Martial-sur-Isop (87330)' => '87163',
		'Saint-Martial-sur-Né (17520)' => '17364',
		'Saint-Martial-Viveyrol (24320)' => '24452',
		'Saint-Martin-Château (23460)' => '23216',
		'Saint-Martin-Curton (47700)' => '47254',
		'Saint-Martin-d\'Arberoue (64640)' => '64489',
		'Saint-Martin-d\'Arrossa (64780)' => '64490',
		'Saint-Martin-d\'Ary (17270)' => '17365',
		'Saint-Martin-d\'Oney (40090)' => '40274',
		'Saint-Martin-de-Beauville (47270)' => '47255',
		'Saint-Martin-de-Bernegoue (79230)' => '79273',
		'Saint-Martin-de-Coux (17360)' => '17366',
		'Saint-Martin-de-Fressengeas (24800)' => '24453',
		'Saint-Martin-de-Gurson (24610)' => '24454',
		'Saint-Martin-de-Hinx (40390)' => '40272',
		'Saint-Martin-de-Juillers (17400)' => '17367',
		'Saint-Martin-de-Jussac (87200)' => '87164',
		'Saint-Martin-de-Laye (33910)' => '33442',
		'Saint-Martin-de-Lerm (33540)' => '33443',
		'Saint-Martin-de-Mâcon (79100)' => '79274',
		'Saint-Martin-de-Ré (17410)' => '17369',
		'Saint-Martin-de-Ribérac (24600)' => '24455',
		'Saint-Martin-de-Saint-Maixent (79400)' => '79276',
		'Saint-Martin-de-Sanzay (79290)' => '79277',
		'Saint-Martin-de-Seignanx (40390)' => '40273',
		'Saint-Martin-de-Sescas (33490)' => '33444',
		'Saint-Martin-de-Villeréal (47210)' => '47256',
		'Saint-Martin-des-Combes (24140)' => '24456',
		'Saint-Martin-du-Bois (33910)' => '33445',
		'Saint-Martin-du-Clocher (16700)' => '16335',
		'Saint-Martin-du-Fouilloux (79420)' => '79278',
		'Saint-Martin-du-Puy (33540)' => '33446',
		'Saint-Martin-l\'Ars (86350)' => '86234',
		'Saint-Martin-l\'Astier (24400)' => '24457',
		'Saint-Martin-la-Méanne (19320)' => '19222',
		'Saint-Martin-Lacaussade (33390)' => '33441',
		'Saint-Martin-le-Mault (87360)' => '87165',
		'Saint-Martin-le-Pin (24300)' => '24458',
		'Saint-Martin-le-Vieux (87700)' => '87166',
		'Saint-Martin-lès-Melle (79500)' => '79279',
		'Saint-Martin-Petit (47180)' => '47257',
		'Saint-Martin-Sainte-Catherine (23430)' => '23217',
		'Saint-Martin-Sepert (19210)' => '19223',
		'Saint-Martin-Terressus (87400)' => '87167',
		'Saint-Mary (16260)' => '16336',
		'Saint-Mathieu (87440)' => '87168',
		'Saint-Maurice-de-Lestapel (47290)' => '47259',
		'Saint-Maurice-des-Lions (16500)' => '16337',
		'Saint-Maurice-la-Clouère (86160)' => '86235',
		'Saint-Maurice-la-Souterraine (23300)' => '23219',
		'Saint-Maurice-les-Brousses (87800)' => '87169',
		'Saint-Maurice-près-Crocq (23260)' => '23218',
		'Saint-Maurice-sur-Adour (40270)' => '40275',
		'Saint-Maurin (47270)' => '47260',
		'Saint-Maxire (79410)' => '79281',
		'Saint-Méard (87130)' => '87170',
		'Saint-Méard-de-Drône (24600)' => '24460',
		'Saint-Méard-de-Gurçon (24610)' => '24461',
		'Saint-Médard (16300)' => '16338',
		'Saint-Médard (17500)' => '17372',
		'Saint-Médard (64370)' => '64491',
		'Saint-Médard (79370)' => '79282',
		'Saint-Médard-d\'Aunis (17220)' => '17373',
		'Saint-Médard-d\'Excideuil (24160)' => '24463',
		'Saint-Médard-d\'Eyrans (33650)' => '33448',
		'Saint-Médard-de-Guizières (33230)' => '33447',
		'Saint-Médard-de-Mussidan (24400)' => '24462',
		'Saint-Médard-en-Jalles (33160)' => '33449',
		'Saint-Médard-la-Rochette (23200)' => '23220',
		'Saint-Même-les-Carrières (16720)' => '16340',
		'Saint-Merd-de-Lapleau (19320)' => '19225',
		'Saint-Merd-la-Breuille (23100)' => '23221',
		'Saint-Merd-les-Oussines (19170)' => '19226',
		'Saint-Mesmin (24270)' => '24464',
		'Saint-Mexant (19330)' => '19227',
		'Saint-Michel (16470)' => '16341',
		'Saint-Michel (64220)' => '64492',
		'Saint-Michel-de-Castelnau (33840)' => '33450',
		'Saint-Michel-de-Double (24400)' => '24465',
		'Saint-Michel-de-Fronsac (33126)' => '33451',
		'Saint-Michel-de-Lapujade (33190)' => '33453',
		'Saint-Michel-de-Montaigne (24230)' => '24466',
		'Saint-Michel-de-Rieufret (33720)' => '33452',
		'Saint-Michel-de-Veisse (23480)' => '23222',
		'Saint-Michel-de-Villadeix (24380)' => '24468',
		'Saint-Michel-Escalus (40550)' => '40276',
		'Saint-Moreil (23400)' => '23223',
		'Saint-Morillon (33650)' => '33454',
		'Saint-Nazaire-sur-Charente (17780)' => '17375',
		'Saint-Nexans (24520)' => '24472',
		'Saint-Nicolas-de-la-Balerme (47220)' => '47262',
		'Saint-Oradoux-de-Chirouze (23100)' => '23224',
		'Saint-Oradoux-près-Crocq (23260)' => '23225',
		'Saint-Ouen-d\'Aunis (17230)' => '17376',
		'Saint-Ouen-la-Thène (17490)' => '17377',
		'Saint-Ouen-sur-Gartempe (87300)' => '87172',
		'Saint-Palais (33820)' => '33456',
		'Saint-Palais (64120)' => '64493',
		'Saint-Palais-de-Négrignac (17210)' => '17378',
		'Saint-Palais-de-Phiolin (17800)' => '17379',
		'Saint-Palais-du-Né (16300)' => '16342',
		'Saint-Palais-sur-Mer (17420)' => '17380',
		'Saint-Pancrace (24530)' => '24474',
		'Saint-Pandelon (40180)' => '40277',
		'Saint-Pantaléon-de-Lapleau (19160)' => '19228',
		'Saint-Pantaléon-de-Larche (19600)' => '19229',
		'Saint-Pantaly-d\'Ans (24640)' => '24475',
		'Saint-Pantaly-d\'Excideuil (24160)' => '24476',
		'Saint-Pardon-de-Conques (33210)' => '33457',
		'Saint-Pardoult (17400)' => '17381',
		'Saint-Pardoux (79310)' => '79285',
		'Saint-Pardoux (87250)' => '87173',
		'Saint-Pardoux-Corbier (19210)' => '19230',
		'Saint-Pardoux-d\'Arnet (23260)' => '23226',
		'Saint-Pardoux-de-Drône (24600)' => '24477',
		'Saint-Pardoux-du-Breuil (47200)' => '47263',
		'Saint-Pardoux-et-Vielvic (24170)' => '24478',
		'Saint-Pardoux-Isaac (47800)' => '47264',
		'Saint-Pardoux-l\'Ortigier (19270)' => '19234',
		'Saint-Pardoux-la-Croisille (19320)' => '19231',
		'Saint-Pardoux-la-Rivière (24470)' => '24479',
		'Saint-Pardoux-le-Neuf (19200)' => '19232',
		'Saint-Pardoux-le-Neuf (23200)' => '23228',
		'Saint-Pardoux-le-Vieux (19200)' => '19233',
		'Saint-Pardoux-les-Cards (23150)' => '23229',
		'Saint-Pardoux-Morterolles (23400)' => '23227',
		'Saint-Pastour (47290)' => '47265',
		'Saint-Paul (19150)' => '19235',
		'Saint-Paul (33390)' => '33458',
		'Saint-Paul (87260)' => '87174',
		'Saint-Paul-de-Serre (24380)' => '24480',
		'Saint-Paul-en-Born (40200)' => '40278',
		'Saint-Paul-en-Gâtine (79240)' => '79286',
		'Saint-Paul-la-Roche (24800)' => '24481',
		'Saint-Paul-lès-Dax (40990)' => '40279',
		'Saint-Paul-Lizonne (24320)' => '24482',
		'Saint-Pé-de-Léren (64270)' => '64494',
		'Saint-Pé-Saint-Simon (47170)' => '47266',
		'Saint-Pée-sur-Nivelle (64310)' => '64495',
		'Saint-Perdon (40090)' => '40280',
		'Saint-Perdoux (24560)' => '24483',
		'Saint-Pey-d\'Armens (33330)' => '33459',
		'Saint-Pey-de-Castets (33350)' => '33460',
		'Saint-Philippe-d\'Aiguille (33350)' => '33461',
		'Saint-Philippe-du-Seignal (33220)' => '33462',
		'Saint-Pierre-Bellevue (23460)' => '23232',
		'Saint-Pierre-Chérignat (23430)' => '23230',
		'Saint-Pierre-d\'Amilly (17700)' => '17382',
		'Saint-Pierre-d\'Aurillac (33490)' => '33463',
		'Saint-Pierre-d\'Exideuil (86400)' => '86237',
		'Saint-Pierre-d\'Eyraud (24130)' => '24487',
		'Saint-Pierre-d\'Irube (64990)' => '64496',
		'Saint-Pierre-d\'Oléron (17310)' => '17385',
		'Saint-Pierre-de-Bat (33760)' => '33464',
		'Saint-Pierre-de-Buzet (47160)' => '47267',
		'Saint-Pierre-de-Chignac (24330)' => '24484',
		'Saint-Pierre-de-Clairac (47270)' => '47269',
		'Saint-Pierre-de-Côle (24800)' => '24485',
		'Saint-Pierre-de-Frugie (24450)' => '24486',
		'Saint-Pierre-de-Fursac (23290)' => '23231',
		'Saint-Pierre-de-Juillers (17400)' => '17383',
		'Saint-Pierre-de-l\'Isle (17330)' => '17384',
		'Saint-Pierre-de-Maillé (86260)' => '86236',
		'Saint-Pierre-de-Mons (33210)' => '33465',
		'Saint-Pierre-des-Échaubrognes (79700)' => '79289',
		'Saint-Pierre-du-Mont (40280)' => '40281',
		'Saint-Pierre-du-Palais (17270)' => '17386',
		'Saint-Pierre-le-Bost (23600)' => '23233',
		'Saint-Pierre-sur-Dropt (47120)' => '47271',
		'Saint-Pompain (79160)' => '79290',
		'Saint-Pompont (24170)' => '24488',
		'Saint-Porchaire (17250)' => '17387',
		'Saint-Preuil (16130)' => '16343',
		'Saint-Priest (23110)' => '23234',
		'Saint-Priest-de-Gimel (19800)' => '19236',
		'Saint-Priest-la-Feuille (23300)' => '23235',
		'Saint-Priest-la-Plaine (23240)' => '23236',
		'Saint-Priest-les-Fougères (24450)' => '24489',
		'Saint-Priest-Ligoure (87800)' => '87176',
		'Saint-Priest-Palus (23400)' => '23237',
		'Saint-Priest-sous-Aixe (87700)' => '87177',
		'Saint-Priest-Taurion (87480)' => '87178',
		'Saint-Privat (19220)' => '19237',
		'Saint-Privat-des-Prés (24410)' => '24490',
		'Saint-Projet-Saint-Constant (16110)' => '16344',
		'Saint-Quantin-de-Rançanne (17800)' => '17388',
		'Saint-Quentin-de-Baron (33750)' => '33466',
		'Saint-Quentin-de-Caplong (33220)' => '33467',
		'Saint-Quentin-de-Chalais (16210)' => '16346',
		'Saint-Quentin-du-Dropt (47330)' => '47272',
		'Saint-Quentin-la-Chabanne (23500)' => '23238',
		'Saint-Quentin-sur-Charente (16150)' => '16345',
		'Saint-Rabier (24210)' => '24491',
		'Saint-Raphaël (24160)' => '24493',
		'Saint-Rémy (19290)' => '19238',
		'Saint-Rémy (24700)' => '24494',
		'Saint-Rémy (79410)' => '79293',
		'Saint-Rémy-sur-Creuse (86220)' => '86241',
		'Saint-Robert (19310)' => '19239',
		'Saint-Robert (47340)' => '47273',
		'Saint-Rogatien (17220)' => '17391',
		'Saint-Romain (16210)' => '16347',
		'Saint-Romain (86250)' => '86242',
		'Saint-Romain-de-Benet (17600)' => '17393',
		'Saint-Romain-de-Monpazier (24540)' => '24495',
		'Saint-Romain-et-Saint-Clément (24800)' => '24496',
		'Saint-Romain-la-Virvée (33240)' => '33470',
		'Saint-Romain-le-Noble (47270)' => '47274',
		'Saint-Romain-sur-Gironde (17240)' => '17392',
		'Saint-Romans-des-Champs (79230)' => '79294',
		'Saint-Romans-lès-Melle (79500)' => '79295',
		'Saint-Salvadour (19700)' => '19240',
		'Saint-Salvy (47360)' => '47275',
		'Saint-Sardos (47360)' => '47276',
		'Saint-Saturnin (16290)' => '16348',
		'Saint-Saturnin-du-Bois (17700)' => '17394',
		'Saint-Saud-Lacoussière (24470)' => '24498',
		'Saint-Sauvant (17610)' => '17395',
		'Saint-Sauvant (86600)' => '86244',
		'Saint-Sauveur (24520)' => '24499',
		'Saint-Sauveur (33250)' => '33471',
		'Saint-Sauveur-d\'Aunis (17540)' => '17396',
		'Saint-Sauveur-de-Meilhan (47180)' => '47277',
		'Saint-Sauveur-de-Puynormand (33660)' => '33472',
		'Saint-Sauveur-Lalande (24700)' => '24500',
		'Saint-Savin (33920)' => '33473',
		'Saint-Savin (86310)' => '86246',
		'Saint-Savinien (17350)' => '17397',
		'Saint-Saviol (86400)' => '86247',
		'Saint-Sébastien (23160)' => '23239',
		'Saint-Secondin (86350)' => '86248',
		'Saint-Selve (33650)' => '33474',
		'Saint-Sernin (47120)' => '47278',
		'Saint-Setiers (19290)' => '19241',
		'Saint-Seurin-de-Bourg (33710)' => '33475',
		'Saint-Seurin-de-Cadourne (33180)' => '33476',
		'Saint-Seurin-de-Cursac (33390)' => '33477',
		'Saint-Seurin-de-Palenne (17800)' => '17398',
		'Saint-Seurin-de-Prats (24230)' => '24501',
		'Saint-Seurin-sur-l\'Isle (33660)' => '33478',
		'Saint-Sève (33190)' => '33479',
		'Saint-Sever (40500)' => '40282',
		'Saint-Sever-de-Saintonge (17800)' => '17400',
		'Saint-Séverin (16390)' => '16350',
		'Saint-Séverin-d\'Estissac (24190)' => '24502',
		'Saint-Séverin-sur-Boutonne (17330)' => '17401',
		'Saint-Sigismond-de-Clermont (17240)' => '17402',
		'Saint-Silvain-Bas-le-Roc (23600)' => '23240',
		'Saint-Silvain-Bellegarde (23190)' => '23241',
		'Saint-Silvain-Montaigut (23320)' => '23242',
		'Saint-Silvain-sous-Toulx (23140)' => '23243',
		'Saint-Simeux (16120)' => '16351',
		'Saint-Simon (16120)' => '16352',
		'Saint-Simon-de-Bordes (17500)' => '17403',
		'Saint-Simon-de-Pellouaille (17260)' => '17404',
		'Saint-Sixte (47220)' => '47279',
		'Saint-Solve (19130)' => '19242',
		'Saint-Sorlin-de-Conac (17150)' => '17405',
		'Saint-Sornin (16220)' => '16353',
		'Saint-Sornin (17600)' => '17406',
		'Saint-Sornin-la-Marche (87210)' => '87179',
		'Saint-Sornin-Lavolps (19230)' => '19243',
		'Saint-Sornin-Leulac (87290)' => '87180',
		'Saint-Sulpice-d\'Arnoult (17250)' => '17408',
		'Saint-Sulpice-d\'Excideuil (24800)' => '24505',
		'Saint-Sulpice-de-Cognac (16370)' => '16355',
		'Saint-Sulpice-de-Faleyrens (33330)' => '33480',
		'Saint-Sulpice-de-Guilleragues (33580)' => '33481',
		'Saint-Sulpice-de-Mareuil (24340)' => '24503',
		'Saint-Sulpice-de-Pommiers (33540)' => '33482',
		'Saint-Sulpice-de-Roumagnac (24600)' => '24504',
		'Saint-Sulpice-de-Royan (17200)' => '17409',
		'Saint-Sulpice-de-Ruffec (16460)' => '16356',
		'Saint-Sulpice-et-Cameyrac (33450)' => '33483',
		'Saint-Sulpice-Laurière (87370)' => '87181',
		'Saint-Sulpice-le-Dunois (23800)' => '23244',
		'Saint-Sulpice-le-Guérétois (23000)' => '23245',
		'Saint-Sulpice-les-Bois (19250)' => '19244',
		'Saint-Sulpice-les-Champs (23480)' => '23246',
		'Saint-Sulpice-les-Feuilles (87160)' => '87182',
		'Saint-Sylvain (19380)' => '19245',
		'Saint-Sylvestre (87240)' => '87183',
		'Saint-Sylvestre-sur-Lot (47140)' => '47280',
		'Saint-Symphorien (33113)' => '33484',
		'Saint-Symphorien (79270)' => '79298',
		'Saint-Symphorien-sur-Couze (87140)' => '87184',
		'Saint-Thomas-de-Conac (17150)' => '17410',
		'Saint-Trojan (33710)' => '33486',
		'Saint-Trojan-les-Bains (17370)' => '17411',
		'Saint-Urcisse (47270)' => '47281',
		'Saint-Vaize (17100)' => '17412',
		'Saint-Vallier (16480)' => '16357',
		'Saint-Varent (79330)' => '79299',
		'Saint-Vaury (23320)' => '23247',
		'Saint-Viance (19240)' => '19246',
		'Saint-Victor (24350)' => '24508',
		'Saint-Victor-en-Marche (23000)' => '23248',
		'Saint-Victour (19200)' => '19247',
		'Saint-Victurnien (87420)' => '87185',
		'Saint-Vincent (64800)' => '64498',
		'Saint-Vincent-de-Connezac (24190)' => '24509',
		'Saint-Vincent-de-Cosse (24220)' => '24510',
		'Saint-Vincent-de-Lamontjoie (47310)' => '47282',
		'Saint-Vincent-de-Paul (33440)' => '33487',
		'Saint-Vincent-de-Paul (40990)' => '40283',
		'Saint-Vincent-de-Pertignas (33420)' => '33488',
		'Saint-Vincent-de-Tyrosse (40230)' => '40284',
		'Saint-Vincent-Jalmoutiers (24410)' => '24511',
		'Saint-Vincent-la-Châtre (79500)' => '79301',
		'Saint-Vincent-le-Paluel (24200)' => '24512',
		'Saint-Vincent-sur-l\'Isle (24420)' => '24513',
		'Saint-Vite (47500)' => '47283',
		'Saint-Vitte-sur-Briance (87380)' => '87186',
		'Saint-Vivien (17220)' => '17413',
		'Saint-Vivien (24230)' => '24514',
		'Saint-Vivien-de-Blaye (33920)' => '33489',
		'Saint-Vivien-de-Médoc (33590)' => '33490',
		'Saint-Vivien-de-Monségur (33580)' => '33491',
		'Saint-Xandre (17138)' => '17414',
		'Saint-Yaguen (40400)' => '40285',
		'Saint-Ybard (19140)' => '19248',
		'Saint-Yrieix-la-Montagne (23460)' => '23249',
		'Saint-Yrieix-la-Perche (87500)' => '87187',
		'Saint-Yrieix-le-Déjalat (19300)' => '19249',
		'Saint-Yrieix-les-Bois (23150)' => '23250',
		'Saint-Yrieix-sous-Aixe (87700)' => '87188',
		'Saint-Yrieix-sur-Charente (16710)' => '16358',
		'Saint-Yzan-de-Soudiac (33920)' => '33492',
		'Saint-Yzans-de-Médoc (33340)' => '33493',
		'Sainte-Alvère-Saint-Laurent Les Bâtons (24510)' => '24362',
		'Sainte-Anne-Saint-Priest (87120)' => '87134',
		'Sainte-Bazeille (47180)' => '47233',
		'Sainte-Blandine (79370)' => '79240',
		'Sainte-Colombe (16230)' => '16309',
		'Sainte-Colombe (17210)' => '17319',
		'Sainte-Colombe (33350)' => '33390',
		'Sainte-Colombe (40700)' => '40252',
		'Sainte-Colombe-de-Duras (47120)' => '47236',
		'Sainte-Colombe-de-Villeneuve (47300)' => '47237',
		'Sainte-Colombe-en-Bruilhois (47310)' => '47238',
		'Sainte-Colome (64260)' => '64473',
		'Sainte-Croix (24440)' => '24393',
		'Sainte-Croix-de-Mareuil (24340)' => '24394',
		'Sainte-Croix-du-Mont (33410)' => '33392',
		'Sainte-Eanne (79800)' => '79246',
		'Sainte-Engrâce (64560)' => '64475',
		'Sainte-Eulalie (33560)' => '33397',
		'Sainte-Eulalie-d\'Ans (24640)' => '24401',
		'Sainte-Eulalie-d\'Eymet (24500)' => '24402',
		'Sainte-Eulalie-en-Born (40200)' => '40257',
		'Sainte-Féréole (19270)' => '19202',
		'Sainte-Feyre (23000)' => '23193',
		'Sainte-Feyre-la-Montagne (23500)' => '23194',
		'Sainte-Florence (33350)' => '33401',
		'Sainte-Fortunade (19490)' => '19203',
		'Sainte-Foy (40190)' => '40258',
		'Sainte-Foy-de-Belvès (24170)' => '24406',
		'Sainte-Foy-de-Longas (24510)' => '24407',
		'Sainte-Foy-la-Grande (33220)' => '33402',
		'Sainte-Foy-la-Longue (33490)' => '33403',
		'Sainte-Gemme (17250)' => '17330',
		'Sainte-Gemme (33580)' => '33404',
		'Sainte-Gemme (79330)' => '79250',
		'Sainte-Gemme-Martaillac (47250)' => '47244',
		'Sainte-Hélène (33480)' => '33417',
		'Sainte-Innocence (24500)' => '24423',
		'Sainte-Lheurine (17520)' => '17355',
		'Sainte-Livrade-sur-Lot (47110)' => '47252',
		'Sainte-Marie-de-Chignac (24330)' => '24447',
		'Sainte-Marie-de-Gosse (40390)' => '40271',
		'Sainte-Marie-de-Ré (17740)' => '17360',
		'Sainte-Marie-de-Vaux (87420)' => '87162',
		'Sainte-Marie-Lapanouze (19160)' => '19219',
		'Sainte-Marthe (47430)' => '47253',
		'Sainte-Maure-de-Peyriac (47170)' => '47258',
		'Sainte-Même (17770)' => '17374',
		'Sainte-Mondane (24370)' => '24470',
		'Sainte-Nathalène (24200)' => '24471',
		'Sainte-Néomaye (79260)' => '79283',
		'Sainte-Orse (24210)' => '24473',
		'Sainte-Ouenne (79220)' => '79284',
		'Sainte-Radegonde (17250)' => '17389',
		'Sainte-Radegonde (24560)' => '24492',
		'Sainte-Radegonde (33350)' => '33468',
		'Sainte-Radegonde (79100)' => '79292',
		'Sainte-Radégonde (86300)' => '86239',
		'Sainte-Ramée (17240)' => '17390',
		'Sainte-Sévère (16200)' => '16349',
		'Sainte-Soline (79120)' => '79297',
		'Sainte-Souline (16480)' => '16354',
		'Sainte-Soulle (17220)' => '17407',
		'Sainte-Terre (33350)' => '33485',
		'Sainte-Trie (24160)' => '24507',
		'Sainte-Verge (79100)' => '79300',
		'Saintes (17100)' => '17415',
		'Saires (86420)' => '86249',
		'Saivres (79400)' => '79302',
		'Saix (86120)' => '86250',
		'Salagnac (24160)' => '24515',
		'Salaunes (33160)' => '33494',
		'Saleignes (17510)' => '17416',
		'Salies-de-Béarn (64270)' => '64499',
		'Salignac-de-Mirambeau (17130)' => '17417',
		'Salignac-Eyvigues (24590)' => '24516',
		'Salignac-sur-Charente (17800)' => '17418',
		'Salleboeuf (33370)' => '33496',
		'Salles (33770)' => '33498',
		'Salles (47150)' => '47284',
		'Salles (79800)' => '79303',
		'Salles-d\'Angles (16130)' => '16359',
		'Salles-de-Barbezieux (16300)' => '16360',
		'Salles-de-Belvès (24170)' => '24517',
		'Salles-de-Villefagnan (16700)' => '16361',
		'Salles-Lavalette (16190)' => '16362',
		'Salles-Mongiscard (64300)' => '64500',
		'Salles-sur-Mer (17220)' => '17420',
		'Sallespisse (64300)' => '64501',
		'Salon (24380)' => '24518',
		'Salon-la-Tour (19510)' => '19250',
		'Samadet (40320)' => '40286',
		'Samazan (47250)' => '47285',
		'Sames (64520)' => '64502',
		'Sammarçolles (86200)' => '86252',
		'Samonac (33710)' => '33500',
		'Samsons-Lion (64350)' => '64503',
		'Sanguinet (40460)' => '40287',
		'Sannat (23110)' => '23167',
		'Sansais (79270)' => '79304',
		'Sanxay (86600)' => '86253',
		'Sarbazan (40120)' => '40288',
		'Sardent (23250)' => '23168',
		'Sare (64310)' => '64504',
		'Sarlande (24270)' => '24519',
		'Sarlat-la-Canéda (24200)' => '24520',
		'Sarliac-sur-l\'Isle (24420)' => '24521',
		'Sarpourenx (64300)' => '64505',
		'Sarran (19800)' => '19251',
		'Sarrance (64490)' => '64506',
		'Sarrazac (24800)' => '24522',
		'Sarraziet (40500)' => '40289',
		'Sarron (40800)' => '40290',
		'Sarroux (19110)' => '19252',
		'Saubion (40230)' => '40291',
		'Saubole (64420)' => '64507',
		'Saubrigues (40230)' => '40292',
		'Saubusse (40180)' => '40293',
		'Saucats (33650)' => '33501',
		'Saucède (64400)' => '64508',
		'Saugnac-et-Cambran (40180)' => '40294',
		'Saugnacq-et-Muret (40410)' => '40295',
		'Saugon (33920)' => '33502',
		'Sauguis-Saint-Étienne (64470)' => '64509',
		'Saujon (17600)' => '17421',
		'Saulgé (86500)' => '86254',
		'Saulgond (16420)' => '16363',
		'Sault-de-Navailles (64300)' => '64510',
		'Sauméjan (47420)' => '47286',
		'Saumont (47600)' => '47287',
		'Saumos (33680)' => '33503',
		'Saurais (79200)' => '79306',
		'Saussignac (24240)' => '24523',
		'Sauternes (33210)' => '33504',
		'Sauvagnac (16310)' => '16364',
		'Sauvagnas (47340)' => '47288',
		'Sauvagnon (64230)' => '64511',
		'Sauvelade (64150)' => '64512',
		'Sauveterre-de-Béarn (64390)' => '64513',
		'Sauveterre-de-Guyenne (33540)' => '33506',
		'Sauveterre-la-Lémance (47500)' => '47292',
		'Sauveterre-Saint-Denis (47220)' => '47293',
		'Sauviac (33430)' => '33507',
		'Sauviat-sur-Vige (87400)' => '87190',
		'Sauvignac (16480)' => '16365',
		'Sauzé-Vaussais (79190)' => '79307',
		'Savennes (23000)' => '23170',
		'Savignac (33124)' => '33508',
		'Savignac-de-Duras (47120)' => '47294',
		'Savignac-de-l\'Isle (33910)' => '33509',
		'Savignac-de-Miremont (24260)' => '24524',
		'Savignac-de-Nontron (24300)' => '24525',
		'Savignac-Lédrier (24270)' => '24526',
		'Savignac-les-Églises (24420)' => '24527',
		'Savignac-sur-Leyze (47150)' => '47295',
		'Savigné (86400)' => '86255',
		'Savigny-Lévescault (86800)' => '86256',
		'Savigny-sous-Faye (86140)' => '86257',
		'Sceau-Saint-Angel (24300)' => '24528',
		'Sciecq (79000)' => '79308',
		'Scillé (79240)' => '79309',
		'Scorbé-Clairvaux (86140)' => '86258',
		'Séby (64410)' => '64514',
		'Secondigné-sur-Belle (79170)' => '79310',
		'Secondigny (79130)' => '79311',
		'Sedze-Maubecq (64160)' => '64515',
		'Sedzère (64160)' => '64516',
		'Ségalas (47410)' => '47296',
		'Segonzac (16130)' => '16366',
		'Segonzac (19310)' => '19253',
		'Segonzac (24600)' => '24529',
		'Ségur-le-Château (19230)' => '19254',
		'Seigné (17510)' => '17422',
		'Seignosse (40510)' => '40296',
		'Seilhac (19700)' => '19255',
		'Séligné (79170)' => '79312',
		'Sembas (47360)' => '47297',
		'Séméacq-Blachon (64350)' => '64517',
		'Semens (33490)' => '33510',
		'Semillac (17150)' => '17423',
		'Semoussac (17150)' => '17424',
		'Semussac (17120)' => '17425',
		'Sencenac-Puy-de-Fourches (24310)' => '24530',
		'Sendets (33690)' => '33511',
		'Sendets (64320)' => '64518',
		'Sénestis (47430)' => '47298',
		'Senillé-Saint-Sauveur (86100)' => '86245',
		'Sepvret (79120)' => '79313',
		'Sérandon (19160)' => '19256',
		'Séreilhac (87620)' => '87191',
		'Sergeac (24290)' => '24531',
		'Sérignac-Péboudou (47410)' => '47299',
		'Sérignac-sur-Garonne (47310)' => '47300',
		'Sérigny (86230)' => '86260',
		'Sérilhac (19190)' => '19257',
		'Sermur (23700)' => '23171',
		'Séron (65320)' => '65422',
		'Serres-Castet (64121)' => '64519',
		'Serres-et-Montguyard (24500)' => '24532',
		'Serres-Gaston (40700)' => '40298',
		'Serres-Morlaàs (64160)' => '64520',
		'Serres-Sainte-Marie (64170)' => '64521',
		'Serreslous-et-Arribans (40700)' => '40299',
		'Sers (16410)' => '16368',
		'Servanches (24410)' => '24533',
		'Servières-le-Château (19220)' => '19258',
		'Sévignacq (64160)' => '64523',
		'Sévignacq-Meyracq (64260)' => '64522',
		'Sèvres-Anxaumont (86800)' => '86261',
		'Sexcles (19430)' => '19259',
		'Seyches (47350)' => '47301',
		'Seyresse (40180)' => '40300',
		'Siecq (17490)' => '17427',
		'Siest (40180)' => '40301',
		'Sigalens (33690)' => '33512',
		'Sigogne (16200)' => '16369',
		'Sigoulès (24240)' => '24534',
		'Sillars (86320)' => '86262',
		'Sillas (33690)' => '33513',
		'Simacourbe (64350)' => '64524',
		'Simeyrols (24370)' => '24535',
		'Sindères (40110)' => '40302',
		'Singleyrac (24500)' => '24536',
		'Sioniac (19120)' => '19260',
		'Siorac-de-Ribérac (24600)' => '24537',
		'Siorac-en-Périgord (24170)' => '24538',
		'Sireuil (16440)' => '16370',
		'Siros (64230)' => '64525',
		'Smarves (86240)' => '86263',
		'Solférino (40210)' => '40303',
		'Solignac (87110)' => '87192',
		'Sommières-du-Clain (86160)' => '86264',
		'Sompt (79110)' => '79314',
		'Sonnac (17160)' => '17428',
		'Soorts-Hossegor (40150)' => '40304',
		'Sorbets (40320)' => '40305',
		'Sorde-l\'Abbaye (40300)' => '40306',
		'Sore (40430)' => '40307',
		'Sorges et Ligueux en Périgord (24420)' => '24540',
		'Sornac (19290)' => '19261',
		'Sort-en-Chalosse (40180)' => '40308',
		'Sos (47170)' => '47302',
		'Sossais (86230)' => '86265',
		'Soubise (17780)' => '17429',
		'Soubran (17150)' => '17430',
		'Soubrebost (23250)' => '23173',
		'Soudaine-Lavinadière (19370)' => '19262',
		'Soudan (79800)' => '79316',
		'Soudat (24360)' => '24541',
		'Soudeilles (19300)' => '19263',
		'Souffrignac (16380)' => '16372',
		'Soulac-sur-Mer (33780)' => '33514',
		'Soulaures (24540)' => '24542',
		'Soulignac (33760)' => '33515',
		'Soulignonne (17250)' => '17431',
		'Soumans (23600)' => '23174',
		'Soumensac (47120)' => '47303',
		'Souméras (17130)' => '17432',
		'Soumoulou (64420)' => '64526',
		'Souprosse (40250)' => '40309',
		'Souraïde (64250)' => '64527',
		'Soursac (19550)' => '19264',
		'Sourzac (24400)' => '24543',
		'Sous-Parsat (23150)' => '23175',
		'Sousmoulins (17130)' => '17433',
		'Soussac (33790)' => '33516',
		'Soussans (33460)' => '33517',
		'Soustons (40140)' => '40310',
		'Soutiers (79310)' => '79318',
		'Souvigné (16240)' => '16373',
		'Souvigné (79800)' => '79319',
		'Soyaux (16800)' => '16374',
		'Suaux (16260)' => '16375',
		'Suhescun (64780)' => '64528',
		'Surdoux (87130)' => '87193',
		'Surgères (17700)' => '17434',
		'Surin (79220)' => '79320',
		'Surin (86250)' => '86266',
		'Suris (16270)' => '16376',
		'Sus (64190)' => '64529',
		'Susmiou (64190)' => '64530',
		'Sussac (87130)' => '87194',
		'Tabaille-Usquain (64190)' => '64531',
		'Tabanac (33550)' => '33518',
		'Tadousse-Ussau (64330)' => '64532',
		'Taillant (17350)' => '17435',
		'Taillebourg (17350)' => '17436',
		'Taillebourg (47200)' => '47304',
		'Taillecavat (33580)' => '33520',
		'Taizé (79100)' => '79321',
		'Taizé-Aizie (16700)' => '16378',
		'Talais (33590)' => '33521',
		'Talence (33400)' => '33522',
		'Taller (40260)' => '40311',
		'Talmont-sur-Gironde (17120)' => '17437',
		'Tamniès (24620)' => '24544',
		'Tanzac (17260)' => '17438',
		'Taponnat-Fleurignac (16110)' => '16379',
		'Tardes (23170)' => '23251',
		'Tardets-Sorholus (64470)' => '64533',
		'Targon (33760)' => '33523',
		'Tarnac (19170)' => '19265',
		'Tarnès (33240)' => '33524',
		'Tarnos (40220)' => '40312',
		'Taron-Sadirac-Viellenave (64330)' => '64534',
		'Tarsacq (64360)' => '64535',
		'Tartas (40400)' => '40313',
		'Taugon (17170)' => '17439',
		'Tauriac (33710)' => '33525',
		'Tayac (33570)' => '33526',
		'Tayrac (47270)' => '47305',
		'Teillots (24390)' => '24545',
		'Temple-Laguyon (24390)' => '24546',
		'Tercé (86800)' => '86268',
		'Tercillat (23350)' => '23252',
		'Tercis-les-Bains (40180)' => '40314',
		'Ternant (17400)' => '17440',
		'Ternay (86120)' => '86269',
		'Terrasson-Lavilledieu (24120)' => '24547',
		'Tersannes (87360)' => '87195',
		'Tesson (17460)' => '17441',
		'Tessonnière (79600)' => '79325',
		'Téthieu (40990)' => '40315',
		'Teuillac (33710)' => '33530',
		'Teyjat (24300)' => '24548',
		'Thaims (17120)' => '17442',
		'Thairé (17290)' => '17443',
		'Thalamy (19200)' => '19266',
		'Thauron (23250)' => '23253',
		'Theil-Rabier (16240)' => '16381',
		'Thénac (17460)' => '17444',
		'Thénac (24240)' => '24549',
		'Thénezay (79390)' => '79326',
		'Thenon (24210)' => '24550',
		'Thézac (17600)' => '17445',
		'Thézac (47370)' => '47307',
		'Thèze (64450)' => '64536',
		'Thiat (87320)' => '87196',
		'Thiviers (24800)' => '24551',
		'Thollet (86290)' => '86270',
		'Thonac (24290)' => '24552',
		'Thorigné (79370)' => '79327',
		'Thorigny-sur-le-Mignon (79360)' => '79328',
		'Thors (17160)' => '17446',
		'Thouars (79100)' => '79329',
		'Thouars-sur-Garonne (47230)' => '47308',
		'Thouron (87140)' => '87197',
		'Thurageau (86110)' => '86271',
		'Thuré (86540)' => '86272',
		'Tilh (40360)' => '40316',
		'Tillou (79110)' => '79330',
		'Tizac-de-Curton (33420)' => '33531',
		'Tizac-de-Lapouyade (33620)' => '33532',
		'Tocane-Saint-Apre (24350)' => '24553',
		'Tombeboeuf (47380)' => '47309',
		'Tonnay-Boutonne (17380)' => '17448',
		'Tonnay-Charente (17430)' => '17449',
		'Tonneins (47400)' => '47310',
		'Torsac (16410)' => '16382',
		'Torxé (17380)' => '17450',
		'Tosse (40230)' => '40317',
		'Toulenne (33210)' => '33533',
		'Toulouzette (40250)' => '40318',
		'Toulx-Sainte-Croix (23600)' => '23254',
		'Tourliac (47210)' => '47311',
		'Tournon-d\'Agenais (47370)' => '47312',
		'Tourriers (16560)' => '16383',
		'Tourtenay (79100)' => '79331',
		'Tourtoirac (24390)' => '24555',
		'Tourtrès (47380)' => '47313',
		'Touvérac (16360)' => '16384',
		'Touvre (16600)' => '16385',
		'Touzac (16120)' => '16386',
		'Toy-Viam (19170)' => '19268',
		'Trayes (79240)' => '79332',
		'Treignac (19260)' => '19269',
		'Trélissac (24750)' => '24557',
		'Trémolat (24510)' => '24558',
		'Trémons (47140)' => '47314',
		'Trensacq (40630)' => '40319',
		'Trentels (47140)' => '47315',
		'Tresses (33370)' => '33535',
		'Triac-Lautrait (16200)' => '16387',
		'Trizay (17250)' => '17453',
		'Troche (19230)' => '19270',
		'Trois-Fonds (23230)' => '23255',
		'Trois-Palis (16730)' => '16388',
		'Trois-Villes (64470)' => '64537',
		'Tudeils (19120)' => '19271',
		'Tugéras-Saint-Maurice (17130)' => '17454',
		'Tulle (19000)' => '19272',
		'Turenne (19500)' => '19273',
		'Turgon (16350)' => '16389',
		'Tursac (24620)' => '24559',
		'Tusson (16140)' => '16390',
		'Tuzie (16700)' => '16391',
		'Uchacq-et-Parentis (40090)' => '40320',
		'Uhart-Cize (64220)' => '64538',
		'Uhart-Mixe (64120)' => '64539',
		'Urcuit (64990)' => '64540',
		'Urdès (64370)' => '64541',
		'Urdos (64490)' => '64542',
		'Urepel (64430)' => '64543',
		'Urgons (40320)' => '40321',
		'Urost (64160)' => '64544',
		'Urrugne (64122)' => '64545',
		'Urt (64240)' => '64546',
		'Urval (24480)' => '24560',
		'Ussac (19270)' => '19274',
		'Usseau (79210)' => '79334',
		'Usseau (86230)' => '86275',
		'Ussel (19200)' => '19275',
		'Usson-du-Poitou (86350)' => '86276',
		'Ustaritz (64480)' => '64547',
		'Uza (40170)' => '40322',
		'Uzan (64370)' => '64548',
		'Uzein (64230)' => '64549',
		'Uzerche (19140)' => '19276',
		'Uzeste (33730)' => '33537',
		'Uzos (64110)' => '64550',
		'Val d\'Issoire (87330)' => '87097',
		'Val de Virvée (33240)' => '33018',
		'Val des Vignes (16250)' => '16175',
		'Valdivienne (86300)' => '86233',
		'Valence (16460)' => '16392',
		'Valeuil (24310)' => '24561',
		'Valeyrac (33340)' => '33538',
		'Valiergues (19200)' => '19277',
		'Vallans (79270)' => '79335',
		'Vallereuil (24190)' => '24562',
		'Vallière (23120)' => '23257',
		'Valojoulx (24290)' => '24563',
		'Vançais (79120)' => '79336',
		'Vandré (17700)' => '17457',
		'Vanxains (24600)' => '24564',
		'Vanzac (17500)' => '17458',
		'Vanzay (79120)' => '79338',
		'Varaignes (24360)' => '24565',
		'Varaize (17400)' => '17459',
		'Vareilles (23300)' => '23258',
		'Varennes (24150)' => '24566',
		'Varennes (86110)' => '86277',
		'Varès (47400)' => '47316',
		'Varetz (19240)' => '19278',
		'Vars (16330)' => '16393',
		'Vars-sur-Roseix (19130)' => '19279',
		'Varzay (17460)' => '17460',
		'Vasles (79340)' => '79339',
		'Vaulry (87140)' => '87198',
		'Vaunac (24800)' => '24567',
		'Vausseroux (79420)' => '79340',
		'Vautebis (79420)' => '79341',
		'Vaux (86700)' => '86278',
		'Vaux-Lavalette (16320)' => '16394',
		'Vaux-Rouillac (16170)' => '16395',
		'Vaux-sur-Mer (17640)' => '17461',
		'Vaux-sur-Vienne (86220)' => '86279',
		'Vayres (33870)' => '33539',
		'Vayres (87600)' => '87199',
		'Végennes (19120)' => '19280',
		'Veix (19260)' => '19281',
		'Vélines (24230)' => '24568',
		'Vellèches (86230)' => '86280',
		'Vendays-Montalivet (33930)' => '33540',
		'Vendeuvre-du-Poitou (86380)' => '86281',
		'Vendoire (24320)' => '24569',
		'Vénérand (17100)' => '17462',
		'Vensac (33590)' => '33541',
		'Ventouse (16460)' => '16396',
		'Vérac (33240)' => '33542',
		'Verdelais (33490)' => '33543',
		'Verdets (64400)' => '64551',
		'Verdille (16140)' => '16397',
		'Verdon (24520)' => '24570',
		'Vergeroux (17300)' => '17463',
		'Vergné (17330)' => '17464',
		'Vergt (24380)' => '24571',
		'Vergt-de-Biron (24540)' => '24572',
		'Vérines (17540)' => '17466',
		'Verneiges (23170)' => '23259',
		'Verneuil (16310)' => '16398',
		'Verneuil-Moustiers (87360)' => '87200',
		'Verneuil-sur-Vienne (87430)' => '87201',
		'Vernon (86340)' => '86284',
		'Vernoux-en-Gâtine (79240)' => '79342',
		'Vernoux-sur-Boutonne (79170)' => '79343',
		'Verrières (16130)' => '16399',
		'Verrières (86410)' => '86285',
		'Verrue (86420)' => '86286',
		'Verruyes (79310)' => '79345',
		'Vert (40420)' => '40323',
		'Verteillac (24320)' => '24573',
		'Verteuil-d\'Agenais (47260)' => '47317',
		'Verteuil-sur-Charente (16510)' => '16400',
		'Vertheuil (33180)' => '33545',
		'Vervant (16330)' => '16401',
		'Vervant (17400)' => '17467',
		'Veyrac (87520)' => '87202',
		'Veyrières (19200)' => '19283',
		'Veyrignac (24370)' => '24574',
		'Veyrines-de-Domme (24250)' => '24575',
		'Veyrines-de-Vergt (24380)' => '24576',
		'Vézac (24220)' => '24577',
		'Vézières (86120)' => '86287',
		'Vialer (64330)' => '64552',
		'Viam (19170)' => '19284',
		'Vianne (47230)' => '47318',
		'Vibrac (16120)' => '16402',
		'Vibrac (17130)' => '17468',
		'Vicq-d\'Auribat (40380)' => '40324',
		'Vicq-sur-Breuilh (87260)' => '87203',
		'Vicq-sur-Gartempe (86260)' => '86288',
		'Vidaillat (23250)' => '23260',
		'Videix (87600)' => '87204',
		'Vielle-Saint-Girons (40560)' => '40326',
		'Vielle-Soubiran (40240)' => '40327',
		'Vielle-Tursan (40320)' => '40325',
		'Viellenave-d\'Arthez (64170)' => '64554',
		'Viellenave-de-Navarrenx (64190)' => '64555',
		'Vielleségure (64150)' => '64556',
		'Viennay (79200)' => '79347',
		'Viersat (23170)' => '23261',
		'Vieux-Boucau-les-Bains (40480)' => '40328',
		'Vieux-Mareuil (24340)' => '24579',
		'Vieux-Ruffec (16350)' => '16404',
		'Vigeois (19410)' => '19285',
		'Vigeville (23140)' => '23262',
		'Vignes (64410)' => '64557',
		'Vignolles (16300)' => '16405',
		'Vignols (19130)' => '19286',
		'Vignonet (33330)' => '33546',
		'Vilhonneur (16220)' => '16406',
		'Villac (24120)' => '24580',
		'Villamblard (24140)' => '24581',
		'Villandraut (33730)' => '33547',
		'Villard (23800)' => '23263',
		'Villars (24530)' => '24582',
		'Villars-en-Pons (17260)' => '17469',
		'Villars-les-Bois (17770)' => '17470',
		'Villebois-Lavalette (16320)' => '16408',
		'Villebramar (47380)' => '47319',
		'Villedoux (17230)' => '17472',
		'Villefagnan (16240)' => '16409',
		'Villefavard (87190)' => '87206',
		'Villefollet (79170)' => '79348',
		'Villefranche-de-Lonchat (24610)' => '24584',
		'Villefranche-du-Périgord (24550)' => '24585',
		'Villefranche-du-Queyran (47160)' => '47320',
		'Villefranque (64990)' => '64558',
		'Villegats (16700)' => '16410',
		'Villegouge (33141)' => '33548',
		'Villejésus (16140)' => '16411',
		'Villejoubert (16560)' => '16412',
		'Villemain (79110)' => '79349',
		'Villemorin (17470)' => '17473',
		'Villemort (86310)' => '86291',
		'Villenave (40110)' => '40330',
		'Villenave-d\'Ornon (33140)' => '33550',
		'Villenave-de-Rions (33550)' => '33549',
		'Villenave-près-Béarn (65500)' => '65476',
		'Villeneuve (33710)' => '33551',
		'Villeneuve-de-Duras (47120)' => '47321',
		'Villeneuve-de-Marsan (40190)' => '40331',
		'Villeneuve-la-Comtesse (17330)' => '17474',
		'Villeneuve-sur-Lot (47300)' => '47323',
		'Villeréal (47210)' => '47324',
		'Villeton (47400)' => '47325',
		'Villetoureix (24600)' => '24586',
		'Villexavier (17500)' => '17476',
		'Villiers (86190)' => '86292',
		'Villiers-Couture (17510)' => '17477',
		'Villiers-en-Bois (79360)' => '79350',
		'Villiers-en-Plaine (79160)' => '79351',
		'Villiers-le-Roux (16240)' => '16413',
		'Villiers-sur-Chizé (79170)' => '79352',
		'Villognon (16230)' => '16414',
		'Vinax (17510)' => '17478',
		'Vindelle (16430)' => '16415',
		'Viodos-Abense-de-Bas (64130)' => '64559',
		'Virazeil (47200)' => '47326',
		'Virelade (33720)' => '33552',
		'Virollet (17260)' => '17479',
		'Virsac (33240)' => '33553',
		'Virson (17290)' => '17480',
		'Vitrac (24200)' => '24587',
		'Vitrac-Saint-Vincent (16310)' => '16416',
		'Vitrac-sur-Montane (19800)' => '19287',
		'Viven (64450)' => '64560',
		'Viville (16120)' => '16417',
		'Vivonne (86370)' => '86293',
		'Voeuil-et-Giget (16400)' => '16418',
		'Voissay (17400)' => '17481',
		'Vouharte (16330)' => '16419',
		'Vouhé (17700)' => '17482',
		'Vouhé (79310)' => '79354',
		'Vouillé (79230)' => '79355',
		'Vouillé (86190)' => '86294',
		'Voulême (86400)' => '86295',
		'Voulgézac (16250)' => '16420',
		'Voulmentin (79150)' => '79242',
		'Voulon (86700)' => '86296',
		'Vouneuil-sous-Biard (86580)' => '86297',
		'Vouneuil-sur-Vienne (86210)' => '86298',
		'Voutezac (19130)' => '19288',
		'Vouthon (16220)' => '16421',
		'Vouzailles (86170)' => '86299',
		'Vouzan (16410)' => '16422',
		'Xaintrailles (47230)' => '47327',
		'Xaintray (79220)' => '79357',
		'Xambes (16330)' => '16423',
		'Ychoux (40160)' => '40332',
		'Ygos-Saint-Saturnin (40110)' => '40333',
		'Yssandon (19310)' => '19289',
		'Yversay (86170)' => '86300',
		'Yves (17340)' => '17483',
		'Yviers (16210)' => '16424',
		'Yvrac (33370)' => '33554',
		'Yvrac-et-Malleyrand (16110)' => '16425',
		'Yzosse (40180)' => '40334'
	);
}
