/*
 * This file is part of the GNOME-keyring signond extension
 *
 * Copyright (C) 2012 Canonical Ltd.
 *
 * Contact: Alberto Mardegan <alberto.mardegan@canonical.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include "mocked-gnome-keyring.h"
#include <libsecret/secret.h>

#define DEFAULT_KEYRING SECRET_COLLECTION_DEFAULT

static GHashTable *keyrings = NULL;
static guint32 last_item_id = 123;

typedef struct {
    guint32 id;
    const SecretSchema *schema;
    char *display_name;
    GHashTable *attributes;
    char *secret;
} MockedItem;

typedef struct {
    GList *items;
} MockedKeyring;

static GHashTable *
copy_g_hash_table(GHashTable *table)
{
    GHashTable *copy;
    GHashTableIter iter;
    gpointer key, value;

    copy = g_hash_table_new_full(g_str_hash, g_str_equal,
                                 g_free, g_free);
    g_hash_table_iter_init(&iter, table);
    while (g_hash_table_iter_next(&iter, &key, &value)) {
        g_hash_table_insert(copy, g_strdup(key), g_strdup(value));
    }

    return copy;
}

static MockedItem *
mocked_item_new(const SecretSchema *schema,
                const char *display_name,
                GHashTable *attributes,
                const char *secret)
{
    MockedItem *item = g_slice_new0(MockedItem);
    item->id = last_item_id++;
    item->schema = schema;
    item->display_name = g_strdup(display_name);
    item->attributes = copy_g_hash_table(attributes);
    item->secret = g_strdup(secret);
    return item;
}

static gboolean
mocked_item_has_attributes(MockedItem *item,
                           GHashTable *attributes)
{
    GHashTableIter iter;
    gpointer key, value;

    g_hash_table_iter_init(&iter, attributes);
    while (g_hash_table_iter_next(&iter, &key, &value)) {
        gpointer item_value;

        item_value = g_hash_table_lookup(item->attributes, key);
        if (item_value == NULL || g_strcmp0(item_value, value) != 0)
            return FALSE;
    }

    return TRUE;
}

static void
mocked_item_free(MockedItem *item)
{
    g_free(item->display_name);
    g_hash_table_unref(item->attributes);
    g_free(item->secret);
    g_slice_free(MockedItem, item);
}

static MockedKeyring *
mocked_keyring_new()
{
    return g_slice_new0(MockedKeyring);
}

void
mocked_gnome_keyring_init()
{
    keyrings = g_hash_table_new(g_str_hash, g_str_equal);
    g_hash_table_insert(keyrings, DEFAULT_KEYRING, mocked_keyring_new());
    /* create the keyring for use with the unit tests */
    g_hash_table_insert(keyrings, TEST_KEYRING, mocked_keyring_new());
}

GList *
find_items(const SecretSchema *schema,
           GHashTable *attributes)
{
    GHashTableIter iter;
    const gchar *keyring_name;
    MockedKeyring *keyring;
    GList *results = NULL;

    g_return_val_if_fail (attributes != NULL, NULL);

    g_hash_table_iter_init(&iter, keyrings);
    while (g_hash_table_iter_next(&iter,
                                  (gpointer)&keyring_name,
                                  (gpointer)&keyring)) {
        GList *list;

        for (list = keyring->items; list != NULL; list = g_list_next(list)) {
            MockedItem *item = list->data;
            if (item->schema != schema) continue;

            if (!mocked_item_has_attributes(item, attributes)) continue;

            results = g_list_prepend(results, item);
        }
    }

    return results;
}

gboolean
secret_password_store_sync(const SecretSchema *schema,
                           const gchar *collection,
                           const gchar *display_name,
                           const gchar *secret,
                           G_GNUC_UNUSED GCancellable *cancellable,
                           GError **error,
                           ...)
{
    GList *found = NULL;
    MockedKeyring *mocked_keyring;
    GHashTable *attributes;
    gpointer existing_result;
    va_list va;

    g_return_val_if_fail (schema != NULL, FALSE);
    g_return_val_if_fail (collection != NULL, FALSE);

    g_debug("%s, collection %s", G_STRFUNC, collection);
    mocked_keyring = g_hash_table_lookup (keyrings, collection);
    if (G_UNLIKELY (mocked_keyring == NULL)) {
        g_set_error_literal(error, G_DBUS_ERROR, G_DBUS_ERROR_BAD_ADDRESS,
                            "No such keyring");
        return FALSE;
    }

    va_start (va, error);
    attributes = secret_attributes_buildv (schema, va);
    va_end (va);

    found = find_items(schema, attributes);
    existing_result = (found != NULL) ? found->data : NULL;
    g_list_free(found);

    if (existing_result != NULL) {
        /* Update the existing item */
        MockedItem *item = existing_result;

        g_free(item->display_name);
        item->display_name = g_strdup(display_name);

        g_free(item->secret);
        item->secret = g_strdup(secret);
    } else {
        MockedItem *item =
            mocked_item_new(schema, display_name, attributes, secret);

        mocked_keyring->items = g_list_prepend(mocked_keyring->items,
                                               item);
    }

    g_hash_table_unref(attributes);
    return TRUE;
}

gchar *
secret_password_lookup_sync(const SecretSchema *schema,
                            G_GNUC_UNUSED GCancellable *cancellable,
                            GError **error,
                            ...)
{
    MockedItem *item;
    GHashTable *attributes;
    GList *found;
    va_list va;

    g_return_val_if_fail (schema != NULL, FALSE);
    g_debug("%s", G_STRFUNC);

    va_start (va, error);
    attributes = secret_attributes_buildv (schema, va);
    va_end (va);

    found = find_items(schema, attributes);
    item = (found != NULL) ? found->data : NULL;
    g_list_free(found);

    if (item) {
        g_debug("Found: %s", item->secret);
    } else {
        g_debug("Item not found");
    }
    return item ? g_strdup(item->secret) : NULL;
}

gboolean
secret_password_clearv_sync(const SecretSchema* schema,
                            GHashTable *attributes,
                            G_GNUC_UNUSED GCancellable *cancellable,
                            G_GNUC_UNUSED GError **error)
{
    GHashTableIter iter;
    const gchar *keyring_name;
    MockedKeyring *keyring;
    gboolean found = FALSE;

    g_return_val_if_fail (schema != NULL, FALSE);
    g_debug("%s", G_STRFUNC);

    g_hash_table_iter_init(&iter, keyrings);
    while (g_hash_table_iter_next(&iter,
                                  (gpointer)&keyring_name,
                                  (gpointer)&keyring)) {
        GList *list, *next;

        for (list = keyring->items; list != NULL; list = next) {
            MockedItem *item = list->data;
            next = g_list_next(list);
            if (item->schema != schema) continue;

            if (!mocked_item_has_attributes(item, attributes)) continue;

            g_debug("Found item!");
            mocked_item_free(item);
            keyring->items = g_list_delete_link(keyring->items, list);
            found = TRUE;
        }
    }
    return found;
}
